<?php
require_once __DIR__ . '/config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$email = isset($_POST['email']) ? trim($_POST['email']) : '';

// Validate email
if (empty($email)) {
    echo json_encode(['success' => false, 'message' => 'Email is required']);
    exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'message' => 'Invalid email address']);
    exit;
}

try {
    $db = get_db();
    
    if (!$db) {
        throw new Exception('Database connection failed');
    }
    
    // Create newsletter_subscribers table if it doesn't exist
    $db->exec("CREATE TABLE IF NOT EXISTS newsletter_subscribers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        email VARCHAR(255) NOT NULL UNIQUE,
        status ENUM('active', 'unsubscribed') DEFAULT 'active',
        ip_address VARCHAR(45),
        user_agent TEXT,
        subscribed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        unsubscribed_at TIMESTAMP NULL,
        INDEX idx_email (email),
        INDEX idx_status (status)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    
    // Check if email already exists
    $check_stmt = $db->prepare('SELECT id, status FROM newsletter_subscribers WHERE email = :email');
    $check_stmt->execute(['email' => $email]);
    $existing = $check_stmt->fetch();
    
    if ($existing) {
        if ($existing['status'] === 'active') {
            echo json_encode(['success' => false, 'message' => 'This email is already subscribed to our newsletter']);
            exit;
        } else {
            // Reactivate subscription
            $update_stmt = $db->prepare('UPDATE newsletter_subscribers SET status = "active", subscribed_at = NOW(), unsubscribed_at = NULL WHERE email = :email');
            $update_stmt->execute(['email' => $email]);
            echo json_encode(['success' => true, 'message' => 'Welcome back! Your subscription has been reactivated.']);
            exit;
        }
    }
    
    // Insert new subscriber
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    $stmt = $db->prepare('INSERT INTO newsletter_subscribers (email, ip_address, user_agent) VALUES (:email, :ip, :ua)');
    $stmt->execute([
        'email' => $email,
        'ip' => $ip_address,
        'ua' => $user_agent
    ]);
    
    // Log the subscription
    error_log("New newsletter subscription: {$email}");
    
    echo json_encode([
        'success' => true, 
        'message' => 'Thank you for subscribing! You will receive the latest hosting tips and cloud insights.'
    ]);
    
} catch (PDOException $e) {
    error_log("Newsletter subscription error: " . $e->getMessage());
    
    if ($e->getCode() == 23000) { // Duplicate entry
        echo json_encode(['success' => false, 'message' => 'This email is already subscribed']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Unable to process subscription. Please try again later.']);
    }
} catch (Exception $e) {
    error_log("Newsletter subscription error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An error occurred. Please try again later.']);
}
