<?php
/**
 * WsplCloud - Main Configuration File
 * This file contains all the core configuration settings for the website.
 */

// Prevent direct script access
if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}
// ---------------------------------------------------------------
// | Root Path Configuration                                      |
// ---------------------------------------------------------------

// File System Paths
define('DS', DIRECTORY_SEPARATOR);
define('BASE_PATH', realpath(__DIR__));
define('PAGES_PATH', BASE_PATH . DS . 'pages');
define('PARTIAL_PATH', BASE_PATH . DS . 'partial');
define('ASSETS_PATH', BASE_PATH . DS . 'assets');
define('CONFIG_PATH', BASE_PATH . DS . 'config');
define('LOG_PATH', BASE_PATH . DS . 'logs');

// Ensure log directory exists
if (!is_dir(LOG_PATH) && is_writable(BASE_PATH)) {
    mkdir(LOG_PATH, 0755, true);
}

// URL Paths
define('DOC_ROOT', rtrim(realpath($_SERVER['DOCUMENT_ROOT']), DS));

// Calculate SITE_ROOT - the path from document root to our application
$site_path = str_replace(DOC_ROOT, '', BASE_PATH);
$site_path = str_replace('\\', '/', $site_path); // Normalize Windows backslashes
$site_path = trim($site_path, '/');
define('SITE_ROOT', '/' . $site_path);

// Server and Protocol Detection
$server_protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
$server_host = $_SERVER['HTTP_HOST'] ?? 'localhost';
define('BASE_URL', $server_protocol . '://' . $server_host . SITE_ROOT);

// URL Components for assets and pages
define('ASSETS_URL', BASE_URL . '/assets');
define('CSS_URL', ASSETS_URL . '/css');
define('JS_URL', ASSETS_URL . '/js');
define('IMAGES_URL', ASSETS_URL . '/images');

// ---------------------------------------------------------------
// | Website Information                                          |
// ---------------------------------------------------------------

// Website information
define('SITE_NAME', 'WsplCloud');
define('SITE_TAGLINE', 'Web Hosting & Cloud Services');
define('SITE_VERSION', '1.0.0');

// Contact information (used throughout the site)
define('CONTACT_EMAIL', 'info@wsplcloud.com');
define('CONTACT_PHONE', '+91 123 456 7890');
define('CONTACT_ADDRESS', '123 Cloud Street, Tech Park, Chennai, India');

// Social media links
define('SOCIAL_FACEBOOK', 'https://facebook.com/wsplcloud');
define('SOCIAL_TWITTER', 'https://twitter.com/wsplcloud');
define('SOCIAL_INSTAGRAM', 'https://instagram.com/wsplcloud');
define('SOCIAL_LINKEDIN', 'https://linkedin.com/company/wsplcloud');

// ---------------------------------------------------------------
// | Helper Functions                                             |
// ---------------------------------------------------------------

/**
 * Generate a URL for the specified path
 * 
 * @param string $path The path relative to site root
 * @return string Full URL
 */
function url($path = '') {
    $path = ltrim($path, '/');
    return rtrim(BASE_URL, '/') . ($path ? '/' . $path : '');
}

/**
 * Generate a URL for an asset
 * 
 * @param string $path The path relative to assets directory
 * @return string Full asset URL
 */
function asset($path = '') {
    $path = ltrim($path, '/');
    return rtrim(ASSETS_URL, '/') . ($path ? '/' . $path : '');
}

/**
 * Generate a URL for a CSS file
 * 
 * @param string $filename The CSS filename
 * @return string Full CSS URL
 */
function css($filename) {
    $filename = ltrim($filename, '/');
    if (!preg_match('/\.css$/', $filename)) {
        $filename .= '.css';
    }
    return rtrim(CSS_URL, '/') . '/' . $filename;
}

/**
 * Generate a URL for a JavaScript file
 * 
 * @param string $filename The JS filename
 * @return string Full JS URL
 */
function js($filename) {
    $filename = ltrim($filename, '/');
    if (!preg_match('/\.js$/', $filename)) {
        $filename .= '.js';
    }
    return rtrim(JS_URL, '/') . '/' . $filename;
}

/**
 * Generate a URL for an image
 * 
 * @param string $filename The image filename
 * @return string Full image URL
 */
function image($filename) {
    $filename = ltrim($filename, '/');
    return rtrim(IMAGES_URL, '/') . '/' . $filename;
}

/**
 * Check if the current page matches a specific route
 * 
 * @param string $route The route to check
 * @return bool True if current page matches the route
 */
function is_current_page($route) {
    global $current_path;
    return ($current_path === $route);
}

/**
 * Include a partial template file
 * 
 * @param string $name The name of the partial file (without .php extension)
 * @param array $data Optional data to pass to the partial
 * @return void
 */
function partial($name, $data = []) {
    $file = PARTIAL_PATH . DS . $name . '.php';
    if (file_exists($file)) {
        // Extract data to make variables available to the partial
        if (!empty($data) && is_array($data)) {
            extract($data);
        }
        include $file;
    } else {
        error_log("Partial file not found: {$file}");
    }
}

/**
 * Load a page template file
 * 
 * @param string $name The name of the page file (without .php extension)
 * @param array $data Optional data to pass to the page
 * @return void
 */
function load_page($name, $data = []) {
    $file = PAGES_PATH . DS . $name . '.php';
    if (file_exists($file)) {
        // Extract data to make variables available to the page
        if (!empty($data) && is_array($data)) {
            extract($data);
        }
        include $file;
    } else {
        error_log("Page file not found: {$file}");
        // Fallback to 404 page
        http_response_code(404);
        include PAGES_PATH . DS . '404.php';
    }
}

// ---------------------------------------------------------------
// | Error Handling & Debugging                                   |
// ---------------------------------------------------------------

// Load environment variables from .env file
require_once BASE_PATH . DS . 'vendor' . DS . 'autoload.php';
$dotenv = Dotenv\Dotenv::createImmutable(BASE_PATH);
$dotenv->load();

// Set the environment ('development' or 'production')
define('ENVIRONMENT', $_ENV['ENVIRONMENT'] ?? 'development');

// Configure error reporting based on environment
if (ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    ini_set('error_log', LOG_PATH . DS . 'error.log'); // Explicitly set error log path
    
    // Enable debugging functions
    function debug($data, $die = false) {
        echo '<pre>';
        print_r($data);
        echo '</pre>';
        if ($die) die();
    }
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    ini_set('error_log', LOG_PATH . DS . 'error.log'); // Explicitly set error log path for production too
    
    // Create empty debug function for production
    function debug($data, $die = false) {
        // Do nothing in production
        if ($die) die();
    }
}

// Custom error handler
function wspl_error_handler($errno, $errstr, $errfile, $errline) {
    $error_message = date('[Y-m-d H:i:s]') . " Error: [$errno] $errstr in $errfile on line $errline" . PHP_EOL;
    
    // Log all errors
    error_log($error_message); // Use default error_log, which is now set
    
    if (ENVIRONMENT === 'development') {
        // Display detailed error in development
        echo "<div style='color: red; border: 1px solid red; padding: 10px; margin: 10px;'>";
        echo "<h3>Error occurred:</h3>";
        echo "<p><strong>Type:</strong> $errno</p>";
        echo "<p><strong>Message:</strong> $errstr</p>";
        echo "<p><strong>File:</strong> $errfile</p>";
        echo "<p><strong>Line:</strong> $errline</p>";
        echo "</div>";
    } else {
        // In production, only log - don't display
        return true;
    }
}

// Set the custom error handler
set_error_handler('wspl_error_handler');

// Custom shutdown function to catch fatal errors and 404s
function wspl_shutdown_function() {
    $last_error = error_get_last();
    if ($last_error && ($last_error['type'] === E_ERROR || $last_error['type'] === E_PARSE || $last_error['type'] === E_CORE_ERROR || $last_error['type'] === E_COMPILE_ERROR)) {
        $error_message = date('[Y-m-d H:i:s]') . " Fatal Error: [{$last_error['type']}] {$last_error['message']} in {$last_error['file']} on line {$last_error['line']}" . PHP_EOL;
        error_log($error_message); // Use default error_log, which is now set
        if (ENVIRONMENT === 'development') {
            echo "<div style='color: red; border: 1px solid red; padding: 10px; margin: 10px;'>";
            echo "<h3>Fatal Error occurred:</h3>";
            echo "<p><strong>Type:</strong> {$last_error['type']}</p>";
            echo "<p><strong>Message:</strong> {$last_error['message']}</p>";
            echo "<p><strong>File:</strong> {$last_error['file']}</p>";
            echo "<p><strong>Line:</strong> {$last_error['line']}</p>";
            echo "</div>";
        }
    }

    // Log 404 errors specifically
    if (http_response_code() === 404) {
        $request_uri = $_SERVER['REQUEST_URI'] ?? 'UNKNOWN';
        $error_message = date('[Y-m-d H:i:s]') . " 404 Not Found: " . $request_uri . PHP_EOL;
        error_log($error_message); // Use default error_log, which is now set
    }
}

register_shutdown_function('wspl_shutdown_function');

// ---------------------------------------------------------------
// | Security Settings                                            |
// ---------------------------------------------------------------

// Session security settings
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on');

// Start session if not already started (centralized here)
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Sanitize user input
 * 
 * @param mixed $input The input to sanitize
 * @return mixed Sanitized input
 */
function sanitize_input($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitize_input($value);
        }
        return $input;
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Generate CSRF token
 * 
 * @return string CSRF token
 */
function generate_csrf_token() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Create CSRF token field for forms
 * 
 * @return string HTML input field with CSRF token
 */
function csrf_token_field() {
    return '<input type="hidden" name="csrf_token" value="' . generate_csrf_token() . '">';
}

/**
 * Verify CSRF token
 * 
 * @param string $token The token to verify
 * @return bool True if token is valid
 */
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Retrieve the client IP address in a proxy-aware way.
 *
 * @return string
 */
function get_client_ip(): string {
    static $cachedIp = null;
    if ($cachedIp !== null) {
        return $cachedIp;
    }

    $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    if (strpos($ip, ',') !== false) {
        $parts = explode(',', $ip);
        $ip = trim($parts[0]);
    }

    $cachedIp = $ip ?: 'unknown';
    return $cachedIp;
}

/**
 * Ensure the rate limit storage directory exists and return its path.
 *
 * @return string
 */
function rate_limit_storage_path(): ?string {
    static $resolved = false;
    static $cachedPath = null;

    if ($resolved) {
        return $cachedPath;
    }

    $resolved = true;
    $dir = LOG_PATH . DS . 'rate_limits';

    if (!is_dir($dir)) {
        if (!is_writable(LOG_PATH)) { // Check writability of LOG_PATH directly
            error_log('Rate limit base directory not writable: ' . LOG_PATH);
            return $cachedPath = null;
        }

        if (!@mkdir($dir, 0755, true) && !is_dir($dir)) {
            error_log('Rate limit directory could not be created: ' . $dir);
            return $cachedPath = null;
        }
    }

    if (!is_writable($dir)) {
        error_log('Rate limit directory not writable: ' . $dir);
        return $cachedPath = null;
    }

    return $cachedPath = $dir;
}

/**
 * Simple file-based rate limiting helper.
 *
 * @param string $key Identifier per action/IP combo.
 * @param int $limit Number of allowed attempts.
 * @param int $window Time window in seconds.
 * @return bool True when within limits, false when throttled.
 */
function check_rate_limit(string $key, int $limit = 5, int $window = 300): bool {
    $dir = rate_limit_storage_path();
    if (empty($dir)) {
        // Fail-open if storage is unavailable.
        return true;
    }

    $file = $dir . DS . sha1($key) . '.json';
    $now = time();
    $attempts = [];

    if (is_file($file)) {
        $contents = file_get_contents($file);
        if ($contents !== false) {
            $decoded = json_decode($contents, true);
            if (is_array($decoded)) {
                $attempts = $decoded;
            }
        }
    }

    // Drop attempts outside the window.
    $attempts = array_values(array_filter($attempts, static function ($timestamp) use ($now, $window) {
        return is_int($timestamp) && ($now - $timestamp) < $window;
    }));

    if (count($attempts) >= $limit) {
        return false;
    }

    $attempts[] = $now;
    @file_put_contents($file, json_encode($attempts), LOCK_EX);
    return true;
}

// ---------------------------------------------------------------
// | Routing                                                      |
// ---------------------------------------------------------------

/**
 * Get the current route path
 * 
 * @return string The current route path
 */
function get_current_path() {
    $path = $_SERVER['REQUEST_URI'] ?? '/';
    $path = parse_url($path, PHP_URL_PATH);
    $path = trim($path, '/');
    
    // Remove site path prefix if it exists
    $site_dir = trim(str_replace(['/', '\\'], '/', SITE_ROOT), '/');
    if ($site_dir && strpos($path, $site_dir) === 0) {
        $path = substr($path, strlen($site_dir));
    }
    
    return trim($path, '/');
}

// Set the current path
$current_path = get_current_path();
$GLOBALS['current_path'] = $current_path;

// ---------------------------------------------------------------
// | Database (PDO) helper                                         |
// ---------------------------------------------------------------
// Default DB configuration (change these in a site-specific config if needed)
// Database configuration
// IMPORTANT: In production, these values should be loaded from environment variables
// or a secure, non-version-controlled configuration file (e.g., .env)
define('DB_HOST', $_ENV['DB_HOST'] ?? '127.0.0.1');
define('DB_NAME', $_ENV['DB_NAME'] ?? 'wsplcloud');
define('DB_USER', $_ENV['DB_USER'] ?? 'root');
define('DB_PASS', $_ENV['DB_PASSWORD'] ?? ''); // Use DB_PASSWORD from env

// Email settings
define('MAIL_FROM', $_ENV['MAIL_FROM'] ?? 'noreply@wsplcloud.com');
define('MAIL_FROM_NAME', $_ENV['MAIL_FROM_NAME'] ?? 'WsplCloud Support');
define('SMTP_HOST', $_ENV['SMTP_HOST'] ?? '');
define('SMTP_PORT', $_ENV['SMTP_PORT'] ?? 587);
define('SMTP_USERNAME', $_ENV['SMTP_USERNAME'] ?? '');
define('SMTP_PASSWORD', $_ENV['SMTP_PASSWORD'] ?? '');
define('SMTP_ENCRYPTION', $_ENV['SMTP_ENCRYPTION'] ?? 'tls'); // or 'ssl' or ''

// Include the SMTP Mailer class
require_once BASE_PATH . DS . 'admin' . DS . 'includes' . DS . 'smtp_mailer.php';

// Override the default send_email function to use SMTP
if (!function_exists('send_email')) {
    function send_email(string $to, string $subject, string $message, array $headers = []): bool {
        // If SMTP settings are provided, use the SmtpMailer
        if (SMTP_HOST && SMTP_USERNAME && SMTP_PASSWORD) {
            $mailer = new SmtpMailer(
                SMTP_HOST,
                SMTP_PORT,
                SMTP_USERNAME,
                SMTP_PASSWORD,
                SMTP_ENCRYPTION,
                MAIL_FROM,
                MAIL_FROM_NAME
            );
            return $mailer->send($to, $subject, $message, $headers);
        } else {
            // In production, SMTP must be configured. Do not fallback to PHP's mail() function.
            error_log("SMTP not configured. Email to: $to failed.");
            return false;
        }
    }
}

/**
 * Return a singleton PDO connection. Creates database if necessary when possible.
 * @return \PDO|null
 */
function get_db() {
    static $pdo = null;
    if ($pdo instanceof PDO) return $pdo;

    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
    try {
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        // try creating database if it doesn't exist
        // In production, do not attempt to create the database automatically.
        // The database should be created and migrated manually or via deployment scripts.
        if (ENVIRONMENT === 'development') {
            try {
                $tmp = new PDO('mysql:host=' . DB_HOST . ';charset=utf8mb4', DB_USER, DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
                $tmp->exec("CREATE DATABASE IF NOT EXISTS `" . str_replace('`','',DB_NAME) . "` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;");
                // reconnect
                $pdo = new PDO($dsn, DB_USER, DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION, PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC]);
                return $pdo;
            } catch (PDOException $e2) {
                error_log('DB connection error during auto-creation attempt: ' . $e2->getMessage());
                return null;
            }
        } else {
            error_log('DB connection error in production: ' . $e->getMessage());
            return null;
        }
    }
}

// No longer using ensure_db_schema directly.
// Database schema is now managed via migration scripts in scripts/migrations.
// To run migrations, use: php scripts/migrate.php migrate
// To rollback, use: php scripts/migrate.php rollback
// To check status, use: php scripts/migrate.php status

// ---------------------------------------------------------------
// | Initialization Complete                                      |
// ---------------------------------------------------------------

// The configuration is now loaded and ready to use

// ---------------------------------------------------------------
// | Pricing Plans Configuration                                 |
// ---------------------------------------------------------------

// Define pricing plans (for use across multiple pages)
$GLOBALS['pricing_plans'] = [
    'shared' => [
        'basic' => [
            'name' => 'Basic',
            'monthly_price' => 99,
            'yearly_price' => 999,
            'storage' => '10GB',
            'bandwidth' => '100GB',
            'websites' => 1,
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                '24/7 Support',
                'cPanel Access'
            ]
        ],
        'standard' => [
            'name' => 'Standard',
            'monthly_price' => 199,
            'yearly_price' => 1999,
            'storage' => '25GB',
            'bandwidth' => '500GB',
            'websites' => 5,
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                '24/7 Support',
                'cPanel Access',
                'Free Domain'
            ]
        ],
        'premium' => [
            'name' => 'Premium',
            'monthly_price' => 399,
            'yearly_price' => 3999,
            'storage' => '100GB',
            'bandwidth' => 'Unlimited',
            'websites' => 10,
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                '24/7 Support',
                'cPanel Access',
                'Free Domain',
                'Priority Support'
            ]
        ]
    ],
    'cloud' => [
        'starter' => [
            'name' => 'Starter',
            'monthly_price' => 199,
            'yearly_price' => 1999,
            'cpu' => '1 Core',
            'ram' => '2GB',
            'storage' => '10GB SSD',
            'bandwidth' => '2TB',
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                'DDoS Protection',
                'Root Access'
            ]
        ],
        'business' => [
            'name' => 'Business',
            'monthly_price' => 399,
            'yearly_price' => 3999,
            'cpu' => '2 Cores',
            'ram' => '4GB',
            'storage' => '20GB SSD',
            'bandwidth' => '3TB',
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                'DDoS Protection',
                'Root Access',
                'Free Domain'
            ]
        ],
        'enterprise' => [
            'name' => 'Enterprise',
            'monthly_price' => 799,
            'yearly_price' => 7999,
            'cpu' => '4 Cores',
            'ram' => '8GB',
            'storage' => '50GB SSD',
            'bandwidth' => '5TB',
            'features' => [
                'Free SSL Certificate',
                'Daily Backups',
                'DDoS Protection',
                'Root Access',
                'Free Domain',
                'Priority Support'
            ]
        ]
    ]
];

// Load page-specific configurations if they exist
$page_config_file = BASE_PATH . DS . 'config' . DS . ($GLOBALS['current_path'] ?? 'default') . '.php';
if (file_exists($page_config_file)) {
    require_once $page_config_file;
} else {
    // If no specific page config, load the default config
    $default_config_file = BASE_PATH . DS . 'config' . DS . 'default.php';
    if (file_exists($default_config_file)) {
        require_once $default_config_file;
    }
}
