<?php
/**
 * Admin Panel - User Management
 * Displays a list of registered users and handles add/edit/delete operations.
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/includes/header.php';

$message = '';
$error = '';

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
        $error = 'Invalid CSRF token. Please try again.';
    } else {
        $action = $_POST['action'] ?? '';
        $user_id = $_POST['user_id'] ?? null;
        $username = sanitize_input($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'user'; // Do not sanitize role with htmlspecialchars yet, validate first

        // Validate role input
        $allowed_roles = ['admin', 'user'];
        if (!in_array($role, $allowed_roles)) {
            $error = 'Invalid role specified.';
        } else {
            // Sanitize role after validation
            $role = htmlspecialchars($role, ENT_QUOTES, 'UTF-8');

            if ($db) {
                if ($action === 'add') {
                    if (empty($username) || empty($password)) {
                        $error = 'Username and password are required for adding a user.';
                    } else {
                        // Enforce password policy (placeholder)
                        // if (!is_strong_password($password)) {
                        //     $error = 'Password does not meet complexity requirements.';
                        // } else {
                            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                            try {
                                $stmt = $db->prepare("INSERT INTO users (username, password, role) VALUES (:username, :password, :role)");
                                $stmt->bindParam(':username', $username);
                                $stmt->bindParam(':password', $hashed_password);
                                $stmt->bindParam(':role', $role);
                                $stmt->execute();
                                $message = 'User added successfully!';
                                header('Location: ' . url('/admin/users.php'));
                                exit;
                            } catch (PDOException $e) {
                                error_log('Add user error: ' . $e->getMessage());
                                if ($e->getCode() === '23000') {
                                    $error = 'Username already exists. Please choose a different username.';
                                } else {
                                    $error = 'Error adding user. Please try again later.';
                                }
                            }
                        // }
                    }
                } elseif ($action === 'edit') {
                    if (!$user_id) {
                        $error = 'User ID is missing for editing.';
                    } else {
                        $update_fields = ['username' => $username, 'role' => $role];
                        $update_sql = "UPDATE users SET username = :username, role = :role";
                        
                        if (!empty($password)) {
                            // Enforce password policy (placeholder)
                            // if (!is_strong_password($password)) {
                            //     $error = 'New password does not meet complexity requirements.';
                            // } else {
                                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                                $update_sql .= ", password = :password";
                                $update_fields['password'] = $hashed_password;
                            // }
                        }
                        
                        if (empty($error)) { // Only proceed if no password policy error
                            $update_sql .= " WHERE id = :id";
                            
                            try {
                                $stmt = $db->prepare($update_sql);
                                $stmt->bindParam(':username', $username);
                                $stmt->bindParam(':role', $role);
                                if (!empty($password)) {
                                    $stmt->bindParam(':password', $hashed_password);
                                }
                                $stmt->bindParam(':id', $user_id);
                                $stmt->execute();
                                $message = 'User updated successfully!';
                                
                                // Invalidate user session if their own password was changed (placeholder)
                                // if ($user_id == ($_SESSION['admin_user']['id'] ?? 0) && !empty($password)) {
                                //     session_destroy();
                                //     header('Location: ' . url('admin/login.php'));
                                //     exit;
                                // }

                                header('Location: ' . url('/admin/users.php'));
                                exit;
                            } catch (PDOException $e) {
                                error_log('Edit user error: ' . $e->getMessage());
                                if ($e->getCode() === '23000') {
                                    $error = 'Username already exists. Please choose a different username.';
                                } else {
                                    $error = 'Error updating user. Please try again later.';
                                }
                            }
                        }
                    }
                }
            } else {
                $error = 'Database connection failed.';
            }
        }
    }
}

// Handle GET requests
$action = $_GET['action'] ?? '';
$user_id_get = $_GET['id'] ?? null;

if ($action === 'delete') {
    if (!$user_id_get) {
        $error = 'User ID is missing for deletion.';
    } elseif ($db) {
        try {
            if (isset($_SESSION['admin_user']) && $_SESSION['admin_user']['id'] == $user_id_get && $_SESSION['admin_user']['role'] == 'admin') {
                 $error = 'You cannot delete the currently logged-in administrator account.';
            } else {
                $stmt = $db->prepare("DELETE FROM users WHERE id = :id");
                $stmt->bindParam(':id', $user_id_get);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $message = 'User deleted successfully!';
                } else {
                    $error = 'User not found or could not be deleted.';
                }
                header('Location: ' . url('/admin/users.php'));
                exit;
            }
        } catch (PDOException $e) {
            error_log('Delete user error: ' . $e->getMessage());
            $error = 'Error deleting user. Please try again later.';
        }
    } else {
        $error = 'Database connection failed.';
    }
}

// Fetch users
$users = [];
if ($db) {
    try {
        $stmt = $db->query("SELECT id, username, role, created FROM users ORDER BY created DESC");
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log('User list fetch error: ' . $e->getMessage());
    }
}
?>

<style>
:root {
    --primary-blue: #0000ff;
    --blue-dark: #0000cc;
    --blue-light: #3366ff;
    --blue-glow: rgba(0, 0, 255, 0.2);
    --success-color: #00d97e;
    --warning-color: #ffc107;
    --danger-color: #e74c3c;
}

.user-management {
    animation: fadeInUp 0.6s ease-out;
}

.stats-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid rgba(0, 0, 0, 0.05);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--primary-blue);
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 30px var(--blue-glow);
}

.stat-card.admin::before { background: var(--primary-blue); }
.stat-card.user::before { background: var(--success-color); }
.stat-card.total::before { background: #6c757d; }

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    color: #6c757d;
    font-weight: 500;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.user-table-container {
    background: white;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    overflow: hidden;
    animation: slideInUp 0.6s ease-out;
}

.table-header {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    color: white;
    padding: 1.5rem 2rem;
    display: flex;
    justify-content: between;
    align-items: center;
}

.table-header h3 {
    margin: 0;
    font-weight: 600;
}

.table-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.search-box {
    position: relative;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 10px;
    padding: 0.5rem 1rem;
    min-width: 250px;
}

.search-box input {
    background: transparent;
    border: none;
    color: white;
    width: 100%;
    padding-left: 2rem;
}

.search-box input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

.search-box i {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: rgba(255, 255, 255, 0.7);
}

.btn-add-user {
    background: rgba(255, 255, 255, 0.2);
    border: 2px solid rgba(255, 255, 255, 0.3);
    color: white;
    padding: 0.5rem 1.5rem;
    border-radius: 10px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-add-user:hover {
    background: white;
    color: var(--primary-blue);
    transform: translateY(-2px);
}

.table {
    margin: 0;
}

.table th {
    background: #f8f9fa;
    border-bottom: 2px solid #e9ecef;
    font-weight: 600;
    color: #2c3e50;
    padding: 1rem 1.5rem;
    text-transform: uppercase;
    font-size: 0.8rem;
    letter-spacing: 0.5px;
}

.table td {
    padding: 1rem 1.5rem;
    vertical-align: middle;
    border-color: #f8f9fa;
}

.user-avatar {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 1rem;
    margin-right: 1rem;
}

.user-info {
    display: flex;
    align-items: center;
}

.role-badge {
    padding: 0.3rem 0.8rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.role-badge.admin {
    background: rgba(0, 0, 255, 0.1);
    color: var(--primary-blue);
    border: 1px solid rgba(0, 0, 255, 0.2);
}

.role-badge.user {
    background: rgba(0, 217, 126, 0.1);
    color: var(--success-color);
    border: 1px solid rgba(0, 217, 126, 0.2);
}

.action-buttons {
    display: flex;
    gap: 0.5rem;
}

.btn-action {
    width: 35px;
    height: 35px;
    border: none;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-edit {
    background: rgba(0, 0, 255, 0.1);
    color: var(--primary-blue);
}

.btn-edit:hover {
    background: var(--primary-blue);
    color: white;
    transform: scale(1.1);
}

.btn-delete {
    background: rgba(231, 76, 60, 0.1);
    color: var(--danger-color);
}

.btn-delete:hover {
    background: var(--danger-color);
    color: white;
    transform: scale(1.1);
}

/* Form Styles */
.user-form {
    background: white;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    overflow: hidden;
    animation: slideInUp 0.6s ease-out;
}

.form-header {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    color: white;
    padding: 1.5rem 2rem;
}

.form-header h3 {
    margin: 0;
    font-weight: 600;
}

.form-body {
    padding: 2rem;
}

.form-control {
    border: 2px solid #e9ecef;
    border-radius: 10px;
    padding: 0.75rem 1rem;
    transition: all 0.3s ease;
}

.form-control:focus {
    border-color: var(--primary-blue);
    box-shadow: 0 0 0 3px rgba(0, 0, 255, 0.1);
    transform: translateY(-2px);
}

.btn-submit {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 10px;
    color: white;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-submit:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px var(--blue-glow);
}

/* Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(50px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive */
@media (max-width: 768px) {
    .stats-cards {
        grid-template-columns: 1fr;
    }
    
    .table-header {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .table-actions {
        flex-direction: column;
    }
    
    .search-box {
        min-width: auto;
    }
    
    .user-info {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
    
    .user-avatar {
        margin-right: 0;
    }
}
</style>

<div class="user-management">
    <?php if (!empty($message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i><?php echo $message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <?php if ($action === 'add' || $action === 'edit'): ?>
        <?php
        $user_data = ['id' => '', 'username' => '', 'role' => 'user'];
        if ($action === 'edit' && $user_id_get && $db) {
            try {
                $stmt = $db->prepare("SELECT id, username, role FROM users WHERE id = :id");
                $stmt->bindParam(':id', $user_id_get);
                $stmt->execute();
                $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                if (!$user_data) {
                    $error = 'User not found.';
                    $action = '';
                }
            } catch (PDOException $e) {
                error_log('Fetch user for edit error: ' . $e->getMessage());
                $error = 'Error fetching user data.';
            }
        }
        ?>
        
        <div class="user-form">
            <div class="form-header">
                <h3><i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'edit'; ?> me-2"></i><?php echo ucfirst($action); ?> User</h3>
            </div>
            <div class="form-body">
                <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST">
                    <?php echo csrf_token_field(); ?>
                    <input type="hidden" name="action" value="<?php echo $action; ?>">
                    <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="username" name="username" 
                                   value="<?php echo htmlspecialchars($user_data['username']); ?>" required>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="password" class="form-label">Password <?php echo ($action === 'edit') ? '(Leave blank to keep current)' : ''; ?></label>
                            <input type="password" class="form-control" id="password" name="password" 
                                   <?php echo ($action === 'add') ? 'required' : ''; ?>>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="role" class="form-label">Role</label>
                            <select class="form-select" id="role" name="role" required>
                                <option value="admin" <?php echo ($user_data['role'] === 'admin') ? 'selected' : ''; ?>>Admin</option>
                                <option value="user" <?php echo ($user_data['role'] === 'user') ? 'selected' : ''; ?>>User</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="d-flex gap-2 mt-4">
                        <button type="submit" class="btn btn-submit">
                            <i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'save'; ?> me-2"></i>
                            <?php echo ucfirst($action); ?> User
                        </button>
                        <a href="<?php echo url('/admin/users.php'); ?>" class="btn btn-outline-secondary">
                            <i class="fas fa-arrow-left me-2"></i>Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
    <?php else: ?>
        <!-- Statistics Cards -->
        <?php
        $total_users = count($users);
        $admin_users = array_filter($users, function($user) { return $user['role'] === 'admin'; });
        $regular_users = array_filter($users, function($user) { return $user['role'] === 'user'; });
        ?>
        
        <div class="stats-cards">
            <div class="stat-card total">
                <div class="stat-number"><?php echo $total_users; ?></div>
                <div class="stat-label">Total Users</div>
                <i class="fas fa-users fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
            </div>
            <div class="stat-card admin">
                <div class="stat-number"><?php echo count($admin_users); ?></div>
                <div class="stat-label">Administrators</div>
                <i class="fas fa-shield-alt fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
            </div>
            <div class="stat-card user">
                <div class="stat-number"><?php echo count($regular_users); ?></div>
                <div class="stat-label">Regular Users</div>
                <i class="fas fa-user fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
            </div>
        </div>

        <!-- User Table -->
        <div class="user-table-container">
            <div class="table-header">
                <h3><i class="fas fa-users me-2"></i>User Management</h3>
                <div class="table-actions">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" id="userSearch" placeholder="Search users...">
                    </div>
                    <a href="<?php echo url('/admin/users.php?action=add'); ?>" class="btn-add-user">
                        <i class="fas fa-plus"></i> Add User
                    </a>
                </div>
            </div>
            
            <div class="table-responsive">
                <table class="table table-hover mb-0" id="usersTable">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Role</th>
                            <th>Registered</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($users)): ?>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td>
                                        <div class="user-info">
                                            <div class="user-avatar">
                                                <?php echo strtoupper(substr($user['username'], 0, 1)); ?>
                                            </div>
                                            <div>
                                                <div class="fw-semibold"><?php echo htmlspecialchars($user['username']); ?></div>
                                                <small class="text-muted">ID: <?php echo $user['id']; ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="role-badge <?php echo $user['role']; ?>">
                                            <?php echo htmlspecialchars($user['role']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="text-muted">
                                            <?php echo date('M j, Y', strtotime($user['created'])); ?>
                                        </div>
                                        <small class="text-muted">
                                            <?php echo date('g:i A', strtotime($user['created'])); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="<?php echo url('/admin/users.php?action=edit&id=' . $user['id']); ?>" 
                                               class="btn-action btn-edit" title="Edit User">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <?php if ($user['id'] != ($_SESSION['admin_user']['id'] ?? 0)): ?>
                                                <button class="btn-action btn-delete" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#deleteUserModal" 
                                                        data-user-id="<?php echo $user['id']; ?>"
                                                        data-user-name="<?php echo htmlspecialchars($user['username']); ?>"
                                                        title="Delete User">
                                                    <i class="fas fa-trash-alt"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4" class="text-center py-4">
                                    <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                    <h5 class="text-muted">No users found</h5>
                                    <p class="text-muted">Get started by adding your first user.</p>
                                    <a href="<?php echo url('/admin/users.php?action=add'); ?>" class="btn btn-primary">
                                        <i class="fas fa-plus me-2"></i>Add User
                                    </a>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- Delete User Modal -->
<div class="modal fade" id="deleteUserModal" tabindex="-1" aria-labelledby="deleteUserModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title text-danger" id="deleteUserModalLabel">
                    <i class="fas fa-exclamation-triangle me-2"></i>Confirm Deletion
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body text-center py-4">
                <i class="fas fa-trash-alt fa-4x text-danger mb-3"></i>
                <h4>Delete User?</h4>
                <p class="text-muted">Are you sure you want to delete <strong id="userNameToDelete"></strong>? This action cannot be undone.</p>
            </div>
            <div class="modal-footer border-0 justify-content-center">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteBtn">
                    <i class="fas fa-trash-alt me-2"></i>Delete User
                </button>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Search functionality
    const searchInput = document.getElementById('userSearch');
    const usersTable = document.getElementById('usersTable');
    
    if (searchInput && usersTable) {
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = usersTable.getElementsByTagName('tbody')[0].getElementsByTagName('tr');
            
            for (let row of rows) {
                const username = row.cells[0].textContent.toLowerCase();
                const role = row.cells[1].textContent.toLowerCase();
                const shouldShow = username.includes(searchTerm) || role.includes(searchTerm);
                row.style.display = shouldShow ? '' : 'none';
            }
        });
    }
    
    // Delete modal functionality
    const deleteUserModal = document.getElementById('deleteUserModal');
    const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');
    const userNameToDelete = document.getElementById('userNameToDelete');
    let userIdToDelete = null;
    
    if (deleteUserModal) {
        deleteUserModal.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            userIdToDelete = button.getAttribute('data-user-id');
            const userName = button.getAttribute('data-user-name');
            userNameToDelete.textContent = userName;
        });
        
        confirmDeleteBtn.addEventListener('click', function() {
            if (userIdToDelete) {
                window.location.href = `<?php echo url('/admin/users.php'); ?>?action=delete&id=${userIdToDelete}`;
            }
        });
    }
    
    // Add animation to table rows
    const tableRows = document.querySelectorAll('#usersTable tbody tr');
    tableRows.forEach((row, index) => {
        row.style.animationDelay = `${index * 0.1}s`;
        row.classList.add('animate__animated', 'animate__fadeInUp');
    });
});
</script>

<?php
$footer_file = __DIR__ . '/includes/footer.php';
if (file_exists($footer_file)) {
    require_once $footer_file;
} else {
    // Minimal fallback footer to avoid fatal errors if include is missing
    echo "</main></div><!-- .admin-container --></div><!-- .admin-layout -->";
    echo "<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js\"></script>";
}
?>
