<?php
/**
 * Admin Panel - User Profile Settings
 * Allows administrators to view and update their profile information, primarily password.
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/includes/header.php';

$message = '';
$error = '';

// Get current user's data from database to ensure fresh data
$user_id = $_SESSION['admin_user']['id'] ?? null;

if (!$user_id) {
    header('Location: ' . (function_exists('url') ? url('admin/login.php') : SITE_ROOT . '/admin/login.php'));
    exit;
}

// Fetch latest user data from database
$stmt_user = $db->prepare("SELECT id, username, full_name, role, profile_image, created FROM users WHERE id = :id");
$stmt_user->bindParam(':id', $user_id);
$stmt_user->execute();
$current_user = $stmt_user->fetch(PDO::FETCH_ASSOC);

if (!$current_user) {
    header('Location: ' . (function_exists('url') ? url('admin/login.php') : SITE_ROOT . '/admin/login.php'));
    exit;
}

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF token verification
    if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
        $error = 'Invalid CSRF token. Please try again.';
    } else {
        $action = $_POST['action'] ?? '';
        
        // Handle profile info update (username and image)
        if ($action === 'update_profile') {
            $new_username = trim($_POST['username'] ?? '');
            $new_fullname = trim($_POST['full_name'] ?? '');
            $profile_image_path = $current_user['profile_image'];
            
            // Validate username
            if (empty($new_username)) {
                $error = 'Username cannot be empty.';
            } elseif (!preg_match('/^[a-zA-Z0-9_]{3,50}$/', $new_username)) {
                $error = 'Username must be 3-50 characters and contain only letters, numbers, and underscores.';
            } else {
                // Check if username is already taken by another user
                $stmt_check = $db->prepare("SELECT id FROM users WHERE username = :username AND id != :id");
                $stmt_check->bindParam(':username', $new_username);
                $stmt_check->bindParam(':id', $user_id);
                $stmt_check->execute();
                
                if ($stmt_check->fetch()) {
                    $error = 'Username is already taken. Please choose another.';
                } else {
                    // Handle profile image upload
                    if (isset($_FILES['profile_image']) && $_FILES['profile_image']['error'] === UPLOAD_ERR_OK) {
                        $file = $_FILES['profile_image'];
                        $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                        $max_size = 5 * 1024 * 1024; // 5MB
                        
                        if (!in_array($file['type'], $allowed_types)) {
                            $error = 'Invalid image type. Only JPG, PNG, GIF, and WebP are allowed.';
                        } elseif ($file['size'] > $max_size) {
                            $error = 'Image size must be less than 5MB.';
                        } else {
                            // Generate unique filename
                            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                            $filename = 'profile_' . $user_id . '_' . time() . '.' . $extension;
                            $upload_dir = dirname(__DIR__) . '/assets/uploads/profiles/';
                            $upload_path = $upload_dir . $filename;
                            
                            // Ensure directory exists with proper permissions
                            if (!is_dir($upload_dir)) {
                                mkdir($upload_dir, 0777, true);
                            }
                            
                            if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                                // Delete old profile image if exists
                                if ($current_user['profile_image'] && file_exists(dirname(__DIR__) . '/' . $current_user['profile_image'])) {
                                    unlink(dirname(__DIR__) . '/' . $current_user['profile_image']);
                                }
                                $profile_image_path = 'assets/uploads/profiles/' . $filename;
                            } else {
                                $error = 'Failed to upload image. Please check directory permissions.';
                            }
                        }
                    }
                    
                    if (!$error) {
                        try {
                            $stmt_update = $db->prepare("UPDATE users SET username = :username, full_name = :full_name, profile_image = :profile_image WHERE id = :id");
                            $stmt_update->bindParam(':username', $new_username);
                            $stmt_update->bindParam(':full_name', $new_fullname);
                            $stmt_update->bindParam(':profile_image', $profile_image_path);
                            $stmt_update->bindParam(':id', $user_id);
                            $stmt_update->execute();
                            
                            // Update session data
                            $_SESSION['admin_user']['username'] = $new_username;
                            $_SESSION['admin_user']['name'] = $new_fullname ?: $new_username;
                            $_SESSION['admin_user']['profile_image'] = $profile_image_path;
                            
                            // Refresh current_user data
                            $current_user['username'] = $new_username;
                            $current_user['full_name'] = $new_fullname;
                            $current_user['profile_image'] = $profile_image_path;
                            
                            $message = 'Profile updated successfully!';
                        } catch (PDOException $e) {
                            error_log('Profile update error: ' . $e->getMessage());
                            $error = 'Error updating profile. Please try again later.';
                        }
                    }
                }
            }
        }
        
        // Handle password change
        elseif ($action === 'change_password') {
            $current_password = $_POST['current_password'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';

            if ($db) {
                if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                    $error = 'All password fields are required.';
                } elseif ($new_password !== $confirm_password) {
                    $error = 'New password and confirm password do not match.';
                } elseif (strlen($new_password) < 8) {
                    $error = 'Password must be at least 8 characters long.';
                } else {
                    // Verify current password
                    $stmt_verify = $db->prepare("SELECT password FROM users WHERE id = :id");
                    $stmt_verify->bindParam(':id', $user_id);
                    $stmt_verify->execute();
                    $user_db_data = $stmt_verify->fetch(PDO::FETCH_ASSOC);

                    if ($user_db_data && password_verify($current_password, $user_db_data['password'])) {
                        // Current password is correct, proceed with updating new password
                        $hashed_new_password = password_hash($new_password, PASSWORD_DEFAULT);
                        
                        try {
                            $stmt_update = $db->prepare("UPDATE users SET password = :password WHERE id = :id");
                            $stmt_update->bindParam(':password', $hashed_new_password);
                            $stmt_update->bindParam(':id', $user_id);
                            $stmt_update->execute();
                            
                            $message = 'Password updated successfully!';
                            
                        } catch (PDOException $e) {
                            error_log('Profile update error: ' . $e->getMessage());
                            $error = 'Error updating password. Please try again later.';
                        }
                    } else {
                        $error = 'Incorrect current password.';
                    }
                }
            } else {
                $error = 'Database connection failed.';
            }
        }
    }
}

// Display messages and errors
?>

<style>
:root {
    --primary-blue: #0000ff;
    --primary-dark: #0000cc;
    --gold: #FFD700;
    --text-dark: #2c3e50;
    --text-muted: #6c757d;
}

.profile-container {
    max-width: 900px;
    margin: 0 auto;
    padding: 2rem 0;
    animation: fadeInUp 0.6s ease-out;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.profile-header {
    background: linear-gradient(135deg, var(--primary-blue) 0%, var(--primary-dark) 50%, var(--primary-blue) 100%);
    border-radius: 24px;
    padding: 2.5rem;
    margin-bottom: 2rem;
    color: white;
    text-align: center;
    border: 3px solid var(--gold);
    position: relative;
    overflow: hidden;
}

.profile-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: radial-gradient(circle at top right, rgba(255, 215, 0, 0.2), transparent 70%);
    pointer-events: none;
}

.profile-avatar {
    width: 120px;
    height: 120px;
    background: white;
    border-radius: 50%;
    margin: 0 auto 1.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 3rem;
    font-weight: 700;
    color: var(--primary-blue);
    border: 4px solid var(--gold);
    position: relative;
    z-index: 1;
    overflow: hidden;
}

.profile-avatar img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.profile-avatar-text {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 100%;
    height: 100%;
}

.profile-name {
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    position: relative;
    z-index: 1;
}

.profile-role {
    font-size: 1.1rem;
    opacity: 0.95;
    text-transform: uppercase;
    letter-spacing: 1px;
    background: rgba(255, 215, 0, 0.3);
    padding: 0.5rem 1.5rem;
    border-radius: 20px;
    display: inline-block;
    position: relative;
    z-index: 1;
}

.profile-content {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

.info-card, .password-card {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    border: 2px solid #e0e0e0;
}

.card-title {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--primary-blue);
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.card-title i {
    font-size: 1.3rem;
}

.info-item {
    margin-bottom: 1.5rem;
}

.info-label {
    font-size: 0.85rem;
    font-weight: 600;
    color: var(--text-muted);
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 0.5rem;
}

.info-value {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-dark);
    padding: 0.75rem;
    background: #f8f9fa;
    border-radius: 10px;
    border-left: 4px solid var(--primary-blue);
}

.form-group-modern {
    margin-bottom: 1.5rem;
}

.form-group-modern label {
    font-size: 0.9rem;
    font-weight: 600;
    color: var(--text-dark);
    margin-bottom: 0.5rem;
    display: block;
}

.form-control-modern {
    width: 100%;
    padding: 0.875rem 1rem;
    border: 2px solid #e0e0e0;
    border-radius: 12px;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.form-control-modern:focus {
    outline: none;
    border-color: var(--primary-blue);
    box-shadow: 0 0 0 3px rgba(0, 0, 255, 0.1);
}

.password-strength {
    margin-top: 0.5rem;
    font-size: 0.85rem;
}

.strength-bar {
    height: 4px;
    background: #e0e0e0;
    border-radius: 2px;
    margin-top: 0.5rem;
    overflow: hidden;
}

.strength-fill {
    height: 100%;
    width: 0%;
    transition: all 0.3s ease;
}

.btn-update {
    width: 100%;
    padding: 1rem;
    background: linear-gradient(135deg, var(--primary-blue), var(--primary-dark));
    color: white;
    border: none;
    border-radius: 12px;
    font-size: 1.1rem;
    font-weight: 700;
    cursor: pointer;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 1px;
}

.btn-update:hover {
    transform: translateY(-2px);
    background: linear-gradient(135deg, var(--primary-dark), var(--primary-blue));
}

.alert-modern {
    padding: 1.25rem 1.5rem;
    border-radius: 12px;
    margin-bottom: 1.5rem;
    font-weight: 500;
    animation: slideDown 0.3s ease;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.alert-success-modern {
    background: #d4edda;
    color: #155724;
    border-left: 4px solid #28a745;
}

.alert-error-modern {
    background: #f8d7da;
    color: #721c24;
    border-left: 4px solid #dc3545;
}

.password-requirements {
    background: #f8f9fa;
    padding: 1rem;
    border-radius: 10px;
    margin-top: 1rem;
    font-size: 0.85rem;
}

.requirement {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
    color: var(--text-muted);
}

.requirement i {
    font-size: 0.75rem;
}

.requirement.valid {
    color: #28a745;
}

.image-upload-wrapper {
    position: relative;
    margin-bottom: 1.5rem;
}

.current-profile-image {
    width: 150px;
    height: 150px;
    border-radius: 50%;
    margin: 0 auto 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    background: white;
    border: 3px solid var(--primary-blue);
    overflow: hidden;
    font-size: 3.5rem;
    font-weight: 700;
    color: var(--primary-blue);
}

.current-profile-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.image-upload-label {
    display: block;
    text-align: center;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, var(--primary-blue), var(--primary-dark));
    color: white;
    border-radius: 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    font-weight: 600;
}

.image-upload-label:hover {
    transform: translateY(-2px);
}

.image-upload-label i {
    margin-right: 0.5rem;
}

#profile_image_input {
    display: none;
}

.image-preview {
    margin-top: 1rem;
    text-align: center;
    font-size: 0.9rem;
    color: var(--text-muted);
}

@media (max-width: 768px) {
    .profile-content {
        grid-template-columns: 1fr;
    }
    
    .profile-header {
        padding: 2rem 1rem;
    }
    
    .profile-avatar {
        width: 100px;
        height: 100px;
        font-size: 2.5rem;
    }
}
</style>

<div class="profile-container">
    <?php if (!empty($message)): ?>
        <div class="alert-modern alert-success-modern">
            <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($error)): ?>
        <div class="alert-modern alert-error-modern">
            <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <!-- Profile Header -->
    <div class="profile-header">
        <div class="profile-avatar">
            <?php if (!empty($current_user['profile_image']) && file_exists(__DIR__ . '/../' . $current_user['profile_image'])): ?>
                <img src="<?php echo url($current_user['profile_image']); ?>" alt="Profile">
            <?php else: ?>
                <div class="profile-avatar-text">
                    <?php echo strtoupper(substr($current_user['username'], 0, 2)); ?>
                </div>
            <?php endif; ?>
        </div>
        <div class="profile-name"><?php echo htmlspecialchars($current_user['full_name'] ?: $current_user['username']); ?></div>
        <div class="profile-role">
            <i class="fas fa-shield-alt"></i> <?php echo htmlspecialchars($current_user['role']); ?>
        </div>
    </div>

    <!-- Profile Content -->
    <div class="profile-content">
        <!-- Account Information & Update -->
        <div class="info-card">
            <h3 class="card-title">
                <i class="fas fa-user-circle"></i>
                Profile Information
            </h3>
            
            <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST" enctype="multipart/form-data" id="profileForm">
                <?php echo csrf_token_field(); ?>
                <input type="hidden" name="action" value="update_profile">
                
                <!-- Profile Image Upload -->
                <div class="image-upload-wrapper">
                    <div class="current-profile-image">
                        <?php if (!empty($current_user['profile_image']) && file_exists(__DIR__ . '/../' . $current_user['profile_image'])): ?>
                            <img src="<?php echo url($current_user['profile_image']); ?>" alt="Profile" id="preview_image">
                        <?php else: ?>
                            <span id="preview_text"><?php echo strtoupper(substr($current_user['username'], 0, 2)); ?></span>
                            <img src="" alt="Profile" id="preview_image" style="display:none;">
                        <?php endif; ?>
                    </div>
                    <label for="profile_image_input" class="image-upload-label">
                        <i class="fas fa-camera"></i> Change Profile Picture
                    </label>
                    <input type="file" id="profile_image_input" name="profile_image" accept="image/*">
                    <div class="image-preview" id="image_preview_text"></div>
                </div>
                
                <div class="form-group-modern">
                    <label for="full_name">
                        <i class="fas fa-id-card"></i> Full Name
                    </label>
                    <input 
                        type="text" 
                        class="form-control-modern" 
                        id="full_name" 
                        name="full_name" 
                        value="<?php echo htmlspecialchars($current_user['full_name'] ?? ''); ?>"
                        placeholder="Enter your full name"
                    >
                </div>
                
                <div class="form-group-modern">
                    <label for="username">
                        <i class="fas fa-user"></i> Username
                    </label>
                    <input 
                        type="text" 
                        class="form-control-modern" 
                        id="username" 
                        name="username" 
                        value="<?php echo htmlspecialchars($current_user['username']); ?>"
                        required
                        pattern="[a-zA-Z0-9_]{3,50}"
                        title="Username must be 3-50 characters and contain only letters, numbers, and underscores"
                    >
                </div>
                
                <div class="info-item">
                    <div class="info-label">Role</div>
                    <div class="info-value">
                        <i class="fas fa-shield-alt"></i> <?php echo htmlspecialchars($current_user['role']); ?>
                    </div>
                </div>
                
                <div class="info-item">
                    <div class="info-label">Account Status</div>
                    <div class="info-value">
                        <i class="fas fa-check-circle" style="color: #28a745;"></i> Active
                    </div>
                </div>
                
                <div class="info-item">
                    <div class="info-label">Member Since</div>
                    <div class="info-value">
                        <i class="fas fa-calendar"></i> <?php echo date('F j, Y', strtotime($current_user['created'])); ?>
                    </div>
                </div>
                
                <button type="submit" class="btn-update">
                    <i class="fas fa-save"></i> Update Profile
                </button>
            </form>
        </div>

        <!-- Change Password -->
        <div class="password-card">
            <h3 class="card-title">
                <i class="fas fa-lock"></i>
                Change Password
            </h3>
            
            <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST" id="passwordForm">
                <?php echo csrf_token_field(); ?>
                <input type="hidden" name="action" value="change_password">
                
                <div class="form-group-modern">
                    <label for="current_password">
                        <i class="fas fa-key"></i> Current Password
                    </label>
                    <input 
                        type="password" 
                        class="form-control-modern" 
                        id="current_password" 
                        name="current_password" 
                        required
                        placeholder="Enter your current password"
                    >
                </div>
                
                <div class="form-group-modern">
                    <label for="new_password">
                        <i class="fas fa-lock"></i> New Password
                    </label>
                    <input 
                        type="password" 
                        class="form-control-modern" 
                        id="new_password" 
                        name="new_password" 
                        required
                        placeholder="Enter new password"
                    >
                    <div class="strength-bar">
                        <div class="strength-fill" id="strengthBar"></div>
                    </div>
                    <div class="password-strength" id="strengthText"></div>
                </div>
                
                <div class="form-group-modern">
                    <label for="confirm_password">
                        <i class="fas fa-check-double"></i> Confirm New Password
                    </label>
                    <input 
                        type="password" 
                        class="form-control-modern" 
                        id="confirm_password" 
                        name="confirm_password" 
                        required
                        placeholder="Confirm new password"
                    >
                </div>
                
                <div class="password-requirements">
                    <strong>Password Requirements:</strong>
                    <div class="requirement" id="req-length">
                        <i class="fas fa-circle"></i> At least 8 characters
                    </div>
                    <div class="requirement" id="req-match">
                        <i class="fas fa-circle"></i> Passwords match
                    </div>
                </div>
                
                <button type="submit" class="btn-update">
                    <i class="fas fa-save"></i> Update Password
                </button>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const newPassword = document.getElementById('new_password');
    const confirmPassword = document.getElementById('confirm_password');
    const strengthBar = document.getElementById('strengthBar');
    const strengthText = document.getElementById('strengthText');
    const passwordForm = document.getElementById('passwordForm');
    const profileImageInput = document.getElementById('profile_image_input');
    const previewImage = document.getElementById('preview_image');
    const previewText = document.getElementById('preview_text');
    const imagePreviewText = document.getElementById('image_preview_text');
    
    // Profile image preview
    if (profileImageInput) {
        profileImageInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                // Validate file type
                const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                if (!allowedTypes.includes(file.type)) {
                    alert('Please select a valid image file (JPG, PNG, GIF, or WebP)');
                    this.value = '';
                    return;
                }
                
                // Validate file size (5MB)
                if (file.size > 5 * 1024 * 1024) {
                    alert('Image size must be less than 5MB');
                    this.value = '';
                    return;
                }
                
                // Show preview
                const reader = new FileReader();
                reader.onload = function(e) {
                    previewImage.src = e.target.result;
                    previewImage.style.display = 'block';
                    if (previewText) previewText.style.display = 'none';
                };
                reader.readAsDataURL(file);
                
                imagePreviewText.textContent = '✓ ' + file.name + ' selected';
                imagePreviewText.style.color = '#28a745';
            }
        });
    }
    
    // Password strength checker
    if (newPassword) {
        newPassword.addEventListener('input', function() {
            const password = this.value;
            let strength = 0;
            
            if (password.length >= 8) strength += 25;
            if (password.length >= 12) strength += 25;
            if (/[a-z]/.test(password) && /[A-Z]/.test(password)) strength += 25;
            if (/\d/.test(password)) strength += 15;
            if (/[^a-zA-Z\d]/.test(password)) strength += 10;
            
            strengthBar.style.width = strength + '%';
            
            if (strength < 40) {
                strengthBar.style.background = '#dc3545';
                strengthText.textContent = 'Weak password';
                strengthText.style.color = '#dc3545';
            } else if (strength < 70) {
                strengthBar.style.background = '#ffc107';
                strengthText.textContent = 'Medium password';
                strengthText.style.color = '#ffc107';
            } else {
                strengthBar.style.background = '#28a745';
                strengthText.textContent = 'Strong password';
                strengthText.style.color = '#28a745';
            }
            
            // Check length requirement
            const reqLength = document.getElementById('req-length');
            if (password.length >= 8) {
                reqLength.classList.add('valid');
                reqLength.querySelector('i').className = 'fas fa-check-circle';
            } else {
                reqLength.classList.remove('valid');
                reqLength.querySelector('i').className = 'fas fa-circle';
            }
            
            checkPasswordMatch();
        });
    }
    
    // Check password match
    if (confirmPassword) {
        confirmPassword.addEventListener('input', checkPasswordMatch);
    }
    
    function checkPasswordMatch() {
        const reqMatch = document.getElementById('req-match');
        if (confirmPassword && newPassword && confirmPassword.value && newPassword.value === confirmPassword.value) {
            reqMatch.classList.add('valid');
            reqMatch.querySelector('i').className = 'fas fa-check-circle';
        } else if (reqMatch) {
            reqMatch.classList.remove('valid');
            reqMatch.querySelector('i').className = 'fas fa-circle';
        }
    }
    
    // Form validation
    if (passwordForm) {
        passwordForm.addEventListener('submit', function(e) {
            if (newPassword.value !== confirmPassword.value) {
                e.preventDefault();
                alert('Passwords do not match!');
                return false;
            }
            
            if (newPassword.value.length < 8) {
                e.preventDefault();
                alert('Password must be at least 8 characters long!');
                return false;
            }
        });
    }
    
    // Auto-dismiss alerts
    setTimeout(function() {
        const alerts = document.querySelectorAll('.alert-modern');
        alerts.forEach(alert => {
            alert.style.animation = 'slideDown 0.3s ease reverse';
            setTimeout(() => alert.remove(), 300);
        });
    }, 5000);
});
</script>

<?php
// Include footer
if (file_exists(__DIR__ . '/includes/footer.php')) {
    require_once __DIR__ . '/includes/footer.php';
} else {
    // Minimal fallback footer to avoid fatal errors if include is missing
    echo "</main></div><!-- .admin-container --></div><!-- .admin-layout -->";
    echo "<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js\"></script>";
}
