<?php
/**
 * Admin Panel - Offers Management
 * Full CRUD operations with image upload, validation, and responsive design
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/includes/header.php';

$message = '';
$error = '';
$edit_offer = null;

// Handle AJAX requests
if (isset($_GET['ajax']) && $_GET['ajax'] === '1') {
    header('Content-Type: application/json');
    
    if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
        echo json_encode(['success' => false, 'error' => 'Invalid CSRF token']);
        exit;
    }
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'toggle_status' && isset($_POST['offer_id'])) {
        try {
            $offer_id = (int)$_POST['offer_id'];
            $stmt = $db->prepare("UPDATE offers SET status = IF(status = 'active', 'inactive', 'active') WHERE id = :id");
            $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
            $stmt->execute();
            
            $stmt = $db->prepare("SELECT status FROM offers WHERE id = :id");
            $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
            $stmt->execute();
            $new_status = $stmt->fetchColumn();
            
            echo json_encode(['success' => true, 'status' => $new_status]);
        } catch (PDOException $e) {
            error_log('Toggle status error: ' . $e->getMessage());
            echo json_encode(['success' => false, 'error' => 'Database error']);
        }
        exit;
    }
}

// Handle POST requests (Add/Edit)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_GET['ajax'])) {
    if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
        $error = 'Invalid CSRF token. Please try again.';
    } else {
        $action = $_POST['action'] ?? '';
        $offer_id = isset($_POST['offer_id']) ? (int)$_POST['offer_id'] : null;
        $title = sanitize_input($_POST['title'] ?? '');
        $description = sanitize_input($_POST['description'] ?? '');
        $discount = sanitize_input($_POST['discount'] ?? '');
        $valid_until = $_POST['valid_until'] ?? null;
        $status = sanitize_input($_POST['status'] ?? 'active');
        $image_filename = '';

        // Validation
        if (empty($title)) {
            $error = 'Title is required.';
        } elseif (strlen($title) > 255) {
            $error = 'Title must be less than 255 characters.';
        } elseif (!empty($discount) && strlen($discount) > 100) {
            $error = 'Discount text must be less than 100 characters.';
        } elseif (!empty($valid_until) && strtotime($valid_until) < strtotime('today')) {
            $error = 'Expiration date cannot be in the past.';
        } elseif (!in_array($status, ['active', 'inactive'])) {
            $error = 'Invalid status value.';
        }

        // Handle image upload with validation
        if (empty($error) && isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['image'];
            $file_size = $file['size'];
            $file_tmp = $file['tmp_name'];
            $file_name = $file['name'];
            $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            
            // Validate file type
            $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
            if (!in_array($file_ext, $allowed_extensions)) {
                $error = 'Invalid file type. Allowed: JPG, JPEG, PNG, GIF, WEBP';
            }
            
            // Validate file size (5MB max)
            elseif ($file_size > 5242880) {
                $error = 'Image file size must be less than 5MB.';
            }
            
            // Validate image dimensions and file integrity
            else {
                $image_info = @getimagesize($file_tmp);
                if ($image_info === false) {
                    $error = 'Invalid image file. Please upload a valid image.';
                } else {
                    // Create upload directory if it doesn't exist
                    $upload_dir = ASSETS_PATH . DS . 'images' . DS . 'offers' . DS;
                    if (!is_dir($upload_dir)) {
                        if (!@mkdir($upload_dir, 0755, true)) {
                            $error = 'Failed to create upload directory. Please check permissions.';
                        }
                    }
                    
                    if (empty($error)) {
                        // Verify directory is writable
                        if (!is_writable($upload_dir)) {
                            $error = 'Upload directory is not writable. Please run: sudo chmod 775 ' . $upload_dir;
                        } else {
                            // Generate secure filename
                            $safe_filename = time() . '_' . bin2hex(random_bytes(8)) . '.' . $file_ext;
                            $dest_path = $upload_dir . $safe_filename;
                            
                            if (move_uploaded_file($file_tmp, $dest_path)) {
                                $image_filename = 'assets/images/offers/' . $safe_filename;
                                
                                // Optimize image size
                                if (function_exists('imagecreatefromjpeg')) {
                                    try {
                                        switch ($file_ext) {
                                            case 'jpg':
                                            case 'jpeg':
                                                $img = imagecreatefromjpeg($dest_path);
                                                if ($img) {
                                                    imagejpeg($img, $dest_path, 85);
                                                    imagedestroy($img);
                                                }
                                                break;
                                            case 'png':
                                                $img = imagecreatefrompng($dest_path);
                                                if ($img) {
                                                    imagepng($img, $dest_path, 8);
                                                    imagedestroy($img);
                                                }
                                                break;
                                        }
                                    } catch (Exception $e) {
                                        error_log('Image optimization error: ' . $e->getMessage());
                                    }
                                }
                            } else {
                                $error = 'Failed to upload image. Error: ' . (error_get_last()['message'] ?? 'Unknown error');
                                error_log('Image upload failed: ' . print_r(error_get_last(), true));
                            }
                        }
                    }
                }
            }
        }

        // Process database operations
        if (empty($error) && $db) {
            try {
                if ($action === 'add') {
                    if (empty($image_filename)) {
                        $error = 'Image is required for new offers.';
                    } else {
                        $stmt = $db->prepare("
                            INSERT INTO offers (title, description, discount, valid_until, image, status) 
                            VALUES (:title, :description, :discount, :valid_until, :image, :status)
                        ");
                        $stmt->bindParam(':title', $title);
                        $stmt->bindParam(':description', $description);
                        $stmt->bindParam(':discount', $discount);
                        $stmt->bindParam(':valid_until', $valid_until);
                        $stmt->bindParam(':image', $image_filename);
                        $stmt->bindParam(':status', $status);
                        $stmt->execute();
                        
                        $_SESSION['success_message'] = 'Offer added successfully!';
                        header('Location: ' . url('/admin/offers.php'));
                        exit;
                    }
                } elseif ($action === 'edit' && $offer_id) {
                    // Get existing offer data
                    $stmt = $db->prepare("SELECT image FROM offers WHERE id = :id");
                    $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
                    $stmt->execute();
                    $existing_offer = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($existing_offer) {
                        $update_sql = "UPDATE offers SET 
                            title = :title, 
                            description = :description, 
                            discount = :discount, 
                            valid_until = :valid_until, 
                            status = :status";
                        
                        if (!empty($image_filename)) {
                            $update_sql .= ", image = :image";
                            // Delete old image
                            if (!empty($existing_offer['image'])) {
                                $old_image_path = BASE_PATH . DS . str_replace('/', DS, $existing_offer['image']);
                                if (file_exists($old_image_path)) {
                                    @unlink($old_image_path);
                                }
                            }
                        }
                        
                        $update_sql .= " WHERE id = :id";
                        
                        $stmt = $db->prepare($update_sql);
                        $stmt->bindParam(':title', $title);
                        $stmt->bindParam(':description', $description);
                        $stmt->bindParam(':discount', $discount);
                        $stmt->bindParam(':valid_until', $valid_until);
                        $stmt->bindParam(':status', $status);
                        if (!empty($image_filename)) {
                            $stmt->bindParam(':image', $image_filename);
                        }
                        $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
                        $stmt->execute();
                        
                        $_SESSION['success_message'] = 'Offer updated successfully!';
                        header('Location: ' . url('/admin/offers.php'));
                        exit;
                    } else {
                        $error = 'Offer not found.';
                    }
                }
            } catch (PDOException $e) {
                error_log('Database error: ' . $e->getMessage());
                $error = 'Database error. Please try again later.';
            }
        }
    }
}

// Handle DELETE requests
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $offer_id = (int)$_GET['id'];
    
    if ($db) {
        try {
            // Get image path before deleting
            $stmt = $db->prepare("SELECT image FROM offers WHERE id = :id");
            $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
            $stmt->execute();
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($offer) {
                // Delete from database
                $stmt = $db->prepare("DELETE FROM offers WHERE id = :id");
                $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
                $stmt->execute();
                
                // Delete image file
                if (!empty($offer['image'])) {
                    $image_path = BASE_PATH . DS . str_replace('/', DS, $offer['image']);
                    if (file_exists($image_path)) {
                        @unlink($image_path);
                    }
                }
                
                $_SESSION['success_message'] = 'Offer deleted successfully!';
            } else {
                $_SESSION['error_message'] = 'Offer not found.';
            }
        } catch (PDOException $e) {
            error_log('Delete error: ' . $e->getMessage());
            $_SESSION['error_message'] = 'Error deleting offer.';
        }
    }
    
    header('Location: ' . url('/admin/offers.php'));
    exit;
}

// Load offer data for editing
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $offer_id = (int)$_GET['id'];
    try {
        $stmt = $db->prepare("SELECT * FROM offers WHERE id = :id");
        $stmt->bindParam(':id', $offer_id, PDO::PARAM_INT);
        $stmt->execute();
        $edit_offer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$edit_offer) {
            $_SESSION['error_message'] = 'Offer not found.';
            header('Location: ' . url('/admin/offers.php'));
            exit;
        }
    } catch (PDOException $e) {
        error_log('Fetch offer error: ' . $e->getMessage());
        $_SESSION['error_message'] = 'Error loading offer.';
        header('Location: ' . url('/admin/offers.php'));
        exit;
    }
}

// Fetch all offers with search
$search = isset($_GET['search']) ? sanitize_input($_GET['search']) : '';
$status_filter = isset($_GET['status']) ? sanitize_input($_GET['status']) : '';

$offers = [];
$total_offers = 0;
$active_offers = 0;
$inactive_offers = 0;
$expired_offers = 0;

if ($db) {
    try {
        // Build query
        $query = "SELECT * FROM offers WHERE 1=1";
        $params = [];
        
        if (!empty($search)) {
            $query .= " AND (title LIKE :search OR description LIKE :search OR discount LIKE :search)";
            $params[':search'] = '%' . $search . '%';
        }
        
        if (!empty($status_filter) && in_array($status_filter, ['active', 'inactive'])) {
            $query .= " AND status = :status";
            $params[':status'] = $status_filter;
        }
        
        $query .= " ORDER BY created DESC";
        
        $stmt = $db->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get statistics
        $stmt = $db->query("SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive,
            SUM(CASE WHEN valid_until IS NOT NULL AND valid_until < CURDATE() THEN 1 ELSE 0 END) as expired
            FROM offers");
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $total_offers = $stats['total'];
        $active_offers = $stats['active'];
        $inactive_offers = $stats['inactive'];
        $expired_offers = $stats['expired'];
        
    } catch (PDOException $e) {
        error_log('Fetch offers error: ' . $e->getMessage());
        $error = 'Error loading offers.';
    }
}

// Display session messages
if (isset($_SESSION['success_message'])) {
    $message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

$page_action = $_GET['action'] ?? 'list';
?>

<style>
/* Offers Admin Panel Styles */
.offers-admin-container {
    padding: 2rem;
    max-width: 1400px;
    margin: 0 auto;
}

.page-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0;
    font-size: 2rem;
    font-weight: 600;
}

.btn-primary-custom {
    background: linear-gradient(135deg, #0000ff 0%, #0000ff 100%);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    border: none;
    cursor: pointer;
    font-weight: 500;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary-custom:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
    color: white;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 4px 16px rgba(0,0,0,0.12);
}

.stat-card .stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    margin-bottom: 1rem;
}

.stat-card.total .stat-icon { background: #e3f2fd; color: #1976d2; }
.stat-card.active .stat-icon { background: #e8f5e9; color: #388e3c; }
.stat-card.inactive .stat-icon { background: #fff3e0; color: #f57c00; }
.stat-card.expired .stat-icon { background: #ffebee; color: #d32f2f; }

.stat-card h3 {
    font-size: 2rem;
    font-weight: 700;
    margin: 0;
    color: #2c3e50;
}

.stat-card p {
    margin: 0.25rem 0 0;
    color: #7f8c8d;
    font-size: 0.9rem;
}

.alert {
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
    animation: slideDown 0.3s ease;
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.alert-success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.alert-danger {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.search-filter-bar {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    margin-bottom: 2rem;
}

.search-filter-grid {
    display: grid;
    grid-template-columns: 1fr auto auto;
    gap: 1rem;
    align-items: end;
}

.form-group {
    margin-bottom: 0;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 500;
    color: #2c3e50;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 1px solid #dfe6e9;
    border-radius: 8px;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.form-control:focus {
    outline: none;
    border-color: #0000ff;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.btn-search, .btn-reset {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    font-weight: 500;
    transition: all 0.3s ease;
}

.btn-search {
    background: #0000ff;
    color: white;
}

.btn-search:hover {
    background: #0000ff;
}

.btn-reset {
    background: #e9ecef;
    color: #495057;
}

.btn-reset:hover {
    background: #dee2e6;
}

.offers-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 1.5rem;
    margin-top: 2rem;
}

.offer-card {
    background: white;
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    transition: all 0.3s ease;
    animation: fadeInUp 0.5s ease;
    animation-fill-mode: both;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.offer-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 24px rgba(0,0,0,0.15);
}

.offer-image-container {
    position: relative;
    padding-top: 56.25%; /* 16:9 aspect ratio */
    overflow: hidden;
    background: #f8f9fa;
}

.offer-image-container img {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.offer-card:hover .offer-image-container img {
    transform: scale(1.05);
}

.offer-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-weight: 700;
    font-size: 1rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
}

.offer-status-badge {
    position: absolute;
    top: 1rem;
    left: 1rem;
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 600;
    text-transform: uppercase;
}

.offer-status-badge.active {
    background: #d4edda;
    color: #155724;
}

.offer-status-badge.inactive {
    background: #f8d7da;
    color: #721c24;
}

.offer-body {
    padding: 1.5rem;
}

.offer-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #2c3e50;
    margin: 0 0 0.75rem;
}

.offer-description {
    color: #7f8c8d;
    margin: 0 0 1rem;
    line-height: 1.6;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.offer-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem 0;
    border-top: 1px solid #ecf0f1;
    margin-top: 1rem;
}

.offer-date {
    font-size: 0.85rem;
    color: #95a5a6;
}

.offer-date i {
    margin-right: 0.5rem;
}

.offer-actions {
    display: flex;
    gap: 0.5rem;
    margin-top: 1rem;
}

.btn-action {
    flex: 1;
    padding: 0.6rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    text-align: center;
    font-size: 0.9rem;
    font-weight: 500;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.btn-edit {
    background: #3498db;
    color: white;
}

.btn-edit:hover {
    background: #2980b9;
    color: white;
}

.btn-delete {
    background: #e74c3c;
    color: white;
}

.btn-delete:hover {
    background: #c0392b;
    color: white;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
}

.empty-state i {
    font-size: 4rem;
    color: #bdc3c7;
    margin-bottom: 1.5rem;
}

.empty-state h3 {
    color: #7f8c8d;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #95a5a6;
    margin-bottom: 2rem;
}

/* Form Styles */
.form-container {
    background: white;
    padding: 2rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    max-width: 800px;
    margin: 0 auto;
}

.form-container .form-group {
    margin-bottom: 1.5rem;
}

.form-container textarea.form-control {
    min-height: 120px;
    resize: vertical;
}

.image-preview {
    margin-top: 1rem;
    padding: 1rem;
    border: 2px dashed #dfe6e9;
    border-radius: 8px;
    text-align: center;
}

.image-preview img {
    max-width: 100%;
    max-height: 300px;
    border-radius: 8px;
}

.current-image {
    margin-top: 0.5rem;
}

.current-image img {
    max-width: 200px;
    border-radius: 8px;
    border: 2px solid #dfe6e9;
}

.form-actions {
    display: flex;
    gap: 1rem;
    margin-top: 2rem;
}

.btn-submit {
    flex: 1;
    padding: 0.875rem 2rem;
    background: linear-gradient(135deg, #0000ff 0%, #0000ff 100%);
    color: white;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-submit:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
}

.btn-cancel {
    padding: 0.875rem 2rem;
    background: #e9ecef;
    color: #495057;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    text-decoration: none;
    transition: all 0.3s ease;
}

.btn-cancel:hover {
    background: #dee2e6;
    color: #495057;
}

/* Responsive Design */
@media (max-width: 768px) {
    .offers-admin-container {
        padding: 1rem;
    }
    
    .page-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .search-filter-grid {
        grid-template-columns: 1fr;
    }
    
    .offers-grid {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}

@media (max-width: 480px) {
    .page-header h1 {
        font-size: 1.5rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
}

/* Loading Animation */
.loading {
    text-align: center;
    padding: 3rem;
    color: #7f8c8d;
}

.loading i {
    font-size: 3rem;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}
</style>

<div class="offers-admin-container">
    <!-- Page Header -->
    <div class="page-header">
        <h1>
            <i class="fas fa-tags"></i> 
            <?php echo $page_action === 'add' ? 'Add New Offer' : ($page_action === 'edit' ? 'Edit Offer' : 'Offers Management'); ?>
        </h1>
        <?php if ($page_action === 'list'): ?>
            <a href="<?php echo url('/admin/offers.php?action=add'); ?>" class="btn-primary-custom">
                <i class="fas fa-plus"></i> Add New Offer
            </a>
        <?php else: ?>
            <a href="<?php echo url('/admin/offers.php'); ?>" class="btn-primary-custom">
                <i class="fas fa-arrow-left"></i> Back to List
            </a>
        <?php endif; ?>
    </div>

    <!-- Alerts -->
    <?php if (!empty($message)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <?php if (!empty($error)): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <?php if ($page_action === 'list'): ?>
        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card total">
                <div class="stat-icon"><i class="fas fa-tags"></i></div>
                <h3><?php echo $total_offers; ?></h3>
                <p>Total Offers</p>
            </div>
            <div class="stat-card active">
                <div class="stat-icon"><i class="fas fa-check-circle"></i></div>
                <h3><?php echo $active_offers; ?></h3>
                <p>Active Offers</p>
            </div>
            <div class="stat-card inactive">
                <div class="stat-icon"><i class="fas fa-pause-circle"></i></div>
                <h3><?php echo $inactive_offers; ?></h3>
                <p>Inactive Offers</p>
            </div>
            <div class="stat-card expired">
                <div class="stat-icon"><i class="fas fa-calendar-times"></i></div>
                <h3><?php echo $expired_offers; ?></h3>
                <p>Expired Offers</p>
            </div>
        </div>

        <!-- Search & Filter -->
        <div class="search-filter-bar">
            <form method="GET" action="<?php echo url('/admin/offers.php'); ?>">
                <div class="search-filter-grid">
                    <div class="form-group">
                        <label for="search"><i class="fas fa-search"></i> Search Offers</label>
                        <input 
                            type="text" 
                            id="search" 
                            name="search" 
                            class="form-control" 
                            placeholder="Search by title, description, or discount..."
                            value="<?php echo htmlspecialchars($search); ?>"
                        >
                    </div>
                    <div class="form-group">
                        <label for="status"><i class="fas fa-filter"></i> Status</label>
                        <select id="status" name="status" class="form-control">
                            <option value="">All Status</option>
                            <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                    <button type="submit" class="btn-search">
                        <i class="fas fa-search"></i> Search
                    </button>
                    <?php if (!empty($search) || !empty($status_filter)): ?>
                        <a href="<?php echo url('/admin/offers.php'); ?>" class="btn-reset">
                            <i class="fas fa-redo"></i> Reset
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Offers Grid -->
        <?php if (!empty($offers)): ?>
            <div class="offers-grid">
                <?php foreach ($offers as $index => $offer): 
                    $is_expired = !empty($offer['valid_until']) && strtotime($offer['valid_until']) < time();
                    $status_text = $is_expired ? 'Expired' : ucfirst($offer['status']);
                    $status_class = $is_expired ? 'expired' : $offer['status'];
                ?>
                    <div class="offer-card" style="animation-delay: <?php echo $index * 0.1; ?>s">
                        <div class="offer-image-container">
                            <?php if (!empty($offer['image'])): ?>
                                <img src="<?php echo url($offer['image']); ?>" alt="<?php echo htmlspecialchars($offer['title']); ?>">
                            <?php else: ?>
                                <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 4rem; color: #bdc3c7;">
                                    <i class="fas fa-image"></i>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($offer['discount'])): ?>
                                <div class="offer-badge"><?php echo htmlspecialchars($offer['discount']); ?></div>
                            <?php endif; ?>
                            
                            <div class="offer-status-badge <?php echo $status_class; ?>">
                                <?php echo $status_text; ?>
                            </div>
                        </div>
                        
                        <div class="offer-body">
                            <h3 class="offer-title"><?php echo htmlspecialchars($offer['title']); ?></h3>
                            
                            <?php if (!empty($offer['description'])): ?>
                                <p class="offer-description"><?php echo htmlspecialchars($offer['description']); ?></p>
                            <?php endif; ?>
                            
                            <div class="offer-meta">
                                <div class="offer-date">
                                    <i class="fas fa-calendar-alt"></i>
                                    <?php if (!empty($offer['valid_until'])): ?>
                                        Valid until: <?php echo date('M j, Y', strtotime($offer['valid_until'])); ?>
                                    <?php else: ?>
                                        No expiration
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="offer-actions">
                                <a href="<?php echo url('/admin/offers.php?action=edit&id=' . $offer['id']); ?>" class="btn-action btn-edit">
                                    <i class="fas fa-edit"></i> Edit
                                </a>
                                <a 
                                    href="<?php echo url('/admin/offers.php?action=delete&id=' . $offer['id']); ?>" 
                                    class="btn-action btn-delete"
                                    onclick="return confirm('Are you sure you want to delete this offer? This action cannot be undone.');"
                                >
                                    <i class="fas fa-trash-alt"></i> Delete
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-tags"></i>
                <h3>No offers found</h3>
                <p><?php echo !empty($search) || !empty($status_filter) ? 'Try adjusting your search filters.' : 'Create your first special offer to attract customers!'; ?></p>
                <a href="<?php echo url('/admin/offers.php?action=add'); ?>" class="btn-primary-custom">
                    <i class="fas fa-plus"></i> Create First Offer
                </a>
            </div>
        <?php endif; ?>

    <?php else: ?>
        <!-- Add/Edit Form -->
        <div class="form-container">
            <form method="POST" action="<?php echo url('/admin/offers.php'); ?>" enctype="multipart/form-data" id="offerForm">
                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                <input type="hidden" name="action" value="<?php echo $page_action; ?>">
                <?php if ($edit_offer): ?>
                    <input type="hidden" name="offer_id" value="<?php echo $edit_offer['id']; ?>">
                <?php endif; ?>

                <div class="form-group">
                    <label for="title">Offer Title <span style="color: red;">*</span></label>
                    <input 
                        type="text" 
                        id="title" 
                        name="title" 
                        class="form-control" 
                        placeholder="e.g., Black Friday Sale - 50% OFF"
                        value="<?php echo $edit_offer ? htmlspecialchars($edit_offer['title']) : ''; ?>"
                        required
                        maxlength="255"
                    >
                </div>

                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea 
                        id="description" 
                        name="description" 
                        class="form-control" 
                        placeholder="Describe your offer in detail..."
                    ><?php echo $edit_offer ? htmlspecialchars($edit_offer['description']) : ''; ?></textarea>
                </div>

                <div class="form-group">
                    <label for="discount">Discount Badge Text</label>
                    <input 
                        type="text" 
                        id="discount" 
                        name="discount" 
                        class="form-control" 
                        placeholder="e.g., 50% OFF, Buy 1 Get 1, Save $100"
                        value="<?php echo $edit_offer ? htmlspecialchars($edit_offer['discount']) : ''; ?>"
                        maxlength="100"
                    >
                    <small style="color: #7f8c8d;">This will appear as a badge on the offer image</small>
                </div>

                <div class="form-group">
                    <label for="valid_until">Expiration Date</label>
                    <input 
                        type="date" 
                        id="valid_until" 
                        name="valid_until" 
                        class="form-control" 
                        value="<?php echo $edit_offer ? $edit_offer['valid_until'] : ''; ?>"
                        min="<?php echo date('Y-m-d'); ?>"
                    >
                    <small style="color: #7f8c8d;">Leave empty for no expiration</small>
                </div>

                <div class="form-group">
                    <label for="image">Offer Image <?php echo !$edit_offer ? '<span style="color: red;">*</span>' : ''; ?></label>
                    <input 
                        type="file" 
                        id="image" 
                        name="image" 
                        class="form-control" 
                        accept="image/jpeg,image/png,image/gif,image/webp"
                        <?php echo !$edit_offer ? 'required' : ''; ?>
                    >
                    <small style="color: #7f8c8d;">Allowed formats: JPG, PNG, GIF, WEBP (Max 5MB, recommended 1200x630px)</small>
                    
                    <?php if ($edit_offer && !empty($edit_offer['image'])): ?>
                        <div class="current-image">
                            <p style="margin-top: 1rem; margin-bottom: 0.5rem; font-weight: 500;">Current Image:</p>
                            <img src="<?php echo url($edit_offer['image']); ?>" alt="Current offer image">
                            <p style="margin-top: 0.5rem; color: #7f8c8d; font-size: 0.9rem;">Upload a new image to replace the current one</p>
                        </div>
                    <?php endif; ?>
                    
                    <div id="imagePreview"></div>
                </div>

                <div class="form-group">
                    <label for="status">Status</label>
                    <select id="status" name="status" class="form-control">
                        <option value="active" <?php echo ($edit_offer && $edit_offer['status'] === 'active') || !$edit_offer ? 'selected' : ''; ?>>Active</option>
                        <option value="inactive" <?php echo $edit_offer && $edit_offer['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                    </select>
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn-submit">
                        <i class="fas fa-save"></i> <?php echo $page_action === 'edit' ? 'Update Offer' : 'Create Offer'; ?>
                    </button>
                    <a href="<?php echo url('/admin/offers.php'); ?>" class="btn-cancel">
                        <i class="fas fa-times"></i> Cancel
                    </a>
                </div>
            </form>
        </div>
    <?php endif; ?>
</div>

<script>
// Image preview
document.getElementById('image')?.addEventListener('change', function(e) {
    const file = e.target.files[0];
    const previewContainer = document.getElementById('imagePreview');
    
    if (file) {
        // Validate file size
        if (file.size > 5242880) {
            alert('File size must be less than 5MB');
            e.target.value = '';
            previewContainer.innerHTML = '';
            return;
        }
        
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            alert('Only JPG, PNG, GIF, and WEBP images are allowed');
            e.target.value = '';
            previewContainer.innerHTML = '';
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(event) {
            previewContainer.innerHTML = `
                <div class="image-preview">
                    <p style="margin-bottom: 0.5rem; font-weight: 500;">Preview:</p>
                    <img src="${event.target.result}" alt="Preview">
                </div>
            `;
        };
        reader.readAsDataURL(file);
    } else {
        previewContainer.innerHTML = '';
    }
});

// Auto-dismiss alerts
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.animation = 'slideUp 0.3s ease';
            setTimeout(() => alert.remove(), 300);
        }, 5000);
    });
});

// Form validation
document.getElementById('offerForm')?.addEventListener('submit', function(e) {
    const title = document.getElementById('title').value.trim();
    const image = document.getElementById('image');
    const isEditMode = <?php echo $edit_offer ? 'true' : 'false'; ?>;
    
    if (!title) {
        e.preventDefault();
        alert('Please enter an offer title');
        document.getElementById('title').focus();
        return false;
    }
    
    if (!isEditMode && !image.files.length) {
        e.preventDefault();
        alert('Please select an image for the offer');
        image.focus();
        return false;
    }
    
    // Show loading state
    const submitBtn = this.querySelector('.btn-submit');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
});

@keyframes slideUp {
    from { opacity: 1; transform: translateY(0); }
    to { opacity: 0; transform: translateY(-10px); }
}
</script>

<?php
require_once __DIR__ . '/includes/footer.php';
?>
