<?php
/**
 * Domain Management System
 * Manage domain TLDs and pricing
 */

// Include config first to get get_db() function
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/db.php';

$page_title = 'Domain Management';

// Get database connection
$db = get_db();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_domain'])) {
        // Add new domain - with validation
        $tld = trim($_POST['tld']);
        
        // Validate TLD format
        if (empty($tld)) {
            $_SESSION['error_message'] = 'TLD is required';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        // Ensure TLD starts with dot
        if (!str_starts_with($tld, '.')) {
            $tld = '.' . $tld;
        }
        
        $register_price = floatval($_POST['register_price']);
        $renew_price = floatval($_POST['renew_price']);
        $transfer_price = !empty($_POST['transfer_price']) ? floatval($_POST['transfer_price']) : null;
        $badge = !empty(trim($_POST['badge'])) ? strtoupper(trim($_POST['badge'])) : null;
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order']);
        
        // Validate prices
        if ($register_price <= 0 || $renew_price <= 0) {
            $_SESSION['error_message'] = 'Prices must be greater than 0';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        try {
            $stmt = $db->prepare("INSERT INTO domains (tld, register_price, renew_price, transfer_price, badge, is_active, display_order) 
                                  VALUES (:tld, :register_price, :renew_price, :transfer_price, :badge, :is_active, :display_order)");
            $stmt->execute([
                'tld' => $tld,
                'register_price' => $register_price,
                'renew_price' => $renew_price,
                'transfer_price' => $transfer_price,
                'badge' => $badge,
                'is_active' => $is_active,
                'display_order' => $display_order
            ]);
            $_SESSION['success_message'] = 'Domain ' . htmlspecialchars($tld) . ' added successfully!';
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $_SESSION['error_message'] = 'Domain ' . htmlspecialchars($tld) . ' already exists!';
            } else {
                $_SESSION['error_message'] = 'Error adding domain: ' . $e->getMessage();
            }
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if (isset($_POST['update_domain'])) {
        // Update existing domain - with validation
        $id = intval($_POST['domain_id']);
        
        if ($id <= 0) {
            $_SESSION['error_message'] = 'Invalid domain ID';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        $tld = trim($_POST['tld']);
        
        // Validate TLD format
        if (empty($tld)) {
            $_SESSION['error_message'] = 'TLD is required';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        // Ensure TLD starts with dot
        if (!str_starts_with($tld, '.')) {
            $tld = '.' . $tld;
        }
        
        $register_price = floatval($_POST['register_price']);
        $renew_price = floatval($_POST['renew_price']);
        $transfer_price = !empty($_POST['transfer_price']) ? floatval($_POST['transfer_price']) : null;
        $badge = !empty(trim($_POST['badge'])) ? strtoupper(trim($_POST['badge'])) : null;
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order']);
        
        // Validate prices
        if ($register_price <= 0 || $renew_price <= 0) {
            $_SESSION['error_message'] = 'Prices must be greater than 0';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        try {
            $stmt = $db->prepare("UPDATE domains SET tld = :tld, register_price = :register_price, renew_price = :renew_price, 
                                  transfer_price = :transfer_price, badge = :badge, is_active = :is_active, display_order = :display_order 
                                  WHERE id = :id");
            $result = $stmt->execute([
                'id' => $id,
                'tld' => $tld,
                'register_price' => $register_price,
                'renew_price' => $renew_price,
                'transfer_price' => $transfer_price,
                'badge' => $badge,
                'is_active' => $is_active,
                'display_order' => $display_order
            ]);
            
            if ($stmt->rowCount() > 0) {
                $_SESSION['success_message'] = 'Domain ' . htmlspecialchars($tld) . ' updated successfully!';
            } else {
                $_SESSION['error_message'] = 'No changes made to domain';
            }
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $_SESSION['error_message'] = 'Domain ' . htmlspecialchars($tld) . ' already exists!';
            } else {
                $_SESSION['error_message'] = 'Error updating domain: ' . $e->getMessage();
            }
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if (isset($_POST['delete_domain'])) {
        // Delete domain - with validation
        $id = intval($_POST['domain_id']);
        
        if ($id <= 0) {
            $_SESSION['error_message'] = 'Invalid domain ID';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        try {
            // First get the domain name for success message
            $stmt = $db->prepare("SELECT tld FROM domains WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $domain = $stmt->fetch();
            
            if ($domain) {
                $stmt = $db->prepare("DELETE FROM domains WHERE id = :id");
                $stmt->execute(['id' => $id]);
                
                if ($stmt->rowCount() > 0) {
                    $_SESSION['success_message'] = 'Domain ' . htmlspecialchars($domain['tld']) . ' deleted successfully!';
                } else {
                    $_SESSION['error_message'] = 'Domain not found';
                }
            } else {
                $_SESSION['error_message'] = 'Domain not found';
            }
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error deleting domain: ' . $e->getMessage();
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if (isset($_POST['bulk_import'])) {
        // Bulk import from current data
        $default_domains = [
            ['.com', 1099, 1165, 1099, null],
            ['.org', 1238, 1524, 1238, 'SALE'],
            ['.co.uk', 737, 913, null, null],
            ['.shop', 3419, 3970, 3519, null],
            ['.dev', 1407, 1638, 1407, 'NEW'],
            ['.ca', 1274, 1561, 1275, null],
            ['.online', 2811, 3090, 2811, null],
            ['.net', 1227, 1535, 1227, null],
            ['.in', 644, 693, 644, null],
            ['.co.in', 534, 572, 534, null],
            ['.info', 2289, 2771, 2287, null],
            ['.us', 754, 941, 754, null],
            ['.co', 3024, 3222, 3024, null],
            ['.org.in', 534, 572, 533, null],
            ['.xyz', 1297, 1451, 1297, null],
            ['.net.in', 534, 572, 533, null],
            ['.site', 2903, 3090, 2903, null],
            ['.ai', 8073, 8568, null, 'HOT'],
            ['.store', 4389, 4927, 4390, null],
            ['.fun', 2903, 3090, 2903, null],
            ['.live', 2672, 2683, 2672, null],
            ['.education', 2793, 2970, 2793, null],
            ['.pro', 2408, 2903, 2408, null],
            ['.app', 1638, 1891, 1638, null],
            ['.io', 5268, 5598, 5268, null],
            ['.in.net', 714, 831, 714, null],
            ['.website', 2331, 2474, 2331, null],
            ['.tech', 5268, 5598, 5268, null],
            ['.blog', 2200, 2662, 2200, null],
            ['.ai.in', 534, 572, 533, null],
            ['.biz', 1869, 2276, 1869, null],
            ['.eu', 704, 880, 704, null],
            ['.financial', 7717, 9267, 7717, null],
            ['.host', 8502, 10118, 8502, null],
            ['.icu', 1418, 1506, 1418, null],
            ['.id', 2023, 2300, 2023, null],
            ['.info.in', 534, 572, 533, null],
            ['.io.in', 534, 572, 533, null],
        ];
        
        $imported = 0;
        $skipped = 0;
        try {
            foreach ($default_domains as $index => $domain) {
                $stmt = $db->prepare("INSERT IGNORE INTO domains (tld, register_price, renew_price, transfer_price, badge, display_order) 
                                      VALUES (:tld, :register_price, :renew_price, :transfer_price, :badge, :display_order)");
                $stmt->execute([
                    'tld' => $domain[0],
                    'register_price' => $domain[1],
                    'renew_price' => $domain[2],
                    'transfer_price' => $domain[3],
                    'badge' => $domain[4],
                    'display_order' => $index
                ]);
                if ($stmt->rowCount() > 0) {
                    $imported++;
                } else {
                    $skipped++;
                }
            }
            $_SESSION['success_message'] = "Imported {$imported} domain(s) successfully!" . ($skipped > 0 ? " ({$skipped} already existed)" : "");
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error importing domains: ' . $e->getMessage();
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if (isset($_POST['toggle_status'])) {
        // Toggle domain active/inactive status
        $id = intval($_POST['domain_id']);
        
        if ($id <= 0) {
            $_SESSION['error_message'] = 'Invalid domain ID';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        try {
            $stmt = $db->prepare("UPDATE domains SET is_active = NOT is_active WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            if ($stmt->rowCount() > 0) {
                $_SESSION['success_message'] = 'Domain status updated successfully!';
            } else {
                $_SESSION['error_message'] = 'Domain not found';
            }
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error updating status: ' . $e->getMessage();
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Get all domains with error handling
try {
    $domains = $db->query("SELECT * FROM domains ORDER BY display_order ASC, tld ASC")->fetchAll();
} catch (Exception $e) {
    error_log("Error fetching domains: " . $e->getMessage());
    $domains = [];
    $_SESSION['error_message'] = 'Error loading domains. Please refresh the page.';
}

// Get domain count
$total_domains = count($domains);
$active_domains = count(array_filter($domains, function($d) { return $d['is_active']; }));

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <h2><i class="fas fa-globe"></i> Domain Management</h2>
                <div>
                    <?php if ($total_domains == 0): ?>
                        <form method="POST" style="display: inline;">
                            <button type="submit" name="bulk_import" class="btn btn-success" onclick="return confirm('Import default domain prices?')">
                                <i class="fas fa-download"></i> Import Default Domains
                            </button>
                        </form>
                    <?php endif; ?>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addDomainModal">
                        <i class="fas fa-plus"></i> Add Domain
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h5><i class="fas fa-list"></i> Total Domains</h5>
                    <h2><?= $total_domains ?></h2>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h5><i class="fas fa-check-circle"></i> Active Domains</h5>
                    <h2><?= $active_domains ?></h2>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <h5><i class="fas fa-eye-slash"></i> Inactive Domains</h5>
                    <h2><?= $total_domains - $active_domains ?></h2>
                </div>
            </div>
        </div>
    </div>

    <!-- Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <?= htmlspecialchars($_SESSION['success_message']) ?>
            <button type="button" class="close" data-dismiss="alert"><span>&times;</span></button>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <?= htmlspecialchars($_SESSION['error_message']) ?>
            <button type="button" class="close" data-dismiss="alert"><span>&times;</span></button>
        </div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>

    <!-- Domains Table -->
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">Domain Price List</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Order</th>
                            <th>TLD</th>
                            <th>Register Price</th>
                            <th>Renew Price</th>
                            <th>Transfer Price</th>
                            <th>Badge</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($domains)): ?>
                            <tr>
                                <td colspan="8" class="text-center py-5">
                                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                    <p class="text-muted">No domains found. Click "Import Default Domains" to get started.</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($domains as $domain): ?>
                                <tr>
                                    <td><?= $domain['display_order'] ?></td>
                                    <td><strong><?= htmlspecialchars($domain['tld']) ?></strong></td>
                                    <td>₹<?= number_format($domain['register_price'], 0) ?></td>
                                    <td>₹<?= number_format($domain['renew_price'], 0) ?></td>
                                    <td><?= $domain['transfer_price'] ? '₹' . number_format($domain['transfer_price'], 0) : '-' ?></td>
                                    <td>
                                        <?php if ($domain['badge']): ?>
                                            <span class="badge bg-warning"><?= htmlspecialchars($domain['badge']) ?></span>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="domain_id" value="<?= $domain['id'] ?>">
                                            <?php if ($domain['is_active']): ?>
                                                <button type="submit" name="toggle_status" class="badge bg-success border-0" 
                                                        style="cursor: pointer;" title="Click to deactivate">
                                                    <i class="fas fa-check-circle"></i> Active
                                                </button>
                                            <?php else: ?>
                                                <button type="submit" name="toggle_status" class="badge bg-secondary border-0" 
                                                        style="cursor: pointer;" title="Click to activate">
                                                    <i class="fas fa-times-circle"></i> Inactive
                                                </button>
                                            <?php endif; ?>
                                        </form>
                                    </td>
                                    <td class="text-nowrap">
                                        <button class="btn btn-sm btn-warning" onclick="editDomain(<?= htmlspecialchars(json_encode($domain)) ?>)" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="domain_id" value="<?= $domain['id'] ?>">
                                            <button type="submit" name="delete_domain" class="btn btn-sm btn-danger" 
                                                    onclick="return confirm('Are you sure you want to delete <?= htmlspecialchars($domain['tld']) ?>?')" title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Add Domain Modal -->
<div class="modal fade" id="addDomainModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Domain</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">TLD (e.g., .com)</label>
                        <input type="text" name="tld" class="form-control" required placeholder=".com">
                        <div class="form-text">Enter domain extension (e.g., .com, .net, .org)</div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Register Price (₹)</label>
                                <input type="number" name="register_price" class="form-control" step="0.01" required min="1" placeholder="1099">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Renew Price (₹)</label>
                                <input type="number" name="renew_price" class="form-control" step="0.01" required min="1" placeholder="1165">
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Transfer Price (₹) <small class="text-muted">(Optional)</small></label>
                        <input type="number" name="transfer_price" class="form-control" step="0.01" min="0" placeholder="Leave empty if not applicable">
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Badge <small class="text-muted">(Optional)</small></label>
                                <select name="badge" class="form-select">
                                    <option value="">None</option>
                                    <option value="SALE">SALE</option>
                                    <option value="NEW">NEW</option>
                                    <option value="HOT">HOT</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Display Order</label>
                                <input type="number" name="display_order" class="form-control" value="0" min="0">
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input type="checkbox" name="is_active" class="form-check-input" id="add_is_active" checked>
                            <label class="form-check-label" for="add_is_active">Active (Show on website)</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_domain" class="btn btn-primary">Add Domain</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Domain Modal -->
<div class="modal fade" id="editDomainModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <input type="hidden" name="domain_id" id="edit_domain_id">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Domain</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">TLD</label>
                        <input type="text" name="tld" id="edit_tld" class="form-control" required placeholder=".com">
                        <div class="form-text">Domain extension (e.g., .com, .net)</div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Register Price (₹)</label>
                                <input type="number" name="register_price" id="edit_register_price" class="form-control" step="0.01" required min="1" placeholder="1099">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Renew Price (₹)</label>
                                <input type="number" name="renew_price" id="edit_renew_price" class="form-control" step="0.01" required min="1" placeholder="1165">
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Transfer Price (₹) <small class="text-muted">(Optional)</small></label>
                        <input type="number" name="transfer_price" id="edit_transfer_price" class="form-control" step="0.01" min="0" placeholder="Leave empty if not applicable">
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Badge <small class="text-muted">(Optional)</small></label>
                                <select name="badge" id="edit_badge" class="form-select">
                                    <option value="">None</option>
                                    <option value="SALE">SALE</option>
                                    <option value="NEW">NEW</option>
                                    <option value="HOT">HOT</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Display Order</label>
                                <input type="number" name="display_order" id="edit_display_order" class="form-control" min="0" value="0">
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input type="checkbox" name="is_active" class="form-check-input" id="edit_is_active">
                            <label class="form-check-label" for="edit_is_active">Active (Show on website)</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="update_domain" class="btn btn-primary">Update Domain</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

<script>
// Edit domain function - Bootstrap should already be loaded
function editDomain(domain) {
    // Populate form fields
    document.getElementById('edit_domain_id').value = domain.id;
    document.getElementById('edit_tld').value = domain.tld;
    document.getElementById('edit_register_price').value = domain.register_price;
    document.getElementById('edit_renew_price').value = domain.renew_price;
    document.getElementById('edit_transfer_price').value = domain.transfer_price || '';
    document.getElementById('edit_badge').value = domain.badge || '';
    document.getElementById('edit_display_order').value = domain.display_order;
    document.getElementById('edit_is_active').checked = domain.is_active == 1;
    
    // Get modal element
    const modalElement = document.getElementById('editDomainModal');
    if (!modalElement) {
        alert('Modal element not found');
        return;
    }
    
    // Use jQuery Bootstrap modal (compatible with both Bootstrap 4 and 5)
    if (typeof $ !== 'undefined' && $.fn.modal) {
        $('#editDomainModal').modal('show');
    } else {
        alert('jQuery or Bootstrap modal not loaded. Please refresh the page.');
    }
}
</script>
