<?php 
require_once __DIR__ . '/../config.php';

// Ensure session for CSRF and flash messages
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Set page-specific variables
$page_title = 'Contact WsplCloud: Get Support for Hosting & Cloud Services';
$page_description = 'Get in touch with WsplCloud support. We\'re here to help with all your web hosting, domain, and cloud service needs. Contact us for assistance or inquiries.';
$page_keywords = 'contact wsplcloud, web hosting support, cloud services help, hosting contact, customer service, WsplCloud support, get in touch, hosting inquiry';
$custom_css = ['breadcrumbs.css']; // Add contact form specific CSS

// Prepare CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Server-side processing for contact form
$errors = [];
$success = false;
$response = ['success' => false, 'errors' => [], 'message' => ''];

if ((isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST')) {
    $client_ip = get_client_ip();
    $rate_key = 'contact_form_' . $client_ip;

    if (!check_rate_limit($rate_key, 5, 600)) {
        $errors[] = 'Too many submissions detected from your network. Please try again in a few minutes.';
    }

    // Basic CSRF and honeypot (company) spam protection
    $token = $_POST['csrf_token'] ?? '';
    $honeypot = $_POST['company'] ?? '';

    if (hash_equals($_SESSION['csrf_token'], (string)$token) === false) {
        $errors[] = 'Invalid session token. Please refresh and try again.';
    }

    if (!empty($honeypot)) {
        // Bot detected — silently fail
        $errors[] = 'Spam detected.';
    }

    // Gather and sanitize inputs
    $fname_raw = trim($_POST['fname'] ?? '');
    $lname_raw = trim($_POST['lname'] ?? '');
    $email_input = trim($_POST['email'] ?? '');
    $phone_raw = trim($_POST['phone'] ?? '');
    $message_raw = trim($_POST['subject'] ?? '');

    $fname = strip_tags($fname_raw);
    $lname = strip_tags($lname_raw);
    $phone = strip_tags($phone_raw);
    $message = strip_tags($message_raw);
    $email = filter_var($email_input, FILTER_VALIDATE_EMAIL);

    if ($fname === '') { $errors[] = 'First name is required.'; }
    if ($email === false) { $errors[] = 'A valid email address is required.'; }
    if ($message === '') { $errors[] = 'Please enter a message.'; }

    // Optional: phone basic normalization
    $phone_clean = preg_replace('/[^+0-9]/', '', $phone);

    if (empty($errors)) {
        // Prepare email
        $to = 'info@wsplcloud.com';
        $subject_line = 'New contact form submission from ' . ($fname . ' ' . $lname);
        $body  = "Name: {$fname} {$lname}\n";
        $body .= "Email: {$email}\n";
        $body .= "Phone: {$phone_clean}\n";
        $body .= "Message:\n" . wordwrap($message, 70) . "\n";

        $headers  = "From: no-reply@wsplcloud.local\r\n";
        $headers .= "Reply-To: {$email}\r\n";
        $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";

        // Attempt to send email — if mail isn't available on system this may fail silently
        $mail_sent = false;
        try {
            $mail_sent = @mail($to, $subject_line, $body, $headers);
        } catch (Exception $e) {
            $mail_sent = false;
        }

        // Persist submission to a simple log file for record-keeping
        $storageDir = BASE_PATH . '/storage';
        if (!is_dir($storageDir)) {
            @mkdir($storageDir, 0755, true);
        }
        $logFile = $storageDir . '/contacts.log';
        $logEntry = sprintf("[%s] %s %s <%s> %s\nMessage: %s\n\n", date('c'), $fname, $lname, $email, $phone_clean, $message);
        @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);

        // Also save to DB (if available)
        try {
            $db = get_db();
            if ($db) {
                $stmt = $db->prepare('INSERT INTO contacts (fname,lname,email,phone,message,created) VALUES (:f,:l,:e,:p,:m,NOW())');
                $stmt->execute([
                    ':f' => $fname,
                    ':l' => $lname,
                    ':e' => $email,
                    ':p' => $phone_clean,
                    ':m' => $message
                ]);
            }
        } catch (Exception $e) {
            // ignore DB errors but keep log
            error_log('Contact DB save error: ' . $e->getMessage());
        }

        if ($mail_sent) {
            $success = true;
            $response['success'] = true;
            $response['message'] = 'Thank you — your message was sent. We will get back to you shortly.';
        } else {
            // Mail not sent, but logged — still inform user with fallback
            $response['success'] = true;
            $response['message'] = 'We could not send email from this server, but your message was saved. We will contact you soon.';
            $success = true;
        }

        // regenerate CSRF token to avoid double submit
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    } else {
        $response['errors'] = $errors;
        $response['message'] = 'Please correct the highlighted errors and try again.';
    }

    // Always refresh token after processing to keep forms in sync
    $response['csrf_token'] = $_SESSION['csrf_token'];

    // If AJAX request, return JSON and stop output
    $isAjax = (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') || (strpos($_SERVER['HTTP_ACCEPT'] ?? '', 'application/json') !== false);
    if ($isAjax) {
        if (!headers_sent()) {
            http_response_code($response['success'] ? 200 : 400);
        }
        if (ob_get_length()) {
            ob_clean();
        }
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }
}

// Include header meta (DOCTYPE, <head>, opening <body> tag)
require_once BASE_PATH . '/partial/header_meta.php';

// Include the navigation header
require_once BASE_PATH . '/partial/header.php';
?>
<style>
    @media only screen and (max-width: 767px) {
    .banner-main-con,
    .sub-banner-main-con,
    .hosting-pages-sub-banner,
    .banner-main-con .sub-banner-inner-con,
    .sub-banner-main-con .container {
        padding-top: 100px !important;
        padding-bottom: 100px !important;
        max-height: 300px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        text-align: center !important;
    }
    /* Ensure headings remain readable and centered */
    .banner-main-con h1,
    .sub-banner-main-con h1,
    .hosting-pages-sub-banner h1 {
        font-size: 28px !important;
        line-height: 1.15 !important;
        margin: 0 auto !important;
        max-width: 92% !important;
    }
}
</style>
    <?php require_once BASE_PATH . '/partial/preloader.php'; ?>
    <!-- HEADER SECTION START HERE -->
      <!-- HEADER SECTION END HERE -->
    <!-- BANNER SECTION START HERE -->
    <section class="sub-banner-main-con w-100 float-left">
        <div class="container">
            <div class="sub-banner-inner-con wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
                <h1>Contact Us</h1>
            <span class="d-block position-relative"></span>
            <h6>
                <a href="<?= url('') ?>" style="color: #fff; transition: color 0.2s;" onmouseover="this.style.color='#FFD700'" onmouseout="this.style.color='#fff'">Home</a>
                <span style="color: #fff;"> / </span>
                <a href="<?= url('pages/contact') ?>" style="color: #fff; transition: color 0.2s;" onmouseover="this.style.color='#FFD700'" onmouseout="this.style.color='#fff'">Contact</a>
            </h6>
            </div>
        </div>
    </section>
    <!-- BANNER SECTION END HERE -->
    <!-- SUPPORT BOXES START -->
    <div class="contact-links-main-con support-boxes-main-con w-100 float-left p-100">
        <div class="container">
            <div class="support-boxes-inner-con wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
                <div class="contact-box support-box">
                    <figure>
                        <img src="<?= asset('images/location-img.png') ?>" alt="location-img">
                    </figure>
                    <span class="d-block">Contact Detail</span>
                    <p>107/2, Sankari Main Road,Seetharampalayam Post,Tiruchengode – 637211</p>
                </div>
                <div class="contact-box support-box">
                    <figure>
                        <img src="<?= asset('images/mail-img.png') ?>" alt="mail-img">
                    </figure>
                    <span class="d-block">Email</span>
                    <a href="mailto:info@WsplCloud.com">info@wsplcloud.com</a>
                      <a href="mailto:support@wsplhosting.com">support@wsplhosting.com</a>
                 
                </div>
                <div class="contact-box support-box">
                    <figure>
                        <img src="<?= asset('images/phone-img.png') ?>" alt="phone-img">
                    </figure>
                    <span class="d-block">Phone</span>
                    <a href="tel:+91 733 920 2005">+91 733 920 2005</a>
                </div>
                <div class="contact-box support-box">
                    <figure>
                        <img src="<?= asset('images/phone-img.png') ?>" alt="whatsapp-chatbot">
                    </figure>
                    <span class="d-block">WhatsApp Chat</span>
                    <button id="openWhatsAppChatbot" class="whatsapp-chatbot-btn" style="background: #25D366; color: white; border: none; padding: 8px 16px; border-radius: 20px; font-weight: bold; cursor: pointer; transition: all 0.3s ease; font-size: 14px;">
                        Open Chatbot
                    </button>
                </div>
            </div>
        </div>
    </div>
    <!-- SUPPORT BOXES END -->
    
    <!-- WHATSAPP CHAT WIDGET FOR CONTACT PAGE -->
    <div class="whatsapp-contact-widget" id="whatsappContactWidget" style="display: none;">
        <!-- Chat Box -->
        <div class="whatsapp-contact-chat-box" id="whatsappContactChatBox">
            <!-- Chat Header -->
            <div class="chat-header">
                <div class="chat-avatar">
                    <img src="<?= asset('images/whatsapp-avatar.png') ?>" alt="WsplCloud Support" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHZpZXdCb3g9IjAgMCA0MCA0MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPGNpcmNsZSBjeD0iMjAiIGN5PSIyMCIgcj0iMjAiIGZpbGw9IiMyNUM2NDYiLz4KPHBhdGggZD0iTTI4LjI1IDE2LjVWMjUuMjVDMjguMjUgMjUuNjY0MiAyOC4wODU4IDI2LjA0OTIgMjcuNzk1NSAyNi4zMzk1QzI3LjUwNTIgMjYuNjI5OCAyNy4xMjAyIDI2Ljc5NDEgMjYuNzUgMjYuNzk0MUgxMy4yNUMyMi44NTc5IDI2Ljc5NDEgMTMuMTY0MiAyNi42Mjk4IDEyLjg3MzkgMjYuMzM5NUMyMi41ODA1IDI2LjA0OTIgMTIuNzUgMjUuNjY0MiAxMi43NSAyNS4yNVYxNi41QzEyLjc1IDE2LjEwNzggMTIuOTE0MiAxNS43MjI4IDEzLjIwNDUgMTUuNDMyNUMxMy40OTQ4IDE1LjE0MjIgMTMuODc5OCAxNC45NzggMTQuMjUgMTQuOTc4SDI1Ljc1QzI2LjE0MjIgMTQuOTc4IDI2LjUyNzIgMTUuMTQyMiAyNi44MTc1IDE1LjQzMjVDMjcuMTA3OCAxNS43MjI4IDI3LjI3MiAxNi4xMDc4IDI3LjI3MiAxNi41SDI4LjI1WiIgZmlsbD0id2hpdGUiLz4KPC9zdmc+Cg=='">
                </div>
                <div class="chat-info">
                    <h4>WsplCloud Support</h4>
                    <p>Typically replies within 30 minutes</p>
                </div>
                <button class="close-contact-chat" id="closeContactChat">
                    <svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor">
                        <path d="M13.707 2.293a1 1 0 010 1.414L9.414 8l4.293 4.293a1 1 0 01-1.414 1.414L8 9.414l-4.293 4.293a1 1 0 01-1.414-1.414L6.586 8 2.293 3.707a1 1 0 011.414-1.414L8 6.586l4.293-4.293a1 1 0 011.414 0z"/>
                    </svg>
                </button>
            </div>

            <!-- Chat Messages -->
            <div class="chat-messages" id="contactChatMessages">
                <div class="message received">
                    <div class="message-content">
                        <p>Hello! 👋</p>
                        <p>Welcome to WsplCloud! How can we help you today?</p>
                    </div>
                    <span class="message-time">10:30 AM</span>
                </div>
                
                <div class="message received">
                    <div class="message-content">
                        <p>We're here to assist you with:</p>
                        <ul>
                            <li>🌐 Web Hosting Plans</li>
                            <li>☁️ Cloud Services</li>
                            <li>🔧 Technical Support</li>
                            <li>💳 Billing Questions</li>
                        </ul>
                    </div>
                    <span class="message-time">10:31 AM</span>
                </div>
            </div>

            <!-- Chat Input -->
            <div class="chat-input">
                <div class="input-group">
                    <input type="text" id="whatsappContactMessage" placeholder="Type your message here..." aria-label="Type your message">
                    <button type="button" id="sendContactWhatsApp" aria-label="Send message">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                        </svg>
                    </button>
                </div>
                <p class="chat-note">Click send to open WhatsApp with your message</p>
            </div>
        </div>
    </div>
    
    <!-- CONTACT FORM SECTION START HERE -->
    <section class="contact-form-main-con w-100 float-left padding-top padding-bottom">
        <div class="container">
            <div class="contact-inner-con wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
                <div class="generic-title text-center">
                    <h2>Contact Form</h2>
                    <p>Let's get this converstaion started. Tell us a bit about yourself, and <br> we'll get in touch as soon as we can.</p>
                </div>
                <form class="form-box" method="post" id="contactpage" novalidate aria-describedby="contact-form-status">
                    <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token'] ?? '') ?>">
                    <!-- Honeypot field to trap bots (hide with CSS) -->
                    <div style="position:absolute; left:-10000px; top:auto; width:1px; height:1px; overflow:hidden;" aria-hidden="true">
                        <label>Company
                            <input type="text" name="company" tabindex="-1" autocomplete="off">
                        </label>
                    </div>

                    <div class="grid">
                        <div class="form-group">
                            <label for="fname">First name</label>
                            <input type="text" name="fname" id="fname" placeholder="First Name" required aria-required="true">
                            <div class="invalid-feedback" id="err-fname"></div>
                        </div>
                        <div class="form-group">
                            <label for="lname">Last name</label>
                            <input type="text" name="lname" id="lname" placeholder="Last Name">
                        </div>
                        <div class="form-group">
                            <label for="email">Email</label>
                            <input type="email" placeholder="Your Email" name="email" id="email" required aria-required="true">
                            <div class="invalid-feedback" id="err-email"></div>
                        </div>
                        <div class="form-group">
                            <label for="phone">Phone</label>
                            <input type="tel" name="phone" id="phone" placeholder="Phone Number">
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="subject">Message</label>
                        <textarea placeholder="Message" name="subject" id="subject" rows="6" required aria-required="true"></textarea>
                        <div class="invalid-feedback" id="err-subject"></div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" id="submit" class="btn-primary">Send Message</button>
                        <div id="contact-form-status" role="status" aria-live="polite"> 
                            <?php if (!empty($response['message'])): ?>
                                <div class="server-message <?= $response['success'] ? 'success' : 'error' ?>"><?= htmlspecialchars($response['message']) ?></div>
                            <?php endif; ?>
                        </div>
                    </div>
                </form>

                <style>
                    /* Scoped minimal modern styles for contact form */
                    .form-box{background:#ffffff; padding:24px; border-radius:12px; box-shadow:0 8px 30px rgba(15,23,42,0.08);}
                    .grid{display:grid; grid-template-columns:1fr 1fr; gap:12px;}
                    .form-group{display:flex; flex-direction:column;}
                    label{font-weight:600; margin-bottom:6px; color:#0f172a}
                    input[type="text"], input[type="email"], input[type="tel"], textarea{padding:10px 12px; border:1px solid #e6edf3; border-radius:8px; font-size:14px; color:#0f172a}
                    textarea{resize:vertical}
                    .btn-primary{background:linear-gradient(90deg,#3b82f6,#06b6d4); color:white; border:none; padding:10px 18px; border-radius:10px; font-weight:700; cursor:pointer}
                    .btn-primary:disabled{opacity:0.6; cursor:not-allowed}
                    .invalid-feedback{color:#b91c1c; font-size:13px; margin-top:6px}
                    .server-message{margin-top:12px; padding:10px; border-radius:8px}
                    .server-message.success{background:#ecfdf5; color:#064e3b; border:1px solid #bbf7d0}
                    .server-message.error{background:#fff1f2; color:#7f1d1d; border:1px solid #fecaca}
                    @media (max-width:700px){.grid{grid-template-columns:1fr}}
                </style>
                
                <!-- WhatsApp Contact Widget Styles -->
                <style>


                    .whatsapp-contact-widget {
                        position: fixed;
                        top: 50%;
                        left: 50%;
                        transform: translate(-50%, -50%);
                        z-index: 10001;
                        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    }

                    .whatsapp-contact-chat-box {
                        width: 400px;
                        height: 500px;
                        background: white;
                        border-radius: 16px;
                        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.25);
                        display: flex;
                        flex-direction: column;
                        overflow: hidden;
                        animation: chatBoxAppear 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                    }

                    @keyframes chatBoxAppear {
                        from {
                            opacity: 0;
                            transform: scale(0.8);
                        }
                        to {
                            opacity: 1;
                            transform: scale(1);
                        }
                    }

                    .whatsapp-contact-widget .chat-header {
                        background: linear-gradient(135deg, #25D366, #128C7E);
                        padding: 16px;
                        display: flex;
                        align-items: center;
                        color: white;
                        position: relative;
                    }

                    .whatsapp-contact-widget .chat-avatar {
                        width: 40px;
                        height: 40px;
                        border-radius: 50%;
                        overflow: hidden;
                        margin-right: 12px;
                        border: 2px solid rgba(255, 255, 255, 0.3);
                    }

                    .whatsapp-contact-widget .chat-avatar img {
                        width: 100%;
                        height: 100%;
                        object-fit: cover;
                    }

                    .whatsapp-contact-widget .chat-info h4 {
                        margin: 0;
                        font-size: 16px;
                        font-weight: 600;
                        color: white;
                    }

                    .whatsapp-contact-widget .chat-info p {
                        margin: 2px 0 0 0;
                        font-size: 12px;
                        opacity: 0.9;
                        color: white;
                    }

                    .close-contact-chat {
                        position: absolute;
                        top: 16px;
                        right: 16px;
                        background: none;
                        border: none;
                        color: white;
                        cursor: pointer;
                        opacity: 0.8;
                        transition: opacity 0.2s;
                        padding: 4px;
                        border-radius: 4px;
                    }

                    .close-contact-chat:hover {
                        opacity: 1;
                        background: rgba(255, 255, 255, 0.1);
                    }

                    .whatsapp-contact-widget .chat-messages {
                        flex: 1;
                        padding: 16px;
                        overflow-y: auto;
                        background: #f0f0f0;
                    }

                    .whatsapp-contact-widget .message {
                        margin-bottom: 16px;
                        max-width: 85%;
                    }

                    .whatsapp-contact-widget .message.received {
                        align-self: flex-start;
                    }

                    .whatsapp-contact-widget .message.sent {
                        align-self: flex-end;
                        text-align: right;
                    }

                    .whatsapp-contact-widget .message-content {
                        background: white;
                        padding: 12px 16px;
                        border-radius: 16px 16px 16px 4px;
                        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
                    }

                    .whatsapp-contact-widget .message.received .message-content {
                        background: white;
                    }

                    .whatsapp-contact-widget .message.sent .message-content {
                        background: linear-gradient(135deg, #25D366, #128C7E);
                        color: white;
                        border-radius: 16px 16px 4px 16px;
                    }

                    .whatsapp-contact-widget .message-content p {
                        margin: 0 0 8px 0;
                        font-size: 14px;
                        line-height: 1.4;
                        color: #333;
                    }

                    .whatsapp-contact-widget .message.sent .message-content p {
                        color: white;
                    }

                    .whatsapp-contact-widget .message-content p:last-child {
                        margin-bottom: 0;
                    }

                    .whatsapp-contact-widget .message-content ul {
                        margin: 8px 0;
                        padding-left: 16px;
                    }

                    .whatsapp-contact-widget .message-content li {
                        font-size: 13px;
                        margin-bottom: 4px;
                        color: #333;
                    }

                    .whatsapp-contact-widget .message-time {
                        display: block;
                        font-size: 11px;
                        color: #666;
                        margin-top: 4px;
                        text-align: right;
                    }

                    .whatsapp-contact-widget .chat-input {
                        padding: 16px;
                        background: white;
                        border-top: 1px solid #e0e0e0;
                    }

                    .whatsapp-contact-widget .input-group {
                        display: flex;
                        gap: 8px;
                        margin-bottom: 8px;
                    }

                    #whatsappContactMessage {
                        flex: 1;
                        padding: 12px 16px;
                        border: 1px solid #e0e0e0;
                        border-radius: 24px;
                        font-size: 14px;
                        outline: none;
                        transition: border-color 0.2s;
                    }

                    #whatsappContactMessage:focus {
                        border-color: #25D366;
                    }

                    #sendContactWhatsApp {
                        background: #25D366;
                        color: white;
                        border: none;
                        border-radius: 50%;
                        width: 44px;
                        height: 44px;
                        cursor: pointer;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        transition: all 0.2s;
                    }

                    #sendContactWhatsApp:hover {
                        background: #128C7E;
                        transform: scale(1.05);
                    }

                    .whatsapp-contact-widget .chat-note {
                        font-size: 11px;
                        color: #666;
                        text-align: center;
                        margin: 0;
                    }

                    /* Responsive Design */
                    @media (max-width: 768px) {
                        .whatsapp-contact-chat-box {
                            width: 90vw;
                            height: 80vh;
                            max-width: 350px;
                        }
                    }

                    @media (max-width: 480px) {
                        .whatsapp-contact-chat-box {
                            width: 95vw;
                            height: 85vh;
                            max-width: 320px;
                        }
                    }

                    /* Scrollbar Styling */
                    .whatsapp-contact-widget .chat-messages::-webkit-scrollbar {
                        width: 4px;
                    }

                    .whatsapp-contact-widget .chat-messages::-webkit-scrollbar-track {
                        background: #f1f1f1;
                    }

                    .whatsapp-contact-widget .chat-messages::-webkit-scrollbar-thumb {
                        background: #c1c1c1;
                        border-radius: 2px;
                    }

                    .whatsapp-contact-widget .chat-messages::-webkit-scrollbar-thumb:hover {
                        background: #a8a8a8;
                    }
                </style>
            </div>
        </div>
    </section>
    <!-- CONTACT FORM SECTION END HERE -->
    <!-- INQUIRY SECTION SATRT HERE -->
    <section class="inquiry-main-con w-100 float-left padding-top padding-bottom">
        <div class="container">
            <div class="inquiry-inner-con wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
                <div class="inqury-box">
                    <div class="inquiry-heading">
                        <figure class="mb-0">
                            <img src="<?= asset('images/inquiry-img1.png') ?>" alt="inquiry-img1">
                        </figure>
                        <h5>Report Abuse</h5>
                    </div>
                    <p>If you have encountered abusive, harmful, or suspicious content or activity related to our hosting services, please report it immediately. At WsplCloud, we take abuse reports seriously and are committed to maintaining a safe and secure online environment.</p>
                </div>
                <div class="inqury-box">
                    <div class="inquiry-heading">
                        <figure class="mb-0">
                            <img src="<?= asset('images/inquiry-img2.png') ?>" alt="inquiry-img2">
                        </figure>
                        <h5>General Inquiry</h5>
                    </div>
                    <p>Whether you're looking for information about our web hosting plans, cloud solutions, custom development services, or just have a general question, the WssplCloud Support Team is ready to assist you.</p>
                </div>
            </div>
        </div>
    </section>
    <!-- INQUIRY SECTION END HERE -->
    
    <!-- WORLD MAP SECTION START HERE -->
    <section class="world-map-main-con w-100 float-left padding-top padding-bottom">
        <div class="container">
            <div class="generic-title text-center wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
                <h2>Our Global Server Network</h2>
                <p>Experience lightning-fast performance with our worldwide server locations</p>
            </div>
            
            <div class="world-map-container wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.5s">
                <div class="floating-particles">
                    <div class="particle particle-1"></div>
                    <div class="particle particle-2"></div>
                    <div class="particle particle-3"></div>
                    <div class="particle particle-4"></div>
                    <div class="particle particle-5"></div>
                </div>
                
                <div class="world-map">
                    <img src="<?= asset('images/worldmap.png') ?>" alt="World Map" class="world-map-img" />
                    
                    <!-- Connection Lines -->
                    <svg class="connection-lines" viewBox="0 0 800 600">
                        <defs>
                            <linearGradient id="lineGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                <stop offset="0%" style="stop-color:#4362f8;stop-opacity:0.8"/>
                                <stop offset="50%" style="stop-color:#06b6d4;stop-opacity:0.6"/>
                                <stop offset="100%" style="stop-color:#10b981;stop-opacity:0.4"/>
                            </linearGradient>
                        </defs>
                        <path class="connection-line line-1" d="M144,270 Q300,200 416,186" stroke="url(#lineGradient)" stroke-width="2" fill="none"/>
                        <path class="connection-line line-2" d="M416,186 Q500,250 540,348" stroke="url(#lineGradient)" stroke-width="2" fill="none"/>
                        <path class="connection-line line-3" d="M540,348 Q580,380 588,405" stroke="url(#lineGradient)" stroke-width="2" fill="none"/>
                        <path class="connection-line line-4" d="M588,405 Q650,350 672,312" stroke="url(#lineGradient)" stroke-width="2" fill="none"/>
                        <path class="connection-line line-5" d="M672,312 Q700,450 656,468" stroke="url(#lineGradient)" stroke-width="2" fill="none"/>
                    </svg>
                    
                    <div class="pin usa" data-country="United States">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/usa.png') ?>" alt="USA" />
                        <div class="pin-tooltip">🇺🇸 USA - New York<br><small>Latency: 15ms</small></div>
                    </div>
                    
                    <div class="pin germany" data-country="Germany">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/germany.png') ?>" alt="Germany" />
                        <div class="pin-tooltip">🇩🇪 Germany - Frankfurt<br><small>Latency: 12ms</small></div>
                    </div>
                    
                    <div class="pin india" data-country="India">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/india.png') ?>" alt="India" />
                        <div class="pin-tooltip">🇮🇳 India - Mumbai<br><small>Latency: 5ms</small></div>
                    </div>
                    
                    <div class="pin singapore" data-country="Singapore">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/singapore.png') ?>" alt="Singapore" />
                        <div class="pin-tooltip">🇸🇬 Singapore<br><small>Latency: 8ms</small></div>
                    </div>
                    
                    <div class="pin japan" data-country="Japan">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/japan.png') ?>" alt="Japan" />
                        <div class="pin-tooltip">🇯🇵 Japan - Tokyo<br><small>Latency: 10ms</small></div>
                    </div>
                    
                    <div class="pin australia" data-country="Australia">
                        <div class="pin-glow"></div>
                        <div class="ping-animation"></div>
                        <img src="<?= asset('images/australia.png') ?>" alt="Australia" />
                        <div class="pin-tooltip">🇦🇺 Australia - Sydney<br><small>Latency: 18ms</small></div>
                    </div>
                </div>
            </div>
        </div>
            
    
        </div>
    </section>
    <!-- WORLD MAP SECTION END HERE -->

    <style>
        /* Server Cards Container - Inside World Map Section */
        .server-cards-container {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 20px;
            margin: 40px auto 0;
            max-width: 1000px;
            position: relative;
            min-height: 260px;
        }
        
        .server-card {
            position: absolute;
            width: 220px;
            height: 150px;
            perspective: 1000px;
            transition: all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            z-index: 1;
            opacity: 0;
            transform: translateX(100px) scale(0.8) rotateY(10deg);
            filter: blur(2px);
        }
        
        .server-card.active {
            opacity: 1;
            transform: translateX(0) scale(1) rotateY(0deg);
            z-index: 3;
            filter: blur(0px);
        }
        
        .server-card.prev {
            opacity: 0.7;
            transform: translateX(-220px) scale(0.85) rotateY(15deg);
            z-index: 2;
            filter: blur(1px);
        }
        
        .server-card.next {
            opacity: 0.7;
            transform: translateX(220px) scale(0.85) rotateY(-15deg);
            z-index: 2;
            filter: blur(1px);
        }
        
        .server-card.far-prev {
            opacity: 0.2;
            transform: translateX(-320px) scale(0.7);
            z-index: 1;
        }
        
        .server-card.far-next {
            opacity: 0.2;
            transform: translateX(320px) scale(0.7);
            z-index: 1;
        }
        
        .card-content {
            width: 100%;
            height: 100%;
           
            background: rgba(0, 21, 255, 0.95);
            border-radius: 20px;
            box-shadow: 
                0 10px 30px rgba(0, 102, 255, 0.2),
                0 4px 10px rgba(0, 102, 255, 0.1);
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 15px;
            text-align: center;
            transition: all 0.4s ease;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 210, 0, 0.3);
        }
        
        .server-card.active .card-content {
            box-shadow: 
                0 15px 40px rgba(0, 102, 255, 0.3),
                0 8px 20px rgba(255, 210, 0, 0.2);
            border: 1px solid rgba(255, 210, 0, 0.5);
            transform: translateY(-5px);
        }
        
        .server-card:hover .card-content {
            transform: translateY(-8px);
            box-shadow: 
                0 20px 40px rgba(0, 102, 255, 0.25),
                0 10px 20px rgba(255, 210, 0, 0.15);
        }
        
        .flag-container {
            width: 100px;
            height: 100px;
            border-radius: 12px;
            overflow: hidden;
            margin: 0 auto;
            box-shadow: 0 6px 12px rgba(0, 102, 255, 0.3);
            transition: all 0.4s ease;
            position: relative;
            z-index: 2;
            border: 3px solid rgba(255, 210, 0, 0.4);
        }
        
        .server-card.active .flag-container {
            transform: scale(1.2);
            box-shadow: 0 10px 20px rgba(0, 102, 255, 0.4);
            border: 3px solid rgba(255, 210, 0, 0.7);
        }
        
        .flag {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: all 0.6s cubic-bezier(0.4, 0, 0.2, 1);
            filter: brightness(1.1) contrast(1.1);
            transform: scale(1);
        }
        
        .server-card.active .flag {
            filter: brightness(1.2) contrast(1.2) saturate(1.1);
            transform: scale(1.05);
        }
        
        .flag-container:hover .flag {
            transform: scale(1.1) rotate(2deg);
            filter: brightness(1.15) contrast(1.15) saturate(1.2);
        }
        
        /* Enhanced Card Animations */
        @keyframes cardFloat {
            0%, 100% {
                transform: translateY(0px);
            }
            50% {
                transform: translateY(-5px);
            }
        }
        
        @keyframes flagPulse {
            0%, 100% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.02);
            }
        }
        
        .server-card.active {
            animation: cardFloat 3s ease-in-out infinite;
        }
        
        .flag-container:hover {
            animation: flagPulse 0.8s ease-in-out;
        }
        
        /* Responsive Design for Server Cards */
        @media (max-width: 1200px) {
            .server-cards-container {
                max-width: 900px;
            }
            
            .server-card {
                width: 200px;
                height: 140px;
            }
            
            .flag-container {
                width: 160px;
                height: 160px;
            }
        }
        
        @media (max-width: 992px) {
            .server-cards-container {
                max-width: 700px;
            }
            
            .server-card {
                width: 180px;
                height: 130px;
            }
            
            .flag-container {
                width: 140px;
                height: 140px;
            }
            
            .server-card.prev {
                transform: translateX(-180px) scale(0.85);
            }
            
            .server-card.next {
                transform: translateX(180px) scale(0.85);
            }
            
            .server-card.far-prev {
                transform: translateX(-240px) scale(0.7);
            }
            
            .server-card.far-next {
                transform: translateX(240px) scale(0.7);
            }
        }
        
        @media (max-width: 768px) {
            .server-cards-container {
                max-width: 500px;
                min-height: 140px;
            }
            
            .server-card {
                width: 160px;
                height: 120px;
            }
            
            .flag-container {
                width: 120px;
                height: 120px;
            }
            
            .server-card.prev {
                transform: translateX(-160px) scale(0.85);
            }
            
            .server-card.next {
                transform: translateX(160px) scale(0.85);
            }
            
            .server-card.far-prev {
                transform: translateX(-200px) scale(0.7);
            }
            
            .server-card.far-next {
                transform: translateX(200px) scale(0.7);
            }
            
            .location-info h3 {
                font-size: 1.1rem;
            }
            
            .location {
                font-size: 0.8rem;
            }
        }
        
        @media (max-width: 576px) {
            .server-cards-container {
                max-width: 100%;
                margin: 30px 0 0;
                min-height: 120px;
            }
            
            .server-card {
                width: 180px;
                height: 110px;
            }
            
            .flag-container {
                width: 100px;
                height: 100px;
            }
            
            .server-card.prev,
            .server-card.next,
            .server-card.far-prev,
            .server-card.far-next {
                display: none;
            }
            
            .server-card.active {
                position: relative;
                transform: none;
            }
        }
        
        /* World Map Section Styles */
        .world-map-main-con {
            position: relative;
            overflow: hidden;
              margin-bottom: 100px;
        }
        
        .world-map-main-con::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-size: cover;
            background-position: center;
            opacity: 1;
            z-index: 1;
        }
        
        .world-map-main-con .container {
            position: relative;
            z-index: 2;
          
        }
        
        .world-map-main-con .generic-title h2 {
            color: #000000ff;
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 15px;
            text-shadow: 0 4px 8px rgba(0,0,0,0.3);
        }
        
        .world-map-main-con .generic-title p {
            color: rgba(255,255,255,0.9);
            font-size: 1.1rem;
            max-width: 700px;
            margin: 0 auto;
        }
        
        .world-map-container {
            position: relative;
            margin-top: 60px;
        }
        
        .floating-particles {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
        }
        
        .particle {
            position: absolute;
            width: 4px;
            height: 4px;
            background: rgba(255,255,255,0.6);
            border-radius: 50%;
            animation: floatParticle 8s ease-in-out infinite;
        }
        
        .particle-1 { top: 20%; left: 20%; animation-delay: 0s; }
        .particle-2 { top: 40%; left: 80%; animation-delay: -2s; }
        .particle-3 { top: 70%; left: 10%; animation-delay: -4s; }
        .particle-4 { top: 30%; left: 60%; animation-delay: -6s; }
        .particle-5 { top: 80%; left: 70%; animation-delay: -8s; }
        
        .world-map {
            position: relative;
            width: 100%;
            max-width: 800px;
            margin: 0 auto;
            padding: 2rem;
           
            z-index: 2;
        }
        
        .world-map-img {
            width: 100%;
            height: auto;
            display: block;
            object-fit: contain;
            filter: drop-shadow(0 10px 20px rgba(0,0,0,0.1)) contrast(1.1) brightness(1.05);
            transition: all 0.5s ease;
        }
        
        .world-map:hover .world-map-img {
            filter: drop-shadow(0 15px 30px rgba(0,0,0,0.15)) contrast(1.2) brightness(1.1);
            transform: scale(1.02);
        }
        
        .connection-lines {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 3;
        }
        
        .connection-line {
            stroke-dasharray: 5,10;
            animation: dashFlow 3s linear infinite;
        }
        
        .line-1 { animation-delay: 0s; }
        .line-2 { animation-delay: 0.6s; }
        .line-3 { animation-delay: 1.2s; }
        .line-4 { animation-delay: 1.8s; }
        .line-5 { animation-delay: 2.4s; }
        
        .pin {
            position: absolute;
            width: 12px;
            height: 12px;
            cursor: pointer;
            z-index: 10;
            transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
            animation: pinFloat 4s ease-in-out infinite;
        }
        
        .pin::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 12px;
            height: 12px;
            background: linear-gradient(45deg, #4362f8, #06b6d4);
            border-radius: 50%;
            box-shadow: 0 4px 15px rgba(67, 98, 248, 0.4);
        }
        
        .pin:hover {
            transform: scale(1.5);
            z-index: 100;
        }
        
        .pin:hover::before {
            background: linear-gradient(45deg, #f59e0b, #ef4444);
            box-shadow: 0 8px 25px rgba(245, 158, 11, 0.6);
            animation: pinPulse 0.6s ease-in-out infinite;
        }
        
        .pin-glow {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 30px;
            height: 30px;
            background: radial-gradient(circle, rgba(67, 98, 248, 0.3) 0%, transparent 70%);
            border-radius: 50%;
            animation: glowPulse 2s ease-in-out infinite;
            z-index: -1;
        }
        
        .ping-animation {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 12px;
            height: 12px;
            border: 2px solid rgba(67, 98, 248, 0.8);
            border-radius: 50%;
            animation: ping 2s cubic-bezier(0, 0, 0.2, 1) infinite;
            z-index: -1;
        }
        
        .pin img {
            position: absolute;
            top: 50%;
            left: 20px;
            transform: translateY(-50%);
            width: 80px;
            
            height: 80px;
            max-height: 300px;
            padding: 4px;
            background: linear-gradient(135deg, #1900ffff, #000dfeff);
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            object-fit: cover;
            transition: all 0.3s ease;
            opacity: 0;
            visibility: hidden;
        }
        
        .pin:hover img {
            opacity: 1;
            visibility: visible;
            transform: translateY(-50%) scale(1.1);
            box-shadow: 0 8px 25px rgba(0,0,0,0.3);
        }
        
        .pin-tooltip {
            position: absolute;
            bottom: 25px;
            left: 50%;
            transform: translateX(-50%);
            background: linear-gradient(135deg, #2d3748, #4a5568);
            color: white;
            padding: 12px 16px;
            border-radius: 12px;
            font-size: 0.9rem;
            font-weight: 500;
            white-space: nowrap;
            text-align: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s cubic-bezier(0.175, 0.885, 0.32, 1.275);
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
            backdrop-filter: blur(10px);
            z-index: 50;
        }
        
        .pin-tooltip::after {
            content: '';
            position: absolute;
            top: 100%;
            left: 50%;
            transform: translateX(-50%);
            border: 8px solid transparent;
            border-top-color: #2d3748;
        }
        
        .pin:hover .pin-tooltip {
            opacity: 1;
            visibility: visible;
            bottom: 35px;
            transform: translateX(-50%) translateY(-5px);
        }
        
        .pin-tooltip small {
            display: block;
            color: rgba(255,255,255,0.8);
            font-size: 0.75rem;
            margin-top: 4px;
        }
        
        /* Pin Positions */
        .usa { top: 45%; left: 18%; animation-delay: 0s; }
        .germany { top: 32%; left: 52%; animation-delay: 0.8s; }
        .india { top: 58.5%; left: 66.5%; animation-delay: 1.6s; }
        .singapore { top: 67%; left: 72.5%; animation-delay: 2.4s; }
        .japan { top: 50%; left: 82.5%; animation-delay: 3.2s; }
        .australia { top: 76%; left: 80%; animation-delay: 4s; }
        
        /* Animations */
        @keyframes floatParticle {
            0%, 100% { transform: translateY(0) translateX(0); opacity: 0.3; }
            50% { transform: translateY(-15px) translateX(10px); opacity: 0.8; }
        }
        
        @keyframes dashFlow {
            0% { stroke-dashoffset: 0; }
            100% { stroke-dashoffset: 15; }
        }
        
        @keyframes pinFloat {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-8px); }
        }
        
        @keyframes pinPulse {
            0%, 100% { transform: translate(-50%, -50%) scale(1); }
            50% { transform: translate(-50%, -50%) scale(1.2); }
        }
        
        @keyframes glowPulse {
            0%, 100% { opacity: 0.5; transform: translate(-50%, -50%) scale(1); }
            50% { opacity: 0.8; transform: translate(-50%, -50%) scale(1.2); }
        }
        
        @keyframes ping {
            75%, 100% {
                transform: translate(-50%, -50%) scale(2);
                opacity: 0;
            }
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .world-map-main-con .generic-title h2 {
                font-size: 2rem;
            }
            
            .world-map {
                padding: 1rem;
                margin: 0 10px;
            }
            
            .pin img {
                width: 30px;
                max-height: 22px;
                left: 15px;
            }
            
            .pin-tooltip {
                font-size: 0.8rem;
                padding: 8px 12px;
            }
            
            .particle {
                width: 2px;
                height: 2px;
            }
        }
        
        @media (max-width: 480px) {
            .world-map-main-con .generic-title h2 {
                font-size: 1.7rem;
            }
            
            .pin img {
                width: 25px;
                max-height: 18px;
                left: 12px;
            }
            
            .pin-tooltip {
                font-size: 0.7rem;
                padding: 6px 10px;
            }
        }
    </style>

  
   
    <!-- FOOTER SECTION START HERE -->
    <?php require_once BASE_PATH . '/partial/footer.php'; ?>
    <!-- Back-to-top removed -->
    <!-- FOOTER SECTION END HERE -->
    <!-- js start -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.server-card');
            let currentIndex = 0;
            
            // Initialize cards
            updateCardPositions();
            
            function updateCardPositions() {
                cards.forEach((card, index) => {
                    card.classList.remove('active', 'prev', 'next', 'far-prev', 'far-next');
                    
                    if (index === currentIndex) {
                        card.classList.add('active');
                    } else if (index === currentIndex - 1 || (currentIndex === 0 && index === cards.length - 1)) {
                        card.classList.add('prev');
                    } else if (index === currentIndex + 1 || (currentIndex === cards.length - 1 && index === 0)) {
                        card.classList.add('next');
                    } else if (index === currentIndex - 2 || (currentIndex <= 1 && index >= cards.length - 2 + currentIndex)) {
                        card.classList.add('far-prev');
                    } else if (index === currentIndex + 2 || (currentIndex >= cards.length - 2 && index <= 1 - (cards.length - 1 - currentIndex))) {
                        card.classList.add('far-next');
                    }
                });
            }
            
            function goToSlide(index) {
                if (index < 0) index = cards.length - 1;
                if (index >= cards.length) index = 0;
                
                currentIndex = index;
                updateCardPositions();
            }
            
            // Auto-rotate cards every 4 seconds
            let autoRotate = setInterval(() => {
                goToSlide(currentIndex + 1);
            }, 4000);
            
            // Pause auto-rotation on hover
            const cardsContainer = document.querySelector('.server-cards-container');
            if (cardsContainer) {
                cardsContainer.addEventListener('mouseenter', () => {
                    clearInterval(autoRotate);
                });
                
                cardsContainer.addEventListener('mouseleave', () => {
                    autoRotate = setInterval(() => {
                        goToSlide(currentIndex + 1);
                    }, 4000);
                });
            }
            
            // Touch/swipe support for mobile
            let startX = 0;
            let endX = 0;
            const minSwipeDistance = 50;
            
            if (cardsContainer) {
                cardsContainer.addEventListener('touchstart', (e) => {
                    startX = e.touches[0].clientX;
                });
                
                cardsContainer.addEventListener('touchend', (e) => {
                    endX = e.changedTouches[0].clientX;
                    const deltaX = startX - endX;
                    
                    if (Math.abs(deltaX) > minSwipeDistance) {
                        if (deltaX > 0) {
                            goToSlide(currentIndex + 1); // Swipe left - next slide
                        } else {
                            goToSlide(currentIndex - 1); // Swipe right - previous slide
                        }
                    }
                });
            }
        });
    </script>
    <?php 
    // Add page-specific scripts if needed
    $page_scripts = ['jquery.min.js'];
    ?>

    <script>
    // WhatsApp Contact Chatbot Functionality
    document.addEventListener('DOMContentLoaded', function() {
        const chatbotBtn = document.getElementById('openWhatsAppChatbot');
        const contactWidget = document.getElementById('whatsappContactWidget');
        const contactChatBox = document.getElementById('whatsappContactChatBox');
        const closeContactChat = document.getElementById('closeContactChat');
        const sendContactWhatsApp = document.getElementById('sendContactWhatsApp');
        const whatsappContactMessage = document.getElementById('whatsappContactMessage');
        const contactChatMessages = document.getElementById('contactChatMessages');
        
        // Function to add user message to contact chat
        function addContactUserMessage(message) {
            const currentTime = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            
            const userMessageHtml = `
                <div class="message sent">
                    <div class="message-content">
                        <p>${message}</p>
                    </div>
                    <span class="message-time">${currentTime}</span>
                </div>
            `;
            
            contactChatMessages.insertAdjacentHTML('beforeend', userMessageHtml);
            contactChatMessages.scrollTop = contactChatMessages.scrollHeight;
        }
        
        // Function to add bot reply to contact chat
        function addContactBotReply() {
            const currentTime = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            
            const botReplyHtml = `
                <div class="message received">
                    <div class="message-content">
                        <p>Thank you for your message! 🙏</p>
                        <p>We're redirecting you to WhatsApp where our support team will assist you shortly. Please wait a moment...</p>
                    </div>
                    <span class="message-time">${currentTime}</span>
                </div>
            `;
            
            contactChatMessages.insertAdjacentHTML('beforeend', botReplyHtml);
            contactChatMessages.scrollTop = contactChatMessages.scrollHeight;
        }

        // Send message to WhatsApp from contact chat
        function sendContactToWhatsApp() {
            const message = whatsappContactMessage.value.trim();
            const phoneNumber = '917339202005';
            
            try {
                if (message) {
                    // Add user message to chat first
                    addContactUserMessage(message);
                    
                    // Clear input immediately
                    whatsappContactMessage.value = '';
                    
                    // Add bot reply after a short delay
                    setTimeout(() => {
                        addContactBotReply();
                    }, 800);
                    
                    // Open WhatsApp after showing the reply
                    setTimeout(() => {
                        const encodedMessage = encodeURIComponent(message);
                        const whatsappUrl = `https://wa.me/${phoneNumber}?text=${encodedMessage}`;
                        
                        const opened = window.open(whatsappUrl, '_blank');
                        
                        if (!opened || opened.closed || typeof opened.closed == 'undefined') {
                            window.location.href = whatsappUrl;
                        }
                    }, 1500);
                    
                    // Close chat after everything is done
                    setTimeout(() => {
                        contactWidget.style.display = 'none';
                    }, 3000);
                } else {
                    // If no message, show default reply and open WhatsApp
                    addContactBotReply();
                    
                    setTimeout(() => {
                        const whatsappUrl = `https://wa.me/${phoneNumber}`;
                        
                        const opened = window.open(whatsappUrl, '_blank');
                        
                        if (!opened || opened.closed || typeof opened.closed == 'undefined') {
                            window.location.href = whatsappUrl;
                        }
                    }, 1000);
                    
                    setTimeout(() => {
                        contactWidget.style.display = 'none';
                    }, 2500);
                }
            } catch (error) {
                console.error('Error opening WhatsApp:', error);
                const currentTime = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
                
                const errorMessageHtml = `
                    <div class="message received">
                        <div class="message-content">
                            <p>We're having trouble opening WhatsApp automatically. Please click the link below:</p>
                            <a href="https://wa.me/917339202005${message ? '?text=' + encodeURIComponent(message) : ''}" target="_blank" style="color: #25D366; font-weight: bold;">Open WhatsApp</a>
                        </div>
                        <span class="message-time">${currentTime}</span>
                    </div>
                `;
                
                contactChatMessages.insertAdjacentHTML('beforeend', errorMessageHtml);
                contactChatMessages.scrollTop = contactChatMessages.scrollHeight;
            }
        }
        
        if (chatbotBtn) {
            chatbotBtn.addEventListener('click', function(e) {
                e.preventDefault();
                
                // Add click feedback
                this.style.transform = 'scale(0.95)';
                this.innerHTML = 'Opening...';
                
                // Show the contact chat widget
                contactWidget.style.display = 'block';
                
                // Visual feedback
                setTimeout(() => {
                    this.innerHTML = 'Chatbot Opened! ✓';
                    this.style.background = '#128C7E';
                }, 200);
                
                // Reset button after delay
                setTimeout(() => {
                    this.innerHTML = 'Open Chatbot';
                    this.style.background = '#25D366';
                    this.style.transform = 'scale(1)';
                }, 2500);
            });
            
            // Add hover effects
            chatbotBtn.addEventListener('mouseenter', function() {
                if (this.innerHTML === 'Open Chatbot') {
                    this.style.background = '#128C7E';
                    this.style.transform = 'scale(1.05)';
                }
            });
            
            chatbotBtn.addEventListener('mouseleave', function() {
                if (this.innerHTML === 'Open Chatbot') {
                    this.style.background = '#25D366';
                    this.style.transform = 'scale(1)';
                }
            });
        }
        
        // Close contact chat
        if (closeContactChat) {
            closeContactChat.addEventListener('click', function() {
                contactWidget.style.display = 'none';
            });
        }
        
        // Send button functionality
        if (sendContactWhatsApp) {
            sendContactWhatsApp.addEventListener('click', function(e) {
                e.preventDefault();
                
                // Visual feedback
                this.style.transform = 'scale(0.95)';
                this.innerHTML = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><circle cx="12" cy="12" r="3"><animate attributeName="r" values="3;5;3" dur="0.5s" repeatCount="1"/></circle></svg>';
                
                // Execute send function
                sendContactToWhatsApp();
                
                // Reset button after delay
                setTimeout(() => {
                    this.style.transform = 'scale(1)';
                    this.innerHTML = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/></svg>';
                }, 1000);
            });
        }
        
        // Enter key to send
        if (whatsappContactMessage) {
            whatsappContactMessage.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    sendContactToWhatsApp();
                }
            });
        }
        
        // Close chat when clicking outside
        if (contactWidget) {
            contactWidget.addEventListener('click', function(e) {
                if (e.target === contactWidget) {
                    contactWidget.style.display = 'none';
                }
            });
        }
    });
    
    (function(){
        const form = document.getElementById('contactpage');
        const submitBtn = document.getElementById('submit');
        const statusEl = document.getElementById('contact-form-status');

        function setError(id, message){
            const el = document.getElementById(id);
            if(el) el.textContent = message;
        }

        function clearErrors(){
            ['err-fname','err-email','err-subject'].forEach(id => setError(id, ''));
            statusEl.innerHTML = '';
        }

        function validate(){
            let ok = true;
            clearErrors();
            const fname = form.fname.value.trim();
            const email = form.email.value.trim();
            const subject = form.subject.value.trim();
            if(fname.length === 0){ setError('err-fname','First name is required.'); ok = false; }
            if(!/^[^@\s]+@[^@\s]+\.[^@\s]+$/.test(email)){ setError('err-email','Please enter a valid email address.'); ok = false; }
            if(subject.length < 5){ setError('err-subject','Message must be at least 5 characters.'); ok = false; }
            return ok;
        }

        async function postForm(data){
            submitBtn.disabled = true;
            try {
                const resp = await fetch(window.location.href, {
                    method: 'POST',
                    headers: { 'X-Requested-With': 'XMLHttpRequest', 'Accept': 'application/json' },
                    body: data
                });

                const raw = await resp.text();
                let parsed = {};

                if (raw.trim().length) {
                    try {
                        parsed = JSON.parse(raw);
                    } catch (jsonErr) {
                        parsed = {
                            success: resp.ok,
                            message: 'Unexpected server response. Please try again.'
                        };
                    }
                } else {
                    parsed = {
                        success: resp.ok,
                        message: resp.ok ? 'Thank you — your message has been received.' : 'Unexpected empty response from server.'
                    };
                }

                parsed.status = resp.status;
                return parsed;
            } catch (err) {
                return { success: false, message: 'Network error. Please try again.', status: 0 };
            } finally {
                submitBtn.disabled = false;
            }
        }

        if(form){
            form.addEventListener('submit', async function(e){
                e.preventDefault();
                if(!validate()) return;
                const data = new FormData(form);
                // Append indicator for ajax
                data.append('ajax', '1');
                statusEl.innerHTML = '<div class="server-message">Sending…</div>';
                const result = await postForm(data);

                if(result && result.csrf_token){
                    const tokenField = form.querySelector('input[name="csrf_token"]');
                    if(tokenField){
                        tokenField.value = result.csrf_token;
                        tokenField.setAttribute('value', result.csrf_token);
                    }
                }

                if(result.success){
                    statusEl.innerHTML = '<div class="server-message success">'+(result.message||'Thanks!')+'</div>';
                    form.reset();
                    const tokenField = form.querySelector('input[name="csrf_token"]');
                    if(tokenField && result && result.csrf_token){
                        tokenField.value = result.csrf_token;
                        tokenField.setAttribute('value', result.csrf_token);
                    }
                } else {
                    statusEl.innerHTML = '<div class="server-message error">'+(result.message||'Please correct the errors')+'</div>';
                    if(result.errors && Array.isArray(result.errors)){
                        result.errors.forEach(function(err){
                            // attempt to attach to field errors
                            if(err.toLowerCase().includes('name')) setError('err-fname', err);
                            if(err.toLowerCase().includes('email')) setError('err-email', err);
                            if(err.toLowerCase().includes('message') || err.toLowerCase().includes('subject')) setError('err-subject', err);
                        });
                    }
                }
            });
        }
    })();
    </script>

    <?php require_once BASE_PATH . '/partial/footer_scripts.php'; ?>
