<?php 
require_once __DIR__ . '/../config.php';

// Set page-specific variables
$page_title = 'WsplCloud Blog: Expert Hosting Tips, Cloud News & Development Guides';
$page_description = 'Explore the WsplCloud blog for the latest articles, tutorials, and news on web hosting, domains, cloud technology, and website development. Stay updated with expert insights and industry trends.';
$page_keywords = 'web hosting blog, cloud technology news, hosting tips, website development tutorials, domain management, WsplCloud blog, tech articles, hosting guides, developer tips, industry trends';
$custom_css = ['blog.css'];

// Pagination settings
$posts_per_page = 9;
$current_page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($current_page - 1) * $posts_per_page;

// Include header meta
require_once BASE_PATH . '/partial/header_meta.php';
require_once BASE_PATH . '/partial/header.php';

// Fetch posts from DB
$posts = [];
$all_categories = [];
$total_posts = 0;
$total_pages = 1;

try {
    $db = get_db();
    if ($db) {
        // Get category filter
        $category_filter = isset($_GET['category']) ? trim($_GET['category']) : '';
        
        // Count total posts
        if ($category_filter) {
            $count_stmt = $db->prepare('SELECT COUNT(*) FROM blogs WHERE status = "published" AND category = :category');
            $count_stmt->execute(['category' => $category_filter]);
        } else {
            $count_stmt = $db->query('SELECT COUNT(*) FROM blogs WHERE status = "published"');
        }
        $total_posts = $count_stmt->fetchColumn();
        $total_pages = ceil($total_posts / $posts_per_page);
        
        // Fetch posts with pagination
        if ($category_filter) {
            $stmt = $db->prepare('SELECT id, uuid, title, slug, content, excerpt, category, tags, author, image, status, featured, views, created 
                                  FROM blogs 
                                  WHERE status = "published" AND category = :category 
                                  ORDER BY featured DESC, created DESC 
                                  LIMIT :limit OFFSET :offset');
            $stmt->bindValue(':category', $category_filter, PDO::PARAM_STR);
            $stmt->bindValue(':limit', $posts_per_page, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $stmt = $db->prepare('SELECT id, uuid, title, slug, content, excerpt, category, tags, author, image, status, featured, views, created 
                                  FROM blogs 
                                  WHERE status = "published" 
                                  ORDER BY featured DESC, created DESC 
                                  LIMIT :limit OFFSET :offset');
            $stmt->bindValue(':limit', $posts_per_page, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
        }
        $posts = $stmt->fetchAll();
        
        // Get all unique categories
        $cat_stmt = $db->query('SELECT DISTINCT category FROM blogs WHERE status = "published" AND category IS NOT NULL AND category != "" ORDER BY category');
        $all_categories = $cat_stmt->fetchAll(PDO::FETCH_COLUMN);
    }
} catch (Exception $e) {
    error_log('Blog fetch error: ' . $e->getMessage());
    $posts = [];
}
?>

<style>
/* Enhanced Blog Styles */
.blog-hero {
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    padding: 80px 0 60px;
    position: relative;
    overflow: hidden;
}

.blog-hero::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('data:image/svg+xml,<svg width="100" height="100" xmlns="http://www.w3.org/2000/svg"><defs><pattern id="grid" width="100" height="100" patternUnits="userSpaceOnUse"><path d="M 100 0 L 0 0 0 100" fill="none" stroke="rgba(255,255,255,0.05)" stroke-width="1"/></pattern></defs><rect width="100%" height="100%" fill="url(%23grid)"/></svg>');
    opacity: 0.3;
}

.blog-category-pills {
    display: flex;
    flex-wrap: wrap;
    gap: 12px;
    justify-content: center;
    margin: 30px 0;
}

.category-pill {
    padding: 10px 25px;
    border-radius: 25px;
    background: rgba(255, 255, 255, 0.1);
    border: 2px solid rgba(255, 255, 255, 0.2);
    color: white;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
    backdrop-filter: blur(10px);
}

.category-pill:hover,
.category-pill.active {
    background: #ffd200;
    color: #0000ff;
    border-color: #ffd200;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(255, 210, 0, 0.4);
}

.blog-grid-enhanced {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 30px;
    margin: 40px 0;
}

.blog-card-enhanced {
    background: white;
    border-radius: 20px;
    overflow: hidden;
    box-shadow: 0 5px 25px rgba(0, 0, 0, 0.08);
    transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
    position: relative;
}

.blog-card-enhanced:hover {
    transform: translateY(-10px);
    box-shadow: 0 15px 50px rgba(0, 0, 255, 0.15);
}

.blog-card-image {
    position: relative;
    height: 240px;
    overflow: hidden;
}

.blog-card-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.6s;
}

.blog-card-enhanced:hover .blog-card-image img {
    transform: scale(1.1);
}

.featured-badge-new {
    position: absolute;
    top: 15px;
    left: 15px;
    background: linear-gradient(135deg, #ffd200 0%, #ffb700 100%);
    color: #0000ff;
    padding: 8px 20px;
    border-radius: 25px;
    font-weight: 700;
    font-size: 12px;
    z-index: 10;
    box-shadow: 0 4px 15px rgba(255, 210, 0, 0.4);
    display: flex;
    align-items: center;
    gap: 5px;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

.blog-card-content {
    padding: 25px;
}

.blog-meta-row {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    margin-bottom: 15px;
    font-size: 13px;
    color: #666;
}

.blog-meta-item {
    display: flex;
    align-items: center;
    gap: 5px;
}

.blog-meta-item i {
    color: #0000ff;
}

.blog-title-enhanced {
    font-size: 1.3rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 12px;
    line-height: 1.4;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
    transition: color 0.3s;
}

.blog-card-enhanced:hover .blog-title-enhanced {
    color: #0000ff;
}

.blog-excerpt {
    color: #666;
    line-height: 1.6;
    margin-bottom: 15px;
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.blog-tags-row {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    margin-bottom: 15px;
}

.tag-badge {
    background: #f0f0f0;
    color: #333;
    padding: 5px 12px;
    border-radius: 15px;
    font-size: 11px;
    font-weight: 500;
    transition: all 0.3s;
}

.tag-badge:hover {
    background: #0000ff;
    color: white;
}

.read-more-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
    padding: 12px 25px;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
    box-shadow: 0 4px 15px rgba(0, 0, 255, 0.2);
}

.read-more-btn:hover {
    transform: translateX(5px);
    box-shadow: 0 6px 20px rgba(0, 0, 255, 0.3);
    color: white;
}

/* Pagination Styles */
.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin: 50px 0;
}

.pagination-enhanced {
    display: flex;
    gap: 10px;
    list-style: none;
    padding: 0;
}

.pagination-enhanced li a,
.pagination-enhanced li span {
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 45px;
    height: 45px;
    padding: 0 15px;
    background: white;
    border: 2px solid #e0e0e0;
    border-radius: 10px;
    color: #333;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
}

.pagination-enhanced li a:hover {
    background: #0000ff;
    color: white;
    border-color: #0000ff;
    transform: translateY(-2px);
}

.pagination-enhanced li.active span {
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
    border-color: #0000ff;
}

.pagination-enhanced li.disabled a {
    opacity: 0.4;
    pointer-events: none;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 80px 20px;
}

.empty-state i {
    font-size: 80px;
    color: #e0e0e0;
    margin-bottom: 20px;
}

.empty-state h3 {
    color: #666;
    font-size: 1.5rem;
    margin-bottom: 10px;
}

.empty-state p {
    color: #999;
}

@media (max-width: 768px) {
    .blog-grid-enhanced {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .category-pill {
        padding: 8px 18px;
        font-size: 14px;
    }
}
</style>

<!-- Blog Hero Section -->
<section class="blog-hero">
    <div class="container">
        <div class="text-center" style="position: relative; z-index: 10;">
            <h1 style="color: white; font-size: 3rem; font-weight: 800; margin-bottom: 15px;">
                Our Blog
            </h1>
            <p style="color: rgba(255,255,255,0.9); font-size: 1.2rem; max-width: 600px; margin: 0 auto 30px;">
                Stay updated with the latest hosting insights, tutorials, and industry news
            </p>
            
            <!-- Category Pills -->
            <div class="blog-category-pills">
                <a href="<?= url('blog_enhanced.php') ?>" class="category-pill <?= empty($_GET['category']) ? 'active' : ''; ?>">
                    <i class="fas fa-th-large"></i> All Posts
                </a>
                <?php foreach ($all_categories as $cat): ?>
                    <a href="<?= url('blog_enhanced.php?category=' . urlencode($cat)) ?>" 
                       class="category-pill <?= (isset($_GET['category']) && $_GET['category'] === $cat) ? 'active' : ''; ?>">
                        <i class="fas fa-folder"></i> <?= htmlspecialchars($cat) ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</section>

<!-- Blog Grid Section -->
<section class="padding-top padding-bottom" style="background: #f8f9fa;">
    <div class="container">
        <?php if (!empty($posts)): ?>
            <div class="blog-grid-enhanced">
                <?php foreach($posts as $post): ?>
                    <article class="blog-card-enhanced">
                        <?php if (!empty($post['featured']) && $post['featured'] == 1): ?>
                            <div class="featured-badge-new">
                                <i class="fas fa-star"></i> FEATURED
                            </div>
                        <?php endif; ?>
                        
                        <div class="blog-card-image">
                            <?php if (!empty($post['image'])): ?>
                                <img src="<?= htmlspecialchars(BASE_URL . '/' . $post['image']) ?>" 
                                     alt="<?= htmlspecialchars($post['title']) ?>" 
                                     loading="lazy">
                            <?php else: ?>
                                <div style="width: 100%; height: 100%; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; align-items: center; justify-content: center;">
                                    <i class="fas fa-image" style="font-size: 60px; color: rgba(255,255,255,0.3);"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="blog-card-content">
                            <div class="blog-meta-row">
                                <span class="blog-meta-item">
                                    <i class="fas fa-user"></i>
                                    <?= htmlspecialchars(!empty($post['author']) ? $post['author'] : 'Admin') ?>
                                </span>
                                <span class="blog-meta-item">
                                    <i class="fas fa-calendar-alt"></i>
                                    <?= htmlspecialchars(date('M j, Y', strtotime($post['created'] ?? 'now'))) ?>
                                </span>
                                <?php if (!empty($post['views'])): ?>
                                    <span class="blog-meta-item">
                                        <i class="fas fa-eye"></i>
                                        <?= number_format($post['views']) ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                            
                            <h2 class="blog-title-enhanced">
                                <?= htmlspecialchars($post['title']) ?>
                            </h2>
                            
                            <p class="blog-excerpt">
                                <?php 
                                    $display_text = !empty($post['excerpt']) ? $post['excerpt'] : substr(strip_tags($post['content']), 0, 150);
                                    echo htmlspecialchars($display_text);
                                    if (strlen($display_text) >= 150 && empty($post['excerpt'])) echo '...';
                                ?>
                            </p>
                            
                            <?php if (!empty($post['tags'])): ?>
                                <div class="blog-tags-row">
                                    <?php 
                                        $tags = array_map('trim', explode(',', $post['tags']));
                                        $tag_count = 0;
                                        foreach ($tags as $tag): 
                                            if (!empty($tag) && $tag_count < 3):
                                                $tag_count++;
                                    ?>
                                        <span class="tag-badge">
                                            <i class="fas fa-tag"></i> <?= htmlspecialchars($tag) ?>
                                        </span>
                                    <?php 
                                            endif;
                                        endforeach; 
                                    ?>
                                </div>
                            <?php endif; ?>
                            
                            <a href="<?= url('single-blog.php?slug=' . ($post['slug'] ?? $post['id'])) ?>" class="read-more-btn">
                                Read More <i class="fas fa-arrow-right"></i>
                            </a>
                        </div>
                    </article>
                <?php endforeach; ?>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="pagination-wrapper">
                    <ul class="pagination-enhanced">
                        <!-- Previous Button -->
                        <li class="<?= $current_page <= 1 ? 'disabled' : '' ?>">
                            <a href="<?= $current_page > 1 ? url('blog_enhanced.php?page=' . ($current_page - 1) . (!empty($_GET['category']) ? '&category=' . urlencode($_GET['category']) : '')) : '#' ?>">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        </li>
                        
                        <!-- Page Numbers -->
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <?php if ($i == 1 || $i == $total_pages || abs($i - $current_page) <= 2): ?>
                                <li class="<?= $i == $current_page ? 'active' : '' ?>">
                                    <?php if ($i == $current_page): ?>
                                        <span><?= $i ?></span>
                                    <?php else: ?>
                                        <a href="<?= url('blog_enhanced.php?page=' . $i . (!empty($_GET['category']) ? '&category=' . urlencode($_GET['category']) : '')) ?>">
                                            <?= $i ?>
                                        </a>
                                    <?php endif; ?>
                                </li>
                            <?php elseif (abs($i - $current_page) == 3): ?>
                                <li class="disabled"><span>...</span></li>
                            <?php endif; ?>
                        <?php endfor; ?>
                        
                        <!-- Next Button -->
                        <li class="<?= $current_page >= $total_pages ? 'disabled' : '' ?>">
                            <a href="<?= $current_page < $total_pages ? url('blog_enhanced.php?page=' . ($current_page + 1) . (!empty($_GET['category']) ? '&category=' . urlencode($_GET['category']) : '')) : '#' ?>">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                    </ul>
                </div>
            <?php endif; ?>
            
        <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-folder-open"></i>
                <h3>No Posts Found</h3>
                <p><?= !empty($_GET['category']) ? 'No posts in this category yet.' : 'No blog posts available at the moment.' ?></p>
                <?php if (!empty($_GET['category'])): ?>
                    <a href="<?= url('blog_enhanced.php') ?>" class="read-more-btn" style="margin-top: 20px;">
                        <i class="fas fa-arrow-left"></i> View All Posts
                    </a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</section>

<!-- Include footer -->
<?php require_once BASE_PATH . '/partial/footer.php'; ?>

<!-- Include footer scripts -->
<?php 
$page_scripts = [];
require_once BASE_PATH . '/partial/footer_scripts.php'; 
?>
