<?php
require_once __DIR__ . '/../config.php';

// Set page-specific variables
$page_title = 'WsplCloud Blog: Expert Hosting Tips, Cloud News & Development Guides';
$page_description = 'Explore the WsplCloud blog for the latest articles, tutorials, and news on web hosting, domains, cloud technology, and website development.';
$page_keywords = 'web hosting blog, cloud technology news, hosting tips, website development tutorials, domain management, WsplCloud blog';
$custom_css = ['blog.css'];

// Include header meta
require_once BASE_PATH . '/partial/header_meta.php';
require_once BASE_PATH . '/partial/header.php';

// Get filter parameters
$current_category = isset($_GET['category']) ? trim($_GET['category']) : '';
$search_query = isset($_GET['search']) ? trim($_GET['search']) : '';

// Fetch posts from database
$posts = [];
$all_categories = [];

try {
    $db = get_db();
    if ($db) {
        // Build query with filters
        $where_conditions = ["status = 'published'"];
        $params = [];
        
        if ($current_category) {
            $where_conditions[] = "category = :category";
            $params[':category'] = $current_category;
        }
        
        if ($search_query) {
            $where_conditions[] = "(title LIKE :search OR content LIKE :search OR excerpt LIKE :search OR tags LIKE :search)";
            $params[':search'] = '%' . $search_query . '%';
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        $stmt = $db->prepare("SELECT id, uuid, title, slug, content, excerpt, category, tags, author, image, status, featured, views, created 
                              FROM blogs 
                              WHERE $where_clause
                              ORDER BY featured DESC, created DESC");
        $stmt->execute($params);
        $posts = $stmt->fetchAll();
        
        // Get all unique categories with post counts
        $cat_stmt = $db->query('SELECT DISTINCT category FROM blogs WHERE status = "published" AND category IS NOT NULL AND category != "" ORDER BY category');
        $all_categories = $cat_stmt->fetchAll(PDO::FETCH_COLUMN);
    }
} catch (Exception $e) { 
    $posts = []; 
    error_log("Blog page error: " . $e->getMessage());
}

// Helper function for category icons
function get_category_icon($category) {
    $icons = [
        'Advices' => 'lightbulb',
        'Announcements' => 'bullhorn',
        'News' => 'newspaper',
        'Consultation' => 'comments',
        'Development' => 'code',
        'General' => 'folder'
    ];
    return $icons[$category] ?? 'folder';
}
?>


<!-- BANNER SECTION START HERE -->
<section class="sub-banner-main-con w-100 float-left">
    <div class="container">
        <div class="sub-banner-inner-con wow fadeInUp" data-wow-duration="1s" data-wow-delay="0.3s">
            <h1>Blog</h1>
            <span class="d-block position-relative"></span>
            <h6>
                <a href="<?= url('') ?>" style="color: #fff; transition: color 0.2s;" onmouseover="this.style.color='#FFD700'" onmouseout="this.style.color='#fff'">Home</a>
                <span style="color: #fff;"> | </span>
                <a href="<?= url('blog') ?>" style="color: #fff; transition: color 0.2s;" onmouseover="this.style.color='#FFD700'" onmouseout="this.style.color='#fff'">Blog</a>
            </h6>
        </div>
    </div>
</section>
<!-- BANNER SECTION END HERE -->

<!-- Blog Main Section -->
<main class="blog-main-section padding-top padding-bottom">
    <div class="container">
        
        <!-- Search and Filter Section -->
        <div class="blog-search-filter-section">
            <div class="search-filter-container">
                <div class="row align-items-end">
                    <div class="col-lg-6 col-md-6 mb-3">
                        <div class="search-box-wrapper">
                            <label for="blogSearch" class="filter-label">
                                <i class="fas fa-search"></i> Search Articles
                            </label>
                            <form action="<?= url('blog') ?>" method="GET" class="search-form">
                                <?php if ($current_category): ?>
                                    <input type="hidden" name="category" value="<?= htmlspecialchars($current_category) ?>">
                                <?php endif; ?>
                                <div class="search-input-group">
                                    <input type="text" 
                                           id="blogSearch" 
                                           name="search" 
                                           class="search-input" 
                                           placeholder="Search by title, content, or tags..."
                                           value="<?= htmlspecialchars($search_query) ?>">
                                    <button type="submit" class="search-btn">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <div class="col-lg-6 col-md-6 mb-3">
                        <div class="category-filter-wrapper">
                            <label class="filter-label">
                                <i class="fas fa-filter"></i> Filter by Category
                            </label>
                            <div class="category-pills">
                                <a href="<?= url('blog' . ($search_query ? '?search=' . urlencode($search_query) : '')) ?>" 
                                   class="category-pill <?= empty($current_category) ? 'active' : '' ?>">
                                    <i class="fas fa-th-large"></i>
                                    All
                                </a>
                                
                                <?php foreach($all_categories as $category): ?>
                                    <a href="<?= url('blog?category=' . urlencode($category) . ($search_query ? '&search=' . urlencode($search_query) : '')) ?>" 
                                       class="category-pill <?= ($current_category === $category) ? 'active' : '' ?>">
                                        <i class="fas fa-<?= get_category_icon($category) ?>"></i>
                                        <?= htmlspecialchars($category) ?>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php if ($search_query || $current_category): ?>
                    <div class="active-filters">
                        <span class="filter-label">Active Filters:</span>
                        <?php if ($search_query): ?>
                            <span class="active-filter-tag">
                                Search: "<?= htmlspecialchars($search_query) ?>"
                                <a href="<?= url('blog' . ($current_category ? '?category=' . urlencode($current_category) : '')) ?>" class="remove-filter">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        <?php endif; ?>
                        <?php if ($current_category): ?>
                            <span class="active-filter-tag">
                                Category: <?= htmlspecialchars($current_category) ?>
                                <a href="<?= url('blog' . ($search_query ? '?search=' . urlencode($search_query) : '')) ?>" class="remove-filter">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        <?php endif; ?>
                        <a href="<?= url('blog') ?>" class="clear-all-filters">
                            <i class="fas fa-times-circle"></i> Clear All
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
      
        <!-- Blog Posts Grid -->
        <div class="blog-posts-section">
            <?php if (!empty($posts)): ?>
                <div class="section-header">
                    <h2>
                        <?php if ($search_query): ?>
                            Search Results (<?= count($posts) ?>)
                        <?php elseif ($current_category): ?>
                            <?= htmlspecialchars($current_category) ?> Articles (<?= count($posts) ?>)
                        <?php else: ?>
                            Latest Articles
                        <?php endif; ?>
                    </h2>
                    <p>
                        <?php if ($search_query): ?>
                            Showing results for "<?= htmlspecialchars($search_query) ?>"
                        <?php else: ?>
                            Discover expert insights and technical guides
                        <?php endif; ?>
                    </p>
                </div>
                
                <div class="blog-posts-grid">
                    <?php foreach($posts as $post): ?>
                        <article class="blog-post-card">
                            <?php if (!empty($post['featured'])): ?>
                                <div class="featured-badge">
                                    <i class="fas fa-star"></i>
                                    Featured
                                </div>
                            <?php endif; ?>
                            
                            <div class="post-image-container">
                                <?php if (!empty($post['image'])): ?>
                                    <img src="<?= htmlspecialchars(BASE_URL . '/' . $post['image']) ?>" 
                                         alt="<?= htmlspecialchars($post['title']) ?>" 
                                         class="post-image" 
                                         loading="lazy">
                                <?php else: ?>
                                    <div class="post-image-placeholder">
                                        <i class="fas fa-image"></i>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($post['category'])): ?>
                                    <div class="post-category-badge">
                                        <i class="fas fa-<?= get_category_icon($post['category']) ?>"></i>
                                        <?= htmlspecialchars($post['category']) ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="post-content">
                                <div class="post-meta">
                                    <div class="meta-item">
                                        <i class="fas fa-user"></i>
                                        <span><?= htmlspecialchars($post['author'] ?? 'Admin') ?></span>
                                    </div>
                                    <div class="meta-item">
                                        <i class="fas fa-calendar-alt"></i>
                                        <span><?= date('M j, Y', strtotime($post['created'] ?? 'now')) ?></span>
                                    </div>
                                    <?php if (!empty($post['views'])): ?>
                                        <div class="meta-item">
                                            <i class="fas fa-eye"></i>
                                            <span><?= number_format($post['views']) ?></span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <h3 class="post-title">
                                    <a href="<?= url('single-blog.php?slug=' . ($post['slug'] ?? $post['id'])) ?>">
                                        <?= htmlspecialchars($post['title']) ?>
                                    </a>
                                </h3>
                                
                                <p class="post-excerpt">
                                    <?php 
                                        $display_text = !empty($post['excerpt']) ? $post['excerpt'] : substr(strip_tags($post['content']), 0, 150);
                                        echo htmlspecialchars($display_text);
                                        if (strlen($display_text) >= 150 && empty($post['excerpt'])) echo '...';
                                    ?>
                                </p>
                                
                                <?php if (!empty($post['tags'])): ?>
                                    <div class="post-tags">
                                        <?php 
                                            $tags = array_map('trim', explode(',', $post['tags']));
                                            foreach (array_slice($tags, 0, 3) as $tag): 
                                                if (!empty($tag)):
                                        ?>
                                            <span class="post-tag">
                                                <i class="fas fa-tag"></i> <?= htmlspecialchars($tag) ?>
                                            </span>
                                        <?php 
                                                endif;
                                            endforeach; 
                                        ?>
                                    </div>
                                <?php endif; ?>
                                
                                <a href="<?= url('single-blog.php?slug=' . ($post['slug'] ?? $post['id'])) ?>" class="read-more-btn">
                                    Read More
                                    <i class="fas fa-arrow-right"></i>
                                </a>
                            </div>
                        </article>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-icon">
                        <i class="fas fa-search"></i>
                    </div>
                    <h3>No Posts Found</h3>
                    <p>
                        <?php if ($search_query): ?>
                            No articles found matching "<?= htmlspecialchars($search_query) ?>". Try different keywords.
                        <?php elseif ($current_category): ?>
                            No articles found in the <?= htmlspecialchars($current_category) ?> category yet.
                        <?php else: ?>
                            No blog posts available at the moment.
                        <?php endif; ?>
                    </p>
                    <a href="<?= url('blog') ?>" class="btn-back-home">
                        <i class="fas fa-arrow-left"></i>
                        View All Posts
                    </a>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Newsletter Section -->
        <div class="blog-newsletter-section">
            <div class="newsletter-container">
                <div class="newsletter-content">
                    <h3 style="color:#ffd700">Stay Updated</h3>
                    <p>Get the latest hosting tips and cloud insights delivered to your inbox</p>
                    <form id="newsletterForm" class="newsletter-form">
                        <div id="newsletterMessage" class="newsletter-message" style="display: none;"></div>
                        <div class="input-group">
                            <input type="email" 
                                   id="newsletterEmail" 
                                   name="email" 
                                   placeholder="Enter your email" 
                                   class="newsletter-input" 
                                   required>
                            <button type="submit" class="newsletter-btn" id="newsletterBtn">
                                <span class="btn-text">Subscribe</span>
                                <i class="fas fa-paper-plane"></i>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Include footer -->
<?php require_once BASE_PATH . '/partial/footer.php'; ?>

<!-- Include footer scripts -->
<?php 
// Add page-specific scripts
$page_scripts = ['blog.js'];
require_once BASE_PATH . '/partial/footer_scripts.php'; 
?>

<style>
/* Blog Main Section */
.blog-main-section {
    background: #f8f9fa;
}

/* Search and Filter Section */
.blog-search-filter-section {
    margin-bottom: 40px;
    margin-top: 380px;
    transition: margin 0.3s ease, padding 0.3s ease;
}

/* Gradually reduce top spacing on smaller screens */
@media (max-width: 1200px) {
    .blog-search-filter-section {
        margin-top: 280px;
    }
}

@media (max-width: 992px) {
    .blog-search-filter-section {
        margin-top: 280px;
    }
}

@media (max-width: 768px) {
    .blog-search-filter-section {
        margin-top: 270px;
        padding: 0 15px;
    }

    /* make the search/filter controls more compact on tablets/phones */
    .search-filter-container {
        padding: 20px;
    }

    .search-input {
        padding: 10px 45px 10px 12px;
        font-size: 0.95rem;
    }

    .search-btn {
        padding: 7px 12px;
    }

    .category-pill {
        padding: 8px 14px;
        font-size: 0.9rem;
    }
}

@media (max-width: 480px) {
    .blog-search-filter-section {
        margin-top: 160px;
        padding: 0 12px;
    }

    .search-filter-container {
        padding: 16px;
    }

    .search-input {
        padding: 10px 42px 10px 10px;
        font-size: 0.9rem;
    }

    .search-btn {
        padding: 6px 10px;
    }

    .category-pills {
        gap: 8px;
        flex-direction: column;
        align-items: stretch;
    }

    .category-pill {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 0.9rem;
    }

    .active-filters {
        flex-direction: column;
        align-items: stretch;
        gap: 8px;
    }
}

.search-filter-container {
    
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.08);
}

.filter-label {
    display: block;
    font-weight: 600;
    color: #333;
    margin-bottom: 10px;
    font-size: 0.95rem;
}

.filter-label i {
    color: #0000ff;
    margin-right: 5px;
}

.search-form {
    width: 100%;
}

.search-input-group {
    position: relative;
    display: flex;
}

.search-input {
    width: 100%;
    padding: 12px 50px 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 10px;
    font-size: 1rem;
    transition: all 0.3s;
}

.search-input:focus {
    outline: none;
    border-color: #0000ff;
    box-shadow: 0 0 0 3px rgba(0,0,255,0.1);
}

.search-btn {
    position: absolute;
    right: 5px;
    top: 50%;
    transform: translateY(-50%);
    background: #0000ff;
    color: white;
    border: none;
    padding: 8px 15px;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s;
}

.search-btn:hover {
    background: #0000cc;
    transform: translateY(-50%) scale(1.05);
}

.category-pills {
    display: flex;
    flex-wrap: wrap;
    gap: 10px;
}

.category-pill {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    padding: 10px 20px;
    background: #f0f0f0;
    border: 2px solid transparent;
    border-radius: 25px;
    color: #333;
    text-decoration: none;
    font-weight: 500;
    font-size: 0.9rem;
    transition: all 0.3s;
}

.category-pill:hover {
    background: #e0e0e0;
    transform: translateY(-2px);
    color: #333;
}

.category-pill.active {
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
    border-color: #0000ff;
}

.category-pill i {
    font-size: 0.9rem;
}

.active-filters {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #e0e0e0;
    display: flex;
    align-items: center;
    flex-wrap: wrap;
    gap: 10px;
}

.active-filter-tag {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: #fff3cd;
    border: 1px solid #ffc107;
    color: #856404;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.85rem;
}

.remove-filter {
    color: #856404;
    text-decoration: none;
    font-weight: bold;
    transition: color 0.3s;
}

.remove-filter:hover {
    color: #ff0000;
}

.clear-all-filters {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
    padding: 6px 15px;
    border-radius: 20px;
    text-decoration: none;
    font-size: 0.85rem;
    font-weight: 500;
    transition: all 0.3s;
}

.clear-all-filters:hover {
    background: #f5c6cb;
    color: #721c24;
}

/* Section Header */
.section-header {
    text-align: center;
    margin-bottom: 40px;
}

.section-header h2 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #0000ff;
    margin-bottom: 10px;
}

.section-header p {
    font-size: 1.1rem;
    color: #666;
}

/* Blog Posts Grid */
.blog-posts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 30px;
}

.blog-post-card {
    background: white;
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 5px 20px rgba(0,0,0,0.08);
    transition: all 0.4s;
    position: relative;
}

.blog-post-card:hover {
    transform: translateY(-10px);
    box-shadow: 0 15px 40px rgba(0,0,255,0.15);
}

.featured-badge {
    position: absolute;
    top: 15px;
    left: 15px;
    background: linear-gradient(135deg, #ffd200 0%, #ffb700 100%);
    color: #0000ff;
    padding: 6px 15px;
    border-radius: 20px;
    font-weight: 700;
    font-size: 0.75rem;
    z-index: 10;
    display: flex;
    align-items: center;
    gap: 5px;
    box-shadow: 0 4px 15px rgba(255,210,0,0.4);
}

.post-image-container {
    position: relative;
    height: 220px;
    overflow: hidden;
}

.post-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.6s;
}

.blog-post-card:hover .post-image {
    transform: scale(1.1);
}

.post-image-placeholder {
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: rgba(255,255,255,0.5);
    font-size: 3rem;
}

.post-category-badge {
    position: absolute;
    bottom: 15px;
    right: 15px;
    background: rgba(0,0,0,0.7);
    backdrop-filter: blur(10px);
    color: white;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 5px;
}

.post-content {
    padding: 25px;
}

.post-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    margin-bottom: 15px;
    font-size: 0.85rem;
    color: #666;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 5px;
}

.meta-item i {
    color: #0000ff;
}

.post-title {
    font-size: 1.4rem;
    font-weight: 700;
    margin-bottom: 12px;
    line-height: 1.4;
}

.post-title a {
    color: #2c3e50;
    text-decoration: none;
    transition: color 0.3s;
}

.post-title a:hover {
    color: #0000ff;
}

.post-excerpt {
    color: #666;
    line-height: 1.6;
    margin-bottom: 15px;
}

.post-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    margin-bottom: 20px;
}

.post-tag {
    background: #f0f0f0;
    color: #333;
    padding: 5px 12px;
    border-radius: 15px;
    font-size: 0.75rem;
    font-weight: 500;
    display: inline-flex;
    align-items: center;
    gap: 4px;
    transition: all 0.3s;
}

.post-tag:hover {
    background: #0000ff;
    color: white;
}

.read-more-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
    padding: 10px 25px;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
}

.read-more-btn:hover {
    transform: translateX(5px);
    box-shadow: 0 5px 15px rgba(0,0,255,0.3);
    color: white;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 80px 20px;
    background: white;
    border-radius: 15px;
}

.empty-icon {
    font-size: 5rem;
    color: #e0e0e0;
    margin-bottom: 20px;
}

.empty-state h3 {
    font-size: 1.8rem;
    color: #666;
    margin-bottom: 10px;
}

.empty-state p {
    color: #999;
    font-size: 1.1rem;
    margin-bottom: 30px;
}

.btn-back-home {
    display: inline-flex;
    align-items: center;
    gap: 10px;
    background: #0000ff;
    color: white;
    padding: 12px 30px;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
}

.btn-back-home:hover {
    background: #0000cc;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,255,0.3);
    color: white;
}

/* Newsletter */
.blog-newsletter-section {
    margin: 60px 0;
}

.newsletter-container {
    background: linear-gradient(135deg, #0000ff 0%, #0303dd 100%);
    border-radius: 25px;
    padding: 50px;
    text-align: center;
    color: white;
}

.newsletter-content h3 {
    font-size: 2rem;
    margin-bottom: 10px;
}

.newsletter-content p {
    opacity: 0.9;
    margin-bottom: 25px;
    font-size: 1.1rem;
}

.newsletter-form {
    max-width: 500px;
    margin: 0 auto;
}

.input-group {
    display: flex;
    gap: 10px;
}

.newsletter-input {
    flex: 1;
    padding: 15px 20px;
    border: none;
    border-radius: 25px;
    font-size: 1rem;
    outline: none;
}

.newsletter-btn {
    background: #ffd700;
    color: #000;
    border: none;
    padding: 15px 30px;
    border-radius: 25px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
}

.newsletter-btn:hover {
    background: #ffed4a;
    transform: translateY(-2px);
}

/* Responsive Design */
@media (max-width: 992px) {
    .blog-posts-grid {
        grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
        gap: 20px;
    }
}

@media (max-width: 768px) {
    .category-pills {
        flex-direction: column;
    }
    
    .category-pill {
        width: 100%;
        justify-content: center;
    }
    
    .blog-posts-grid {
        grid-template-columns: 1fr;
    }
    
    .section-header h2 {
        font-size: 2rem;
    }
    
    .search-filter-container {
        padding: 20px;
    }
    
    .input-group {
        flex-direction: column;
    }
    
    .newsletter-container {
        padding: 30px 20px;
    }
}

@media (max-width: 480px) {
    .post-meta {
        flex-direction: column;
        gap: 8px;
    }
    
    .active-filters {
        flex-direction: column;
        align-items: flex-start;
    }
}



.read-more-btn {
    color: white;
    text-decoration: none;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 25px;
    transition: all 0.3s ease;
}

.read-more-btn:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-2px);
}

.post-content {
    padding: 25px;
}

.post-meta {
    display: flex;
    gap: 15px;
    margin-bottom: 15px;
    flex-wrap: wrap;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 5px;
    font-size: 0.85rem;
    color: #666;
}

.meta-item i {
    color: #0000ff;
}

.post-title {
    font-size: 1.3rem;
    font-weight: 600;
    margin-bottom: 12px;
    line-height: 1.4;
}

.post-title a {
    color: #333;
    text-decoration: none;
    transition: color 0.3s;
}

.post-title a:hover {
    color: #0000ff;
}

.post-excerpt {
    color: #666;
    line-height: 1.6;
    margin-bottom: 15px;
}

.post-category {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: #f0f4ff;
    color: #0000ff;
    padding: 5px 12px;
    border-radius: 15px;
    font-size: 0.8rem;
    font-weight: 500;
    margin-bottom: 15px;
}

.post-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    margin-bottom: 20px;
}

.post-tag {
    background: #f8f9fa;
    color: #666;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 0.75rem;
    border: 1px solid #e9ecef;
}

.more-tags {
    background: transparent;
    border: 1px dashed #ccc;
}

.post-footer {
    border-top: 1px solid #f0f0f0;
    padding-top: 15px;
}

.post-read-more {
    color: #0000ff;
    text-decoration: none;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: all 0.3s;
}

.post-read-more:hover {
    gap: 12px;
    color: #0303dd;
}

/* Newsletter */
.blog-newsletter-section {
    margin: 60px 0;
}

.newsletter-container {
    background: linear-gradient(135deg, #0000ff 0%, #0303dd 100%);
    border-radius: 25px;
    padding: 50px;
    text-align: center;
    color: white;
    position: relative;
    overflow: hidden;
}

.newsletter-container::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: pulseBg 3s ease-in-out infinite;
}

@keyframes pulseBg {
    0%, 100% { opacity: 0.5; }
    50% { opacity: 1; }
}

.newsletter-content {
    position: relative;
    z-index: 1;
}

.newsletter-content h3 {
    font-size: 2rem;
    margin-bottom: 10px;
    text-shadow: 0 2px 10px rgba(0,0,0,0.2);
}

.newsletter-content p {
    opacity: 0.9;
    margin-bottom: 25px;
    font-size: 1.1rem;
}

.newsletter-form {
    max-width: 500px;
    margin: 0 auto;
}

.newsletter-message {
    padding: 12px 20px;
    border-radius: 10px;
    margin-bottom: 15px;
    font-weight: 500;
    animation: slideDown 0.3s ease;
}

.newsletter-message.success {
    background: rgba(40, 167, 69, 0.9);
    color: white;
}

.newsletter-message.error {
    background: rgba(220, 53, 69, 0.9);
    color: white;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.input-group {
    display: flex;
    gap: 10px;
    position: relative;
}

.newsletter-input {
    flex: 1;
    padding: 15px 20px;
    border: 2px solid rgba(255,255,255,0.3);
    border-radius: 25px;
    font-size: 1rem;
    outline: none;
    background: rgba(255,255,255,0.95);
    transition: all 0.3s;
}

.newsletter-input:focus {
    border-color: #ffd700;
    background: white;
    box-shadow: 0 0 0 3px rgba(255,215,0,0.2);
}

.newsletter-input.error {
    border-color: #ff4444;
    animation: shake 0.5s;
}

@keyframes shake {
    0%, 100% { transform: translateX(0); }
    25% { transform: translateX(-5px); }
    75% { transform: translateX(5px); }
}

.newsletter-btn {
    background: #ffd700;
    color: #000;
    border: none;
    padding: 15px 30px;
    border-radius: 25px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    gap: 8px;
    white-space: nowrap;
}

.newsletter-btn:hover:not(:disabled) {
    background: #ffed4a;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(255,215,0,0.4);
}

.newsletter-btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.newsletter-btn i {
    transition: transform 0.3s;
}

.newsletter-btn:hover:not(:disabled) i {
    transform: translateX(3px);
}

.newsletter-btn.loading i {
    animation: spinIcon 1s linear infinite;
}

@keyframes spinIcon {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Empty State */
.blog-empty-state {
    text-align: center;
    padding: 80px 20px;
}

.empty-state-icon {
    font-size: 4rem;
    color: #ccc;
    margin-bottom: 20px;
}

.blog-empty-state h3 {
    font-size: 1.8rem;
    color: #666;
    margin-bottom: 10px;
}

.blog-empty-state p {
    color: #888;
    margin-bottom: 30px;
    font-size: 1.1rem;
}

.btn-primary {
    background: #0000ff;
    color: white;
    padding: 12px 25px;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: all 0.3s;
}

.btn-primary:hover {
    background: #0303dd;
    transform: translateY(-2px);
    color: white;
}

/* Animations */
@keyframes float {
    0%, 100% {
        transform: translateY(0);
    }
    50% {
        transform: translateY(-10px);
    }
}

/* Responsive Design */
@media (max-width: 1024px) {
    .blog-hero-content {
        grid-template-columns: 1fr;
        text-align: center;
        gap: 40px;
    }
    
    .hero-text-content {
        max-width: 100%;
        margin: 0 auto;
    }
    
    .hero-title {
        font-size: 3rem;
    }
}

@media (max-width: 768px) {
    .blog-hero-banner {
        padding: 80px 0;
    }
    
    .hero-title {
        font-size: 2.5rem;
    }
    
    .hero-stats {
        justify-content: center;
        gap: 30px;
    }
    
    .category-filter-header {
        flex-direction: column;
        gap: 15px;
        text-align: center;
    }
    
    .filter-toggle {
        display: inline-flex;
        align-items: center;
        gap: 8px;
    }
    
    .category-filter-grid {
        grid-template-columns: 1fr;
        display: none;
    }
    
    .category-filter-grid.active {
        display: grid;
    }
    
    .blog-posts-grid {
        grid-template-columns: 1fr;
    }
    
    .input-group {
        flex-direction: column;
    }
    
    .newsletter-container {
        padding: 30px 20px;
    }
    
    .section-header h2 {
        font-size: 2rem;
    }
}

@media (max-width: 480px) {
    .blog-hero-banner {
        padding: 60px 0;
    }
    
    .hero-title {
        font-size: 2rem;
    }
    
    .hero-subtitle {
        font-size: 1.1rem;
    }
    
    .hero-stats {
        flex-direction: column;
        gap: 20px;
    }
    
    .stat-number {
        font-size: 2rem;
    }
    
    .post-meta {
        flex-direction: column;
        gap: 8px;
    }
    
    .hero-main-icon {
        font-size: 6rem;
    }
    
    .floating-element {
        width: 50px;
        height: 50px;
        font-size: 1.2rem;
    }
}
</style>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Smooth scroll animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);
    
    // Observe blog cards
    document.querySelectorAll('.blog-post-card').forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(30px)';
        card.style.transition = `opacity 0.6s ease ${index * 0.1}s, transform 0.6s ease ${index * 0.1}s`;
        observer.observe(card);
    });
    
    // Newsletter subscription
    const newsletterForm = document.getElementById('newsletterForm');
    const newsletterEmail = document.getElementById('newsletterEmail');
    const newsletterBtn = document.getElementById('newsletterBtn');
    const newsletterMessage = document.getElementById('newsletterMessage');
    
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const email = newsletterEmail.value.trim();
            
            // Basic validation
            if (!email) {
                showMessage('Please enter your email address', 'error');
                newsletterEmail.classList.add('error');
                setTimeout(() => newsletterEmail.classList.remove('error'), 500);
                return;
            }
            
            if (!isValidEmail(email)) {
                showMessage('Please enter a valid email address', 'error');
                newsletterEmail.classList.add('error');
                setTimeout(() => newsletterEmail.classList.remove('error'), 500);
                return;
            }
            
            // Disable button and show loading
            newsletterBtn.disabled = true;
            newsletterBtn.classList.add('loading');
            const btnText = newsletterBtn.querySelector('.btn-text');
            const originalText = btnText.textContent;
            btnText.textContent = 'Subscribing...';
            
            // Send AJAX request
            const formData = new FormData();
            formData.append('email', email);
            
            fetch('<?= BASE_URL ?>/newsletter_subscribe.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message, 'success');
                    newsletterForm.reset();
                    
                    // Optional: Fire analytics event
                    if (typeof gtag !== 'undefined') {
                        gtag('event', 'newsletter_subscribe', {
                            'event_category': 'engagement',
                            'event_label': 'blog_page'
                        });
                    }
                } else {
                    showMessage(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Newsletter subscription error:', error);
                showMessage('Unable to process your request. Please try again later.', 'error');
            })
            .finally(() => {
                newsletterBtn.disabled = false;
                newsletterBtn.classList.remove('loading');
                btnText.textContent = originalText;
            });
        });
    }
    
    function showMessage(message, type) {
        newsletterMessage.textContent = message;
        newsletterMessage.className = 'newsletter-message ' + type;
        newsletterMessage.style.display = 'block';
        
        // Auto-hide success messages after 5 seconds
        if (type === 'success') {
            setTimeout(() => {
                newsletterMessage.style.display = 'none';
            }, 5000);
        }
    }
    
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
});
</script>