<?php
/**
 * Email Campaign System
 * Send newsletters to subscribers
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/email_helper.php';

$page_title = 'Send Newsletter Campaign';

// Handle campaign sending
$campaign_sent = false;
$campaign_stats = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_campaign'])) {
    $subject = trim($_POST['subject'] ?? '');
    $message = $_POST['message'] ?? '';
    $status_filter = $_POST['status_filter'] ?? 'active';
    
    if (empty($subject) || empty($message)) {
        $_SESSION['error_message'] = 'Subject and message are required';
    } else {
        try {
            // Initialize email helper
            $emailHelper = new EmailHelper($db);
            
            // Get subscribers
            $query = "SELECT email, id FROM newsletter_subscribers WHERE status = :status";
            $stmt = $db->prepare($query);
            $stmt->execute(['status' => $status_filter]);
            $subscribers = $stmt->fetchAll();
            
            if (count($subscribers) === 0) {
                $_SESSION['error_message'] = 'No subscribers found with status: ' . htmlspecialchars($status_filter);
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
            
            $sent_count = 0;
            $failed_count = 0;
            $failed_emails = [];
            
            // Send to each subscriber
            foreach ($subscribers as $subscriber) {
                $unsubscribe_link = BASE_URL . '/unsubscribe.php?email=' . urlencode($subscriber['email']) . '&token=' . md5($subscriber['email'] . 'wsplcloud_secret');
                
                // Wrap message in template
                $html_email = $emailHelper->wrapInTemplate($message, $unsubscribe_link);
                
                // Send email using configured method
                if ($emailHelper->send($subscriber['email'], $subject, $html_email)) {
                    $sent_count++;
                } else {
                    $failed_count++;
                    $failed_emails[] = $subscriber['email'];
                }
                
                // Small delay to avoid being flagged as spam
                usleep(100000); // 0.1 second
            }
            
            // Log campaign
            $log_stmt = $db->prepare("INSERT INTO campaign_logs (subject, message, sent_count, failed_count, sent_at) VALUES (:subject, :message, :sent, :failed, NOW())");
            $log_stmt->execute([
                'subject' => $subject,
                'message' => strip_tags($message),
                'sent' => $sent_count,
                'failed' => $failed_count
            ]);
            
            $campaign_sent = true;
            $campaign_stats = [
                'sent' => $sent_count,
                'failed' => $failed_count,
                'failed_emails' => $failed_emails
            ];
            
            $_SESSION['success_message'] = "Campaign sent successfully! Delivered to {$sent_count} subscribers.";
            
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error sending campaign: ' . $e->getMessage();
            error_log("Campaign error: " . $e->getMessage());
        }
    }
}

// Create campaign logs table if not exists
try {
    $db->exec("CREATE TABLE IF NOT EXISTS campaign_logs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        subject VARCHAR(255) NOT NULL,
        message TEXT,
        sent_count INT DEFAULT 0,
        failed_count INT DEFAULT 0,
        sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_sent_at (sent_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
} catch (Exception $e) {
    error_log("Error creating campaign_logs table: " . $e->getMessage());
}

// Get subscriber counts
$active_count = $db->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE status = 'active'")->fetchColumn();

// Get recent campaigns
$recent_campaigns = $db->query("SELECT * FROM campaign_logs ORDER BY sent_at DESC LIMIT 10")->fetchAll();

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><i class="fas fa-paper-plane"></i> Send Newsletter Campaign</h4>
                </div>
                <div class="card-body">
                    <?php if (isset($_SESSION['success_message'])): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <?= htmlspecialchars($_SESSION['success_message']) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php unset($_SESSION['success_message']); ?>
                    <?php endif; ?>

                    <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <?= htmlspecialchars($_SESSION['error_message']) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php unset($_SESSION['error_message']); ?>
                    <?php endif; ?>

                    <?php if ($campaign_sent && $campaign_stats): ?>
                        <div class="alert alert-info">
                            <h5><i class="fas fa-chart-bar"></i> Campaign Results:</h5>
                            <ul class="mb-0">
                                <li><strong>Successfully Sent:</strong> <?= $campaign_stats['sent'] ?> emails</li>
                                <li><strong>Failed:</strong> <?= $campaign_stats['failed'] ?> emails</li>
                                <?php if (!empty($campaign_stats['failed_emails'])): ?>
                                    <li><strong>Failed Emails:</strong> <?= implode(', ', $campaign_stats['failed_emails']) ?></li>
                                <?php endif; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST" id="campaignForm">
                        <div class="mb-3">
                            <label class="form-label"><strong>Recipients</strong></label>
                            <select name="status_filter" class="form-select" required>
                                <option value="active">Active Subscribers (<?= number_format($active_count) ?>)</option>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label for="subject" class="form-label"><strong>Email Subject *</strong></label>
                            <input type="text" 
                                   class="form-control" 
                                   id="subject" 
                                   name="subject" 
                                   placeholder="e.g., WsplCloud Newsletter - Hosting Tips for December"
                                   required>
                        </div>

                        <div class="mb-3">
                            <label for="message" class="form-label"><strong>Email Message * (HTML supported)</strong></label>
                            <textarea class="form-control" 
                                      id="message" 
                                      name="message" 
                                      rows="15" 
                                      placeholder="Enter your email content here. HTML is supported."
                                      required></textarea>
                            <small class="text-muted">
                                You can use HTML tags. Preview before sending!
                            </small>
                        </div>

                        <div class="mb-3">
                            <button type="button" class="btn btn-info" data-bs-toggle="modal" data-bs-target="#previewModal">
                                <i class="fas fa-eye"></i> Preview Email
                            </button>
                        </div>

                        <hr>

                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <small class="text-muted">
                                    <i class="fas fa-info-circle"></i> 
                                    This will send to all active subscribers.
                                </small>
                            </div>
                            <button type="submit" 
                                    name="send_campaign" 
                                    class="btn btn-primary btn-lg"
                                    onclick="return confirm('Are you sure you want to send this campaign to all subscribers? This action cannot be undone.')">
                                <i class="fas fa-paper-plane"></i> Send Campaign
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Quick Templates -->
            <div class="card mt-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-file-alt"></i> Quick Templates</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-2">
                            <button class="btn btn-outline-primary btn-sm w-100" onclick="loadTemplate('welcome')">
                                Welcome Email
                            </button>
                        </div>
                        <div class="col-md-6 mb-2">
                            <button class="btn btn-outline-primary btn-sm w-100" onclick="loadTemplate('promotion')">
                                Promotional Offer
                            </button>
                        </div>
                        <div class="col-md-6 mb-2">
                            <button class="btn btn-outline-primary btn-sm w-100" onclick="loadTemplate('blog')">
                                New Blog Post
                            </button>
                        </div>
                        <div class="col-md-6 mb-2">
                            <button class="btn btn-outline-primary btn-sm w-100" onclick="loadTemplate('tips')">
                                Hosting Tips
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <!-- Stats Card -->
            <div class="card mb-4">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="fas fa-users"></i> Subscriber Stats</h5>
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between mb-3">
                        <span>Active Subscribers:</span>
                        <strong><?= number_format($active_count) ?></strong>
                    </div>
                    <a href="newsletter.php" class="btn btn-sm btn-outline-primary w-100">
                        <i class="fas fa-list"></i> View All Subscribers
                    </a>
                </div>
            </div>

            <!-- Recent Campaigns -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-history"></i> Recent Campaigns</h5>
                </div>
                <div class="card-body">
                    <?php if (empty($recent_campaigns)): ?>
                        <p class="text-muted mb-0">No campaigns sent yet</p>
                    <?php else: ?>
                        <div class="list-group list-group-flush">
                            <?php foreach ($recent_campaigns as $campaign): ?>
                                <div class="list-group-item px-0">
                                    <div class="d-flex justify-content-between">
                                        <strong><?= htmlspecialchars(substr($campaign['subject'], 0, 30)) ?>...</strong>
                                        <small class="text-muted"><?= date('M d', strtotime($campaign['sent_at'])) ?></small>
                                    </div>
                                    <small class="text-success">
                                        <i class="fas fa-check"></i> <?= $campaign['sent_count'] ?> sent
                                    </small>
                                    <?php if ($campaign['failed_count'] > 0): ?>
                                        <small class="text-danger ms-2">
                                            <i class="fas fa-times"></i> <?= $campaign['failed_count'] ?> failed
                                        </small>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Email Service Integration -->
            <div class="card mt-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="fas fa-plug"></i> Email Services</h5>
                </div>
                <div class="card-body">
                    <p class="small mb-2">For better deliverability, integrate with:</p>
                    <ul class="small">
                        <li><strong>SendGrid</strong> - Free 100 emails/day</li>
                        <li><strong>MailChimp</strong> - Free up to 500 subscribers</li>
                        <li><strong>AWS SES</strong> - $0.10 per 1000 emails</li>
                    </ul>
                    <a href="email_settings.php" class="btn btn-sm btn-info w-100">
                        <i class="fas fa-cog"></i> Configure Services
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-eye"></i> Email Preview</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="emailPreview" style="max-height: 600px; overflow-y: auto;"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
// Email templates
const templates = {
    welcome: {
        subject: 'Welcome to WsplCloud Newsletter!',
        message: `<h2>Welcome to WsplCloud!</h2>
<p>Thank you for subscribing to our newsletter. We're excited to have you on board!</p>
<p>Here's what you can expect:</p>
<ul>
    <li>Weekly hosting tips and tricks</li>
    <li>Exclusive offers and discounts</li>
    <li>Latest blog posts and tutorials</li>
    <li>Industry news and updates</li>
</ul>
<p>As a welcome gift, use code <strong>WELCOME20</strong> for 20% off your first order.</p>
<a href="${'<?= BASE_URL ?>'}/shared-hosting" class="button">Explore Our Hosting Plans</a>
<p>Best regards,<br>The WsplCloud Team</p>`
    },
    promotion: {
        subject: 'Special Offer: 50% Off VPS Hosting!',
        message: `<h2>🎉 Limited Time Offer!</h2>
<p>Get <strong>50% OFF</strong> on all VPS hosting plans this month!</p>
<h3>Why Choose WsplCloud VPS?</h3>
<ul>
    <li>✓ 99.9% Uptime Guarantee</li>
    <li>✓ SSD Storage for Maximum Speed</li>
    <li>✓ 24/7 Expert Support</li>
    <li>✓ Free SSL Certificate</li>
</ul>
<p>Use code: <strong>VPS50</strong> at checkout</p>
<a href="${'<?= BASE_URL ?>'}/vps" class="button">Get Started Now</a>
<p><small>Offer valid until ${new Date(Date.now() + 7*24*60*60*1000).toLocaleDateString()}</small></p>`
    },
    blog: {
        subject: 'New Blog Post: 10 Ways to Speed Up Your Website',
        message: `<h2>📝 New on Our Blog</h2>
<h3>10 Proven Ways to Speed Up Your Website</h3>
<p>Website speed is crucial for user experience and SEO. In our latest blog post, we share practical tips to make your site lightning fast.</p>
<p><strong>What you'll learn:</strong></p>
<ul>
    <li>Image optimization techniques</li>
    <li>Caching strategies that work</li>
    <li>CDN implementation</li>
    <li>Database optimization</li>
</ul>
<a href="${'<?= BASE_URL ?>'}/blog" class="button">Read Full Article</a>
<p>Happy reading!<br>WsplCloud Team</p>`
    },
    tips: {
        subject: 'Quick Hosting Tips for This Week',
        message: `<h2>💡 This Week's Hosting Tips</h2>
<h3>3 Quick Wins for Better Website Performance</h3>

<p><strong>1. Enable Gzip Compression</strong><br>
Reduce file sizes by up to 70%. Add this to your .htaccess file.</p>

<p><strong>2. Optimize Your Images</strong><br>
Use WebP format for 30% smaller file sizes without quality loss.</p>

<p><strong>3. Implement Browser Caching</strong><br>
Store static files locally to reduce server requests.</p>

<a href="${'<?= BASE_URL ?>'}/blog" class="button">More Tips on Our Blog</a>

<p>Need help? Our support team is available 24/7!</p>`
    }
};

function loadTemplate(type) {
    const template = templates[type];
    if (template) {
        document.getElementById('subject').value = template.subject;
        document.getElementById('message').value = template.message;
    }
}

// Preview email
document.querySelector('[data-bs-target="#previewModal"]').addEventListener('click', function() {
    const message = document.getElementById('message').value;
    const htmlTemplate = `
        <div style="max-width: 600px; margin: 0 auto; font-family: Arial, sans-serif;">
            <div style="background: linear-gradient(135deg, #0000ff 0%, #0303dd 100%); color: white; padding: 30px; text-align: center;">
                <h1 style="margin: 0;">WsplCloud</h1>
            </div>
            <div style="padding: 30px; background: white;">
                ${message}
            </div>
            <div style="background: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #666;">
                <p>You received this email because you subscribed to WsplCloud newsletter.</p>
                <p><a href="#" style="color: #0000ff;">Unsubscribe</a> | <a href="<?= BASE_URL ?>" style="color: #0000ff;">Visit Website</a></p>
            </div>
        </div>
    `;
    document.getElementById('emailPreview').innerHTML = htmlTemplate;
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
