<?php
/**
 * Admin Panel - Newsletter Subscribers Management
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';

$page_title = 'Newsletter Subscribers';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $id = $_POST['id'] ?? '';
    
    if ($action === 'delete' && $id) {
        try {
            $stmt = $db->prepare('DELETE FROM newsletter_subscribers WHERE id = :id');
            $stmt->execute(['id' => $id]);
            $_SESSION['success_message'] = 'Subscriber deleted successfully';
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error deleting subscriber: ' . $e->getMessage();
        }
    } elseif ($action === 'unsubscribe' && $id) {
        try {
            $stmt = $db->prepare('UPDATE newsletter_subscribers SET status = "unsubscribed", unsubscribed_at = NOW() WHERE id = :id');
            $stmt->execute(['id' => $id]);
            $_SESSION['success_message'] = 'Subscriber unsubscribed successfully';
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error unsubscribing: ' . $e->getMessage();
        }
    } elseif ($action === 'reactivate' && $id) {
        try {
            $stmt = $db->prepare('UPDATE newsletter_subscribers SET status = "active", unsubscribed_at = NULL WHERE id = :id');
            $stmt->execute(['id' => $id]);
            $_SESSION['success_message'] = 'Subscriber reactivated successfully';
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error reactivating: ' . $e->getMessage();
        }
    } elseif ($action === 'export') {
        try {
            $status_filter = $_POST['status_filter'] ?? 'all';
            $query = 'SELECT email, status, subscribed_at FROM newsletter_subscribers';
            
            if ($status_filter !== 'all') {
                $query .= ' WHERE status = :status';
            }
            
            $query .= ' ORDER BY subscribed_at DESC';
            
            $stmt = $db->prepare($query);
            if ($status_filter !== 'all') {
                $stmt->execute(['status' => $status_filter]);
            } else {
                $stmt->execute();
            }
            
            $subscribers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Generate CSV
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="newsletter_subscribers_' . date('Y-m-d') . '.csv"');
            
            $output = fopen('php://output', 'w');
            fputcsv($output, ['Email', 'Status', 'Subscribed Date']);
            
            foreach ($subscribers as $subscriber) {
                fputcsv($output, [
                    $subscriber['email'],
                    $subscriber['status'],
                    $subscriber['subscribed_at']
                ]);
            }
            
            fclose($output);
            exit;
        } catch (Exception $e) {
            $_SESSION['error_message'] = 'Error exporting subscribers: ' . $e->getMessage();
        }
    }
    
    header('Location: newsletter.php');
    exit;
}

// Get filter and search parameters
$status_filter = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 50;
$offset = ($page - 1) * $per_page;

// Build query
$where_conditions = [];
$params = [];

if ($status_filter !== 'all') {
    $where_conditions[] = 'status = :status';
    $params['status'] = $status_filter;
}

if (!empty($search)) {
    $where_conditions[] = 'email LIKE :search';
    $params['search'] = '%' . $search . '%';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get total count
$count_query = "SELECT COUNT(*) FROM newsletter_subscribers $where_clause";
$count_stmt = $db->prepare($count_query);
$count_stmt->execute($params);
$total_subscribers = $count_stmt->fetchColumn();
$total_pages = ceil($total_subscribers / $per_page);

// Get subscribers
$query = "SELECT * FROM newsletter_subscribers $where_clause ORDER BY subscribed_at DESC LIMIT $per_page OFFSET $offset";
$stmt = $db->prepare($query);
$stmt->execute($params);
$subscribers = $stmt->fetchAll();

// Get statistics
$stats_query = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
    SUM(CASE WHEN status = 'unsubscribed' THEN 1 ELSE 0 END) as unsubscribed,
    SUM(CASE WHEN DATE(subscribed_at) = CURDATE() THEN 1 ELSE 0 END) as today,
    SUM(CASE WHEN DATE(subscribed_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as week
FROM newsletter_subscribers";
$stats_result = $db->query($stats_query)->fetch();

// Ensure stats have default values if null
$stats = [
    'total' => (int)($stats_result['total'] ?? 0),
    'active' => (int)($stats_result['active'] ?? 0),
    'unsubscribed' => (int)($stats_result['unsubscribed'] ?? 0),
    'today' => (int)($stats_result['today'] ?? 0),
    'week' => (int)($stats_result['week'] ?? 0)
];

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid py-4">
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card stat-card bg-primary text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h3 class="mb-0"><?= number_format($stats['total']) ?></h3>
                            <p class="mb-0">Total Subscribers</p>
                        </div>
                        <i class="fas fa-users fa-3x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card bg-success text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h3 class="mb-0"><?= number_format($stats['active']) ?></h3>
                            <p class="mb-0">Active</p>
                        </div>
                        <i class="fas fa-check-circle fa-3x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card bg-warning text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h3 class="mb-0"><?= number_format($stats['today']) ?></h3>
                            <p class="mb-0">Today</p>
                        </div>
                        <i class="fas fa-calendar-day fa-3x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card bg-info text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h3 class="mb-0"><?= number_format($stats['week']) ?></h3>
                            <p class="mb-0">This Week</p>
                        </div>
                        <i class="fas fa-chart-line fa-3x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters and Actions -->
    <div class="card mb-4">
        <div class="card-header">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h4 class="mb-0"><i class="fas fa-envelope"></i> Newsletter Subscribers</h4>
                </div>
                <div class="col-md-6 text-end">
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="action" value="export">
                        <input type="hidden" name="status_filter" value="<?= htmlspecialchars($status_filter) ?>">
                        <button type="submit" class="btn btn-success btn-sm">
                            <i class="fas fa-download"></i> Export CSV
                        </button>
                    </form>
                </div>
            </div>
        </div>
        <div class="card-body">
            <form method="GET" class="row g-3 mb-3">
                <div class="col-md-4">
                    <input type="text" 
                           name="search" 
                           class="form-control" 
                           placeholder="Search by email..." 
                           value="<?= htmlspecialchars($search) ?>">
                </div>
                <div class="col-md-3">
                    <select name="status" class="form-select">
                        <option value="all" <?= $status_filter === 'all' ? 'selected' : '' ?>>All Status</option>
                        <option value="active" <?= $status_filter === 'active' ? 'selected' : '' ?>>Active</option>
                        <option value="unsubscribed" <?= $status_filter === 'unsubscribed' ? 'selected' : '' ?>>Unsubscribed</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-filter"></i> Filter
                    </button>
                </div>
                <div class="col-md-3">
                    <a href="newsletter.php" class="btn btn-secondary w-100">
                        <i class="fas fa-redo"></i> Reset
                    </a>
                </div>
            </form>

            <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?= htmlspecialchars($_SESSION['success_message']) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php unset($_SESSION['success_message']); ?>
            <?php endif; ?>

            <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?= htmlspecialchars($_SESSION['error_message']) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php unset($_SESSION['error_message']); ?>
            <?php endif; ?>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Email</th>
                            <th>Status</th>
                            <th>Subscribed Date</th>
                            <th>IP Address</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($subscribers)): ?>
                            <tr>
                                <td colspan="6" class="text-center py-4">
                                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                    <p>No subscribers found</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($subscribers as $subscriber): ?>
                                <tr>
                                    <td><?= $subscriber['id'] ?></td>
                                    <td>
                                        <i class="fas fa-envelope text-muted me-2"></i>
                                        <?= htmlspecialchars($subscriber['email']) ?>
                                    </td>
                                    <td>
                                        <?php if ($subscriber['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Unsubscribed</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <i class="fas fa-calendar text-muted me-2"></i>
                                        <?= date('M d, Y H:i', strtotime($subscriber['subscribed_at'])) ?>
                                    </td>
                                    <td>
                                        <small class="text-muted"><?= htmlspecialchars($subscriber['ip_address'] ?? 'N/A') ?></small>
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <?php if ($subscriber['status'] === 'active'): ?>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Unsubscribe this email?')">
                                                    <input type="hidden" name="action" value="unsubscribe">
                                                    <input type="hidden" name="id" value="<?= $subscriber['id'] ?>">
                                                    <button type="submit" class="btn btn-warning" title="Unsubscribe">
                                                        <i class="fas fa-ban"></i>
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="reactivate">
                                                    <input type="hidden" name="id" value="<?= $subscriber['id'] ?>">
                                                    <button type="submit" class="btn btn-success" title="Reactivate">
                                                        <i class="fas fa-check"></i>
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Permanently delete this subscriber?')">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?= $subscriber['id'] ?>">
                                                <button type="submit" class="btn btn-danger" title="Delete">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($total_pages > 1): ?>
                <nav>
                    <ul class="pagination justify-content-center">
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>&status=<?= urlencode($status_filter) ?>&search=<?= urlencode($search) ?>">
                                    <?= $i ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.stat-card {
    border: none;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    transition: transform 0.3s;
}

.stat-card:hover {
    transform: translateY(-5px);
}

.opacity-50 {
    opacity: 0.5;
}
</style>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
