<?php
/**
 * Admin Panel - Login Page
 * Handles user authentication.
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';

// If already logged in, redirect to dashboard
if (isset($_SESSION['admin_user'])) {
    header('Location: ' . url('admin/index.php'));
    exit;
}

$login_error = '';

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Brute-force protection (session-based)
    if (isset($_SESSION['login_attempts']) && $_SESSION['login_attempts'] >= 5) {
        $time_since_last_attempt = time() - $_SESSION['last_login_attempt'];
        if ($time_since_last_attempt < 300) { // 5 minutes lockout
            $login_error = 'Too many failed login attempts. Please try again in 5 minutes.';
        } else {
            // Reset attempts after lockout period
            unset($_SESSION['login_attempts']);
            unset($_SESSION['last_login_attempt']);
        }
    }

    // Brute-force protection (IP-based)
    $ip_key = 'login_attempt_' . get_client_ip();
    if (!check_rate_limit($ip_key, 10, 600)) { // 10 attempts in 10 minutes
        $login_error = 'Too many login attempts from your IP address. Please try again later.';
    }

    // CAPTCHA implementation (placeholder)
    // if (empty($login_error) && (isset($_SESSION['login_attempts']) && $_SESSION['login_attempts'] >= 3)) {
    //     // Display CAPTCHA and verify it here
    //     // if (!verify_captcha($_POST['captcha_response'])) {
    //     //     $login_error = 'Please complete the CAPTCHA.';
    //     // }
    // }

    if (empty($login_error)) {
        // CSRF token verification
        if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
            $login_error = 'Invalid CSRF token. Please try again.';
        } else {
            // Sanitize and validate input
            $username = sanitize_input($_POST['username']);
            $password = $_POST['password']; // Password should not be sanitized with htmlspecialchars, as password_verify() handles the comparison securely.

            // Basic validation
            if (empty($username) || empty($password)) {
                $login_error = 'Username and password are required.';
            } else {
                if ($db) {
                    try {
                        // Prepare and execute query to find user
                        $stmt = $db->prepare("SELECT id, username, password, role FROM users WHERE username = :username");
                        $stmt->bindParam(':username', $username);
                        $stmt->execute();
                        $user = $stmt->fetch(PDO::FETCH_ASSOC);

                        // Verify user and password
                        if ($user && password_verify($password, $user['password'])) {
                            // Authentication successful
                            session_regenerate_id(true); // Prevent session fixation
                            $_SESSION['admin_user'] = $user; // Store user data in session
                            
                            // Clear login attempts on success
                            unset($_SESSION['login_attempts']);
                            unset($_SESSION['last_login_attempt']);
                            // Also clear IP-based rate limit for this IP on success
                            check_rate_limit($ip_key, 0, 0); // Reset by setting limit to 0

                            header('Location: ' . (function_exists('url') ? url('admin/index.php') : SITE_ROOT . '/admin/index.php')); // Redirect to dashboard
                            exit;
                        } else {
                            // Invalid credentials
                            // Increment session-based login attempts
                            if (!isset($_SESSION['login_attempts'])) {
                                $_SESSION['login_attempts'] = 0;
                            }
                            $_SESSION['login_attempts']++;
                            $_SESSION['last_login_attempt'] = time();
                            
                            $login_error = 'Invalid username or password.';
                        }
                    } catch (PDOException $e) {
                        // Database error
                        error_log('Login query error: ' . $e->getMessage());
                        $login_error = 'An error occurred during login. Please try again later.';
                    }
                } else {
                    // DB connection error
                    $login_error = 'Database connection failed. Please contact the administrator.';
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login - <?php echo htmlspecialchars(SITE_NAME); ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Modern CSS -->
    <link rel="stylesheet" href="<?php echo css('admin-login'); ?>">
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <div class="card-header">
                <h4>
                    <i class="fas fa-user-shield me-2"></i>Admin Portal
                </h4>
            </div>
            <div class="card-body">
                <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST" id="loginForm">
                    <?php echo csrf_token_field(); // Include CSRF token field ?>
                    
                    <div class="input-group">
                        <input type="text" 
                               class="form-control" 
                               id="username" 
                               name="username" 
                               placeholder="Enter your username"
                               required
                               autocomplete="username">
                        <i class="fas fa-user input-icon"></i>
                    </div>
                    
                    <div class="input-group">
                        <input type="password" 
                               class="form-control" 
                               id="password" 
                               name="password" 
                               placeholder="Enter your password"
                               required
                               autocomplete="current-password">
                        <i class="fas fa-lock input-icon"></i>
                        <button type="button" class="password-toggle" id="passwordToggle">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    
                    <?php if (!empty($login_error)): ?>
                        <div class="alert alert-dismissible fade show" role="alert">
                            <div class="d-flex align-items-center">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <span><?php echo $login_error; ?></span>
                            </div>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    <?php endif; ?>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-login">
                            <span class="login-text">Sign In</span>
                            <span class="loading-spinner" style="display: none;">
                                <i class="fas fa-spinner fa-spin me-2"></i>Signing In...
                            </span>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Modern JavaScript Enhancements -->
    <script src="<?php echo js('admin-login'); ?>"></script>
</body>
</html>
