<?php
/**
 * Simple SMTP Mailer Class
 * A basic implementation to send emails via SMTP, replacing the unreliable mail() function.
 * This is a simplified version for demonstration and would typically be replaced by a robust library like PHPMailer.
 */

if (!defined('SECURE_ACCESS')) {
    die('Direct access not allowed');
}

class SmtpMailer {
    private $host;
    private $port;
    private $username;
    private $password;
    private $encryption; // 'tls' or 'ssl' or ''
    private $from_email;
    private $from_name;
    private $timeout = 30;

    public function __construct(
        string $host,
        int $port,
        string $username,
        string $password,
        string $encryption,
        string $from_email,
        string $from_name
    ) {
        $this->host = $host;
        $this->port = $port;
        $this->username = $username;
        $this->password = $password;
        $this->encryption = strtolower($encryption);
        $this->from_email = $from_email;
        $this->from_name = $from_name;
    }

    /**
     * Send an email via SMTP.
     *
     * @param string $to The recipient's email address.
     * @param string $subject The email subject.
     * @param string $message The email body (HTML).
     * @param array $headers Additional headers (e.g., Reply-To, CC, BCC).
     * @return bool True on success, false on failure.
     */
    public function send(string $to, string $subject, string $message, array $headers = []): bool {
        $socket = $this->connect();
        if (!$socket) {
            error_log("SMTP Mailer: Could not connect to SMTP host.");
            return false;
        }

        try {
            $this->ehlo($socket);
            $this->startTls($socket);
            $this->ehlo($socket); // EHLO again after STARTTLS
            $this->authenticate($socket);
            $this->mailFrom($socket);
            $this->rcptTo($socket, $to);
            $this->data($socket, $to, $subject, $message, $headers);
            $this->quit($socket);
            fclose($socket);
            return true;
        } catch (Exception $e) {
            error_log("SMTP Mailer Error: " . $e->getMessage());
            if (is_resource($socket)) {
                fclose($socket);
            }
            return false;
        }
    }

    private function connect() {
        $protocol = '';
        if ($this->encryption === 'ssl') {
            $protocol = 'ssl://';
        }

        $socket = @fsockopen($protocol . $this->host, $this->port, $errno, $errstr, $this->timeout);
        if (!$socket) {
            throw new Exception("Failed to connect to SMTP server: ($errno) $errstr");
        }
        $this->getResponse($socket); // Read initial greeting
        return $socket;
    }

    private function getResponse($socket): string {
        $response = '';
        while ($str = fgets($socket, 515)) {
            $response .= $str;
            if (substr($str, 3, 1) == ' ') {
                break;
            }
        }
        if (empty($response)) {
            throw new Exception("SMTP server did not respond.");
        }
        return $response;
    }

    private function sendCommand($socket, string $command, array $expectedCodes) {
        fputs($socket, $command . "\r\n");
        $response = $this->getResponse($socket);
        $code = (int) substr($response, 0, 3);
        if (!in_array($code, $expectedCodes)) {
            throw new Exception("SMTP command failed: '$command' Response: '$response'");
        }
        return $response;
    }

    private function ehlo($socket) {
        $this->sendCommand($socket, "EHLO " . $_SERVER['SERVER_NAME'], [250]);
    }

    private function startTls($socket) {
        if ($this->encryption === 'tls') {
            $this->sendCommand($socket, "STARTTLS", [220]);
            if (!stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                throw new Exception("Failed to enable TLS encryption.");
            }
        }
    }

    private function authenticate($socket) {
        if (!empty($this->username) && !empty($this->password)) {
            $this->sendCommand($socket, "AUTH LOGIN", [334]);
            $this->sendCommand($socket, base64_encode($this->username), [334]);
            $this->sendCommand($socket, base64_encode($this->password), [235]);
        }
    }

    private function mailFrom($socket) {
        $this->sendCommand($socket, "MAIL FROM:<{$this->from_email}>", [250]);
    }

    private function rcptTo($socket, string $to) {
        $this->sendCommand($socket, "RCPT TO:<{$to}>", [250, 251]);
    }

    private function data($socket, string $to, string $subject, string $message, array $extraHeaders) {
        $this->sendCommand($socket, "DATA", [354]);

        $headers = [
            "From: {$this->from_name} <{$this->from_email}>",
            "To: {$to}",
            "Subject: {$subject}",
            "MIME-Version: 1.0",
            "Content-Type: text/html; charset=UTF-8",
            "X-Mailer: PHP/" . phpversion()
        ];

        $allHeaders = array_merge($headers, $extraHeaders);
        $data = implode("\r\n", $allHeaders) . "\r\n\r\n" . $message;

        fputs($socket, $data . "\r\n.\r\n");
        $this->getResponse($socket); // Read response for DATA command
    }

    private function quit($socket) {
        $this->sendCommand($socket, "QUIT", [221]);
    }
}
