<?php
/**
 * Admin Panel - Hero Banner Management
 * Manage hero banner content with image upload
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/includes/header.php';

$message = '';
$error = '';

// Create hero_banner table if it doesn't exist
try {
    $db->exec("CREATE TABLE IF NOT EXISTS hero_banner (
        id INT PRIMARY KEY AUTO_INCREMENT,
        title TEXT NOT NULL,
        subtitle TEXT NOT NULL,
        button_text VARCHAR(100) NOT NULL,
        button_link VARCHAR(255) NOT NULL,
        image_path VARCHAR(255),
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )");
} catch (PDOException $e) {
    error_log('Hero banner table creation error: ' . $e->getMessage());
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
        $error = 'Invalid CSRF token. Please try again.';
    } else {
        $title = trim($_POST['title'] ?? '');
        $subtitle = trim($_POST['subtitle'] ?? '');
        $button_text = trim($_POST['button_text'] ?? '');
        $button_link = trim($_POST['button_link'] ?? '');
        $status = $_POST['status'] ?? 'active';
        
        // Validate required fields
        if (empty($title) || empty($subtitle) || empty($button_text) || empty($button_link)) {
            $error = 'All fields are required.';
        } else {
            try {
                // Handle image upload
                $image_path = null;
                if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                    $allowed_types = ['image/jpeg', 'image/png', 'image/webp', 'image/gif'];
                    $file_type = $_FILES['image']['type'];
                    
                    if (!in_array($file_type, $allowed_types)) {
                        throw new Exception('Invalid file type. Only JPG, PNG, WebP, and GIF are allowed.');
                    }
                    
                    if ($_FILES['image']['size'] > 5 * 1024 * 1024) {
                        throw new Exception('File size must be less than 5MB.');
                    }
                    
                    $upload_dir = BASE_PATH . '/assets/images/hero/';
                    if (!is_dir($upload_dir)) {
                        if (!mkdir($upload_dir, 0755, true)) {
                            throw new Exception('Failed to create upload directory. Please check permissions.');
                        }
                    }
                    
                    $file_extension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                    $new_filename = 'hero_' . time() . '.' . $file_extension;
                    $upload_path = $upload_dir . $new_filename;
                    
                    if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path)) {
                        $image_path = 'assets/images/hero/' . $new_filename;
                    } else {
                        throw new Exception('Failed to upload image. Please try again.');
                    }
                }
                
                // Check if hero banner exists
                $stmt = $db->query("SELECT id, image_path FROM hero_banner LIMIT 1");
                $existing = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($existing) {
                    // Update existing
                    if ($image_path) {
                        // Delete old image if exists
                        if (!empty($existing['image_path'])) {
                            $old_image_path = BASE_PATH . '/' . $existing['image_path'];
                            if (file_exists($old_image_path)) {
                                unlink($old_image_path);
                            }
                        }
                        $stmt = $db->prepare("UPDATE hero_banner SET title = :title, subtitle = :subtitle, 
                                             button_text = :button_text, button_link = :button_link, 
                                             image_path = :image_path, status = :status WHERE id = :id");
                        $stmt->bindParam(':image_path', $image_path);
                    } else {
                        $stmt = $db->prepare("UPDATE hero_banner SET title = :title, subtitle = :subtitle, 
                                             button_text = :button_text, button_link = :button_link, 
                                             status = :status WHERE id = :id");
                    }
                    $stmt->bindParam(':id', $existing['id']);
                } else {
                    // Insert new
                    if ($image_path) {
                        $stmt = $db->prepare("INSERT INTO hero_banner (title, subtitle, button_text, button_link, image_path, status) 
                                             VALUES (:title, :subtitle, :button_text, :button_link, :image_path, :status)");
                        $stmt->bindParam(':image_path', $image_path);
                    } else {
                        $stmt = $db->prepare("INSERT INTO hero_banner (title, subtitle, button_text, button_link, status) 
                                             VALUES (:title, :subtitle, :button_text, :button_link, :status)");
                    }
                }
                
                $stmt->bindParam(':title', $title);
                $stmt->bindParam(':subtitle', $subtitle);
                $stmt->bindParam(':button_text', $button_text);
                $stmt->bindParam(':button_link', $button_link);
                $stmt->bindParam(':status', $status);
                
                if ($stmt->execute()) {
                    $message = 'Hero banner updated successfully!';
                } else {
                    $error = 'Failed to update hero banner.';
                }
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
    }
}

// Get current hero banner data
$hero_data = null;
try {
    $stmt = $db->query("SELECT * FROM hero_banner LIMIT 1");
    $hero_data = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log('Fetch hero banner error: ' . $e->getMessage());
}
?>

<style>
.hero-admin-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.hero-card {
    background: white;
    border-radius: 15px;
    box-shadow: 0 2px 20px rgba(0,0,0,0.08);
    padding: 30px;
    margin-bottom: 30px;
}

.hero-card h2 {
    color: #0000ff;
    font-size: 1.8rem;
    margin-bottom: 10px;
    font-weight: 700;
}

.hero-card p {
    color: #666;
    margin-bottom: 25px;
}

.form-group {
    margin-bottom: 25px;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #333;
    margin-bottom: 8px;
    font-size: 0.95rem;
}

.form-group label .required {
    color: #ff0000;
    margin-left: 3px;
}

.form-control {
    width: 100%;
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 8px;
    font-size: 1rem;
    transition: all 0.3s;
    font-family: inherit;
}

.form-control:focus {
    outline: none;
    border-color: #0000ff;
    box-shadow: 0 0 0 3px rgba(0,0,255,0.1);
}

textarea.form-control {
    min-height: 100px;
    resize: vertical;
}

.file-upload-wrapper {
    position: relative;
    display: inline-block;
    width: 100%;
}

.file-upload-input {
    opacity: 0;
    position: absolute;
    z-index: -1;
}

.file-upload-label {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 20px;
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s;
    font-weight: 600;
    justify-content: center;
}

.file-upload-label:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,255,0.3);
}

.file-upload-label i {
    font-size: 1.2rem;
}

.image-preview {
    margin-top: 15px;
    border-radius: 10px;
    overflow: hidden;
    border: 2px solid #e0e0e0;
    max-width: 400px;
}

.image-preview img {
    width: 100%;
    height: auto;
    display: block;
}

.status-toggle {
    display: flex;
    align-items: center;
    gap: 15px;
}

.toggle-switch {
    position: relative;
    display: inline-block;
    width: 60px;
    height: 30px;
}

.toggle-switch input {
    opacity: 0;
    width: 0;
    height: 0;
}

.toggle-slider {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: #ccc;
    transition: 0.3s;
    border-radius: 30px;
}

.toggle-slider:before {
    position: absolute;
    content: "";
    height: 22px;
    width: 22px;
    left: 4px;
    bottom: 4px;
    background-color: white;
    transition: 0.3s;
    border-radius: 50%;
}

.toggle-switch input:checked + .toggle-slider {
    background-color: #0000ff;
}

.toggle-switch input:checked + .toggle-slider:before {
    transform: translateX(30px);
}

.status-label {
    font-weight: 600;
    color: #333;
}

.btn-group {
    display: flex;
    gap: 15px;
    margin-top: 30px;
}

.btn {
    padding: 14px 35px;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.btn-primary {
    background: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 20px rgba(0,0,255,0.4);
}

.btn-secondary {
    background: #f5f5f5;
    color: #333;
}

.btn-secondary:hover {
    background: #e0e0e0;
}

.alert {
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 10px;
}

.alert-success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.alert-danger {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.help-text {
    font-size: 0.85rem;
    color: #666;
    margin-top: 5px;
    font-style: italic;
}

.preview-section {
    background: linear-gradient(135deg, #f5f7fa 0%, #e9ecef 100%);
    padding: 30px;
    border-radius: 10px;
    margin-top: 30px;
}

.preview-section h3 {
    color: #0000ff;
    font-size: 1.4rem;
    margin-bottom: 20px;
    font-weight: 700;
}

.hero-preview {
    background: #0000ff;
    padding: 60px 30px;
    border-radius: 15px;
    color: white;
    text-align: center;
}

.hero-preview h1 {
    font-size: 2.5rem;
    font-weight: 800;
    margin-bottom: 15px;
}

.hero-preview .highlight {
    color: #ffd200;
}

.hero-preview p {
    font-size: 1.1rem;
    margin-bottom: 20px;
    color: rgba(255,255,255,0.9);
}

.hero-preview .btn {
    background: #ffd200;
    color: #0000ff;
}

@media (max-width: 768px) {
    .hero-card {
        padding: 20px;
    }
    
    .btn-group {
        flex-direction: column;
    }
    
    .btn {
        width: 100%;
        justify-content: center;
    }
}
</style>

<div class="hero-admin-container">
    <?php if ($message): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i>
            <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <div class="hero-card">
        <h2><i class="fas fa-star"></i> Hero Banner Management</h2>
        <p>Manage your homepage hero banner content and image</p>

        <form method="POST" enctype="multipart/form-data">
            <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">

            <div class="form-group">
                <label for="title">
                    Hero Title <span class="required">*</span>
                </label>
                <textarea 
                    id="title" 
                    name="title" 
                    class="form-control" 
                    rows="3" 
                    required
                    placeholder="Enter hero title"
                ><?php echo htmlspecialchars($hero_data['title'] ?? 'Fast • Secure • Reliable<br><span class="highlight">Web Hosting in India</span>'); ?></textarea>
                <div class="help-text">Use &lt;span class="highlight"&gt; for yellow highlighted text</div>
            </div>

            <div class="form-group">
                <label for="subtitle">
                    Hero Subtitle <span class="required">*</span>
                </label>
                <textarea 
                    id="subtitle" 
                    name="subtitle" 
                    class="form-control" 
                    rows="3" 
                    required
                    placeholder="Enter hero subtitle"
                ><?php echo htmlspecialchars($hero_data['subtitle'] ?? 'Empower your business with high-performance hosting, 24/7 expert support, and guaranteed 99.9% uptime.'); ?></textarea>
            </div>

            <div class="form-group">
                <label for="button_text">
                    Button Text <span class="required">*</span>
                </label>
                <input 
                    type="text" 
                    id="button_text" 
                    name="button_text" 
                    class="form-control" 
                    value="<?php echo htmlspecialchars($hero_data['button_text'] ?? 'Get Started Now'); ?>"
                    required
                    placeholder="e.g., Get Started Now"
                >
            </div>

            <div class="form-group">
                <label for="button_link">
                    Button Link <span class="required">*</span>
                </label>
                <input 
                    type="text" 
                    id="button_link" 
                    name="button_link" 
                    class="form-control" 
                    value="<?php echo htmlspecialchars($hero_data['button_link'] ?? 'shared-hosting.php'); ?>"
                    required
                    placeholder="e.g., shared-hosting.php or contact.php"
                >
                <div class="help-text">Enter relative URL (e.g., shared-hosting.php) or full URL</div>
            </div>

            <div class="form-group">
                <label for="image">
                    Hero Image (Optional)
                </label>
                <div class="file-upload-wrapper">
                    <input 
                        type="file" 
                        id="image" 
                        name="image" 
                        class="file-upload-input" 
                        accept="image/*"
                        onchange="previewImage(this)"
                    >
                    <label for="image" class="file-upload-label">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <span>Choose Hero Image</span>
                    </label>
                </div>
                <div class="help-text">Recommended size: 800x600px. Max 5MB. (JPG, PNG, WebP, GIF)</div>
                
                <?php if (!empty($hero_data['image_path'])): ?>
                    <div class="image-preview" id="currentImagePreview">
                        <img src="<?php echo url('/' . $hero_data['image_path']); ?>" alt="Current Hero Image">
                    </div>
                <?php endif; ?>
                
                <div class="image-preview" id="imagePreview" style="display: none;">
                    <img src="" alt="Image Preview" id="previewImg">
                </div>
            </div>

            <div class="form-group">
                <label>Status</label>
                <div class="status-toggle">
                    <label class="toggle-switch">
                        <input 
                            type="checkbox" 
                            name="status" 
                            value="active"
                            <?php echo (!isset($hero_data['status']) || $hero_data['status'] === 'active') ? 'checked' : ''; ?>
                        >
                        <span class="toggle-slider"></span>
                    </label>
                    <span class="status-label">
                        <?php echo (!isset($hero_data['status']) || $hero_data['status'] === 'active') ? 'Active' : 'Inactive'; ?>
                    </span>
                </div>
            </div>

            <div class="btn-group">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i>
                    Save Hero Banner
                </button>
                <a href="index.php" class="btn btn-secondary">
                    <i class="fas fa-times"></i>
                    Cancel
                </a>
            </div>
        </form>
    </div>

    <?php if ($hero_data): ?>
    <div class="preview-section">
        <h3><i class="fas fa-eye"></i> Current Hero Banner Preview</h3>
        <div class="hero-preview">
            <h1><?php echo $hero_data['title']; ?></h1>
            <p><?php echo htmlspecialchars($hero_data['subtitle']); ?></p>
            <a href="#" class="btn"><?php echo htmlspecialchars($hero_data['button_text']); ?></a>
        </div>
    </div>
    <?php endif; ?>
</div>

<script>
function previewImage(input) {
    const preview = document.getElementById('imagePreview');
    const previewImg = document.getElementById('previewImg');
    const currentPreview = document.getElementById('currentImagePreview');
    
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        
        reader.onload = function(e) {
            previewImg.src = e.target.result;
            preview.style.display = 'block';
            if (currentPreview) {
                currentPreview.style.display = 'none';
            }
        };
        
        reader.readAsDataURL(input.files[0]);
    }
}

// Update status label when toggle changes
document.querySelector('.toggle-switch input').addEventListener('change', function() {
    const statusLabel = document.querySelector('.status-label');
    statusLabel.textContent = this.checked ? 'Active' : 'Inactive';
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
