<?php
/**
 * Email Helper Class
 * Handles email sending with multiple backends
 */

class EmailHelper {
    private $db;
    private $config;
    
    public function __construct($db) {
        $this->db = $db;
        $this->loadConfig();
    }
    
    private function loadConfig() {
        try {
            $stmt = $this->db->query("SELECT * FROM email_settings WHERE is_active = 1 LIMIT 1");
            $this->config = $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->config = null;
            error_log("Email config error: " . $e->getMessage());
        }
    }
    
    /**
     * Send email using configured method
     */
    public function send($to, $subject, $html_message, $plain_message = null) {
        if (!$this->config) {
            return $this->sendWithPHPMail($to, $subject, $html_message);
        }
        
        switch ($this->config['service_type']) {
            case 'smtp':
                return $this->sendWithSMTP($to, $subject, $html_message);
            case 'sendgrid':
                return $this->sendWithSendGrid($to, $subject, $html_message);
            case 'mailchimp':
                return $this->sendWithMailChimp($to, $subject, $html_message);
            case 'aws_ses':
                return $this->sendWithAWS($to, $subject, $html_message);
            default:
                return $this->sendWithPHPMail($to, $subject, $html_message);
        }
    }
    
    /**
     * Send using native PHP mail() - Works if postfix is configured
     */
    private function sendWithPHPMail($to, $subject, $html_message) {
        $headers = [
            'From: ' . ($this->config['sender_name'] ?? 'WsplCloud') . ' <' . ($this->config['sender_email'] ?? 'noreply@wsplcloud.com') . '>',
            'Reply-To: ' . ($this->config['sender_email'] ?? 'support@wsplcloud.com'),
            'MIME-Version: 1.0',
            'Content-Type: text/html; charset=UTF-8',
            'X-Mailer: PHP/' . phpversion(),
            'X-Priority: 3',
            'Importance: Normal'
        ];
        
        $result = @mail($to, $subject, $html_message, implode("\r\n", $headers));
        
        if (!$result) {
            $error = error_get_last();
            error_log("PHP mail() failed: " . ($error['message'] ?? 'Unknown error'));
        }
        
        return $result;
    }
    
    /**
     * Send using SMTP (with socket connection)
     */
    private function sendWithSMTP($to, $subject, $html_message) {
        try {
            $host = $this->config['smtp_host'];
            $port = $this->config['smtp_port'];
            $username = $this->config['smtp_username'];
            $password = $this->config['smtp_password'];
            $encryption = $this->config['smtp_encryption'];
            $from = $this->config['sender_email'];
            $from_name = $this->config['sender_name'];
            
            // Create socket connection
            if ($encryption === 'ssl') {
                $socket = @fsockopen('ssl://' . $host, $port, $errno, $errstr, 30);
            } else {
                $socket = @fsockopen($host, $port, $errno, $errstr, 30);
            }
            
            if (!$socket) {
                error_log("SMTP connection failed: $errstr ($errno)");
                return false;
            }
            
            // Read server response
            $response = fgets($socket);
            
            // Send EHLO
            fputs($socket, "EHLO " . $host . "\r\n");
            $response = fgets($socket);
            
            // STARTTLS for TLS encryption
            if ($encryption === 'tls') {
                fputs($socket, "STARTTLS\r\n");
                $response = fgets($socket);
                stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
                fputs($socket, "EHLO " . $host . "\r\n");
                $response = fgets($socket);
            }
            
            // AUTH LOGIN
            fputs($socket, "AUTH LOGIN\r\n");
            $response = fgets($socket);
            fputs($socket, base64_encode($username) . "\r\n");
            $response = fgets($socket);
            fputs($socket, base64_encode($password) . "\r\n");
            $response = fgets($socket);
            
            if (strpos($response, '235') === false) {
                error_log("SMTP authentication failed");
                fclose($socket);
                return false;
            }
            
            // MAIL FROM
            fputs($socket, "MAIL FROM: <$from>\r\n");
            $response = fgets($socket);
            
            // RCPT TO
            fputs($socket, "RCPT TO: <$to>\r\n");
            $response = fgets($socket);
            
            // DATA
            fputs($socket, "DATA\r\n");
            $response = fgets($socket);
            
            // Email headers and body
            $email_content = "From: $from_name <$from>\r\n";
            $email_content .= "To: <$to>\r\n";
            $email_content .= "Subject: $subject\r\n";
            $email_content .= "MIME-Version: 1.0\r\n";
            $email_content .= "Content-Type: text/html; charset=UTF-8\r\n";
            $email_content .= "\r\n";
            $email_content .= $html_message . "\r\n";
            $email_content .= ".\r\n";
            
            fputs($socket, $email_content);
            $response = fgets($socket);
            
            // QUIT
            fputs($socket, "QUIT\r\n");
            fclose($socket);
            
            return true;
            
        } catch (Exception $e) {
            error_log("SMTP send error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send using SendGrid API
     */
    private function sendWithSendGrid($to, $subject, $html_message) {
        $api_key = $this->config['sendgrid_api_key'];
        
        $data = [
            'personalizations' => [[
                'to' => [['email' => $to]]
            ]],
            'from' => [
                'email' => $this->config['sender_email'],
                'name' => $this->config['sender_name']
            ],
            'subject' => $subject,
            'content' => [[
                'type' => 'text/html',
                'value' => $html_message
            ]]
        ];
        
        $ch = curl_init('https://api.sendgrid.com/v3/mail/send');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $api_key,
            'Content-Type: application/json'
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code >= 200 && $http_code < 300) {
            return true;
        } else {
            error_log("SendGrid API error: HTTP $http_code - $response");
            return false;
        }
    }
    
    /**
     * Send using MailChimp Transactional API (Mandrill)
     */
    private function sendWithMailChimp($to, $subject, $html_message) {
        $api_key = $this->config['mailchimp_api_key'];
        
        // MailChimp Transactional (Mandrill)
        $data = [
            'key' => $api_key,
            'message' => [
                'html' => $html_message,
                'subject' => $subject,
                'from_email' => $this->config['sender_email'],
                'from_name' => $this->config['sender_name'],
                'to' => [
                    ['email' => $to, 'type' => 'to']
                ]
            ]
        ];
        
        $ch = curl_init('https://mandrillapp.com/api/1.0/messages/send.json');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($http_code == 200 && isset($result[0]['status']) && $result[0]['status'] === 'sent') {
            return true;
        } else {
            error_log("MailChimp API error: " . json_encode($result));
            return false;
        }
    }
    
    /**
     * Send using AWS SES
     */
    private function sendWithAWS($to, $subject, $html_message) {
        // AWS SES implementation would require AWS SDK
        // For now, fallback to PHP mail
        error_log("AWS SES not implemented, using PHP mail() instead");
        return $this->sendWithPHPMail($to, $subject, $html_message);
    }
    
    /**
     * Wrap message in HTML template
     */
    public function wrapInTemplate($message, $unsubscribe_link = '') {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background: #f4f4f4; }
        .container { max-width: 600px; margin: 20px auto; background: white; border-radius: 10px; overflow: hidden; box-shadow: 0 0 20px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #0000ff 0%, #0303dd 100%); color: white; padding: 30px; text-align: center; }
        .header h1 { margin: 0; font-size: 28px; }
        .content { padding: 30px; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #666; border-top: 1px solid #ddd; }
        .footer a { color: #0000ff; text-decoration: none; }
        .button { display: inline-block; background: #0000ff; color: white !important; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
        @media only screen and (max-width: 600px) {
            .container { margin: 0; border-radius: 0; }
            .content { padding: 20px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>WsplCloud</h1>
        </div>
        <div class="content">
            ' . $message . '
        </div>
        <div class="footer">
            <p>You received this email because you subscribed to WsplCloud newsletter.</p>
            ' . ($unsubscribe_link ? '<p><a href="' . $unsubscribe_link . '">Unsubscribe</a> | <a href="' . BASE_URL . '">Visit Website</a></p>' : '') . '
            <p>&copy; ' . date('Y') . ' WsplCloud. All rights reserved.</p>
        </div>
    </div>
</body>
</html>';
    }
}
