<?php
/**
 * Admin Panel - Contact Requests Management
 * Displays a list of contact requests and allows deletion.
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/includes/header.php';

$message = '';
$error = '';

// Handle AJAX mark-as-read requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'mark_read' && isset($_POST['id'])) {
    header('Content-Type: application/json');
    $resp = ['success' => false, 'message' => ''];
    $contact_id = intval($_POST['id']);
    $csrf = $_POST['csrf_token'] ?? '';
    
    if (!verify_csrf_token($csrf)) {
        $resp['message'] = 'Invalid CSRF token';
        echo json_encode($resp);
        exit;
    }

    if ($db) {
        try {
            $stmt = $db->prepare("UPDATE contacts SET is_read = 1 WHERE id = :id");
            $stmt->bindParam(':id', $contact_id, PDO::PARAM_INT);
            $stmt->execute();
            $resp['success'] = true;
            $resp['message'] = 'Marked as read';
        } catch (PDOException $e) {
            error_log('Mark contact read error: ' . $e->getMessage());
            $resp['message'] = 'DB error';
        }
    } else {
        $resp['message'] = 'DB connection failed';
    }

    echo json_encode($resp);
    exit;
}

// Handle AJAX mark-as-unread requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'mark_unread' && isset($_POST['id'])) {
    header('Content-Type: application/json');
    $resp = ['success' => false, 'message' => ''];
    $contact_id = intval($_POST['id']);
    $csrf = $_POST['csrf_token'] ?? '';
    
    if (!verify_csrf_token($csrf)) {
        $resp['message'] = 'Invalid CSRF token';
        echo json_encode($resp);
        exit;
    }

    if ($db) {
        try {
            $stmt = $db->prepare("UPDATE contacts SET is_read = 0 WHERE id = :id");
            $stmt->bindParam(':id', $contact_id, PDO::PARAM_INT);
            $stmt->execute();
            $resp['success'] = true;
            $resp['message'] = 'Marked as unread';
        } catch (PDOException $e) {
            error_log('Mark contact unread error: ' . $e->getMessage());
            $resp['message'] = 'DB error';
        }
    } else {
        $resp['message'] = 'DB connection failed';
    }

    echo json_encode($resp);
    exit;
}

// Fetch single contact via AJAX (used by modal) e.g., ?action=fetch&id=123
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'fetch' && isset($_GET['id'])) {
    header('Content-Type: application/json');
    $resp = ['success' => false, 'data' => null, 'message' => ''];
    $cid = intval($_GET['id']);
    
    if ($db) {
        try {
            $stmt = $db->prepare("SELECT id, fname, lname, email, phone, message, created, is_read FROM contacts WHERE id = :id LIMIT 1");
            $stmt->bindParam(':id', $cid, PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) {
                $resp['success'] = true;
                $resp['data'] = $row;
            } else {
                $resp['message'] = 'Not found';
            }
        } catch (PDOException $e) {
            error_log('Fetch contact ajax error: ' . $e->getMessage());
            $resp['message'] = 'DB error';
        }
    } else {
        $resp['message'] = 'DB connection failed';
    }
    echo json_encode($resp);
    exit;
}

// Handle deletion of contact requests
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $contact_id = intval($_GET['id']);
    $csrf = $_GET['csrf_token'] ?? '';
    
    if (!verify_csrf_token($csrf)) {
        $error = 'Invalid CSRF token';
    } elseif ($db) {
        try {
            $stmt = $db->prepare("DELETE FROM contacts WHERE id = :id");
            $stmt->bindParam(':id', $contact_id, PDO::PARAM_INT);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $message = 'Contact request deleted successfully!';
            } else {
                $error = 'Contact request not found or could not be deleted.';
            }
        } catch (PDOException $e) {
            error_log('Delete contact error: ' . $e->getMessage());
            $error = 'Error deleting contact request. Please try again later.';
        }
    } else {
        $error = 'Database connection failed.';
    }
}

// Fetch contact requests
$contacts = [];
$total_contacts = 0;
$today_contacts = 0;
$unread_contacts = 0;

if ($db) {
    try {
        $stmt = $db->query("SELECT id, fname, lname, email, phone, message, created, is_read FROM contacts ORDER BY created DESC");
        $contacts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // If the error is 'Unknown column is_read', try to add the column and retry once
        $msg = $e->getMessage();
        error_log('Contact list fetch error: ' . $msg);
        $attemptedFix = false;
        
        if (stripos($msg, "Unknown column 'is_read'") !== false || (stripos($msg, 'is_read') !== false && stripos($msg, 'Unknown') !== false)) {
            try {
                // Check if column exists first
                $cols = $db->query("SHOW COLUMNS FROM `contacts` LIKE 'is_read'")->fetchAll();
                if (empty($cols)) {
                    $db->exec("ALTER TABLE `contacts` ADD COLUMN `is_read` TINYINT(1) NOT NULL DEFAULT 0");
                    $attemptedFix = true;
                }
            } catch (PDOException $e2) {
                error_log('Failed to add is_read column: ' . $e2->getMessage());
            }
        }

        if ($attemptedFix) {
            // Retry fetching contacts once
            try {
                $stmt = $db->query("SELECT id, fname, lname, email, phone, message, created, is_read FROM contacts ORDER BY created DESC");
                $contacts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (PDOException $e4) {
                error_log('Contact list fetch retry failed: ' . $e4->getMessage());
                $error = defined('ENVIRONMENT') && ENVIRONMENT === 'development' 
                    ? 'Error fetching contact requests after attempting fix: ' . $e4->getMessage()
                    : 'Error fetching contact requests.';
            }
        } else {
            $error = defined('ENVIRONMENT') && ENVIRONMENT === 'development' 
                ? 'Error fetching contact requests: ' . $msg
                : 'Error fetching contact requests.';
        }
    }

    // If contacts were fetched successfully, compute statistics
    if (!empty($contacts) && is_array($contacts)) {
        $total_contacts = count($contacts);
        $today = date('Y-m-d');
        
        $today_contacts = array_filter($contacts, function($contact) use ($today) {
            return date('Y-m-d', strtotime($contact['created'])) === $today;
        });
        $today_contacts = count($today_contacts);
        
        $unread_contacts = array_filter($contacts, function($contact) {
            return empty($contact['is_read']) || $contact['is_read'] == 0;
        });
        $unread_contacts = count($unread_contacts);
    }
}
?>

<style>
:root {
    --primary-blue: #0000ff;
    --blue-dark: #0000cc;
    --blue-light: #3366ff;
    --blue-glow: rgba(0, 0, 255, 0.2);
    --success-color: #00d97e;
    --warning-color: #ffc107;
    --danger-color: #e74c3c;
}

.contact-management {
    animation: fadeInUp 0.6s ease-out;
}

.stats-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid rgba(0, 0, 0, 0.05);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
}

.stat-card.total::before { background: var(--primary-blue); }
.stat-card.today::before { background: var(--success-color); }
.stat-card.unread::before { background: var(--warning-color); }

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 30px var(--blue-glow);
}

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    color: #6c757d;
    font-weight: 500;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.contact-table-container {
    background: white;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    overflow: hidden;
    animation: slideInUp 0.6s ease-out;
}

.table-header {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    color: white;
    padding: 1.5rem 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.table-header h3 {
    margin: 0;
    font-weight: 600;
}

.table-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.search-box {
    position: relative;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 10px;
    padding: 0.5rem 1rem;
    min-width: 250px;
}

.search-box input {
    background: transparent;
    border: none;
    color: white;
    width: 100%;
    padding-left: 2rem;
    outline: none;
}

.search-box input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

.search-box i {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: rgba(255, 255, 255, 0.7);
}

.table {
    margin: 0;
    width: 100%;
}

.table th {
    background: #f8f9fa;
    border-bottom: 2px solid #e9ecef;
    font-weight: 600;
    color: #2c3e50;
    padding: 1rem 1.5rem;
    text-transform: uppercase;
    font-size: 0.8rem;
    letter-spacing: 0.5px;
}

.table td {
    padding: 1rem 1.5rem;
    vertical-align: middle;
    border-color: #f8f9fa;
}

.contact-avatar {
    width: 45px;
    height: 45px;
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 1.1rem;
    margin-right: 1rem;
}

.contact-info {
    display: flex;
    align-items: center;
}

.contact-name {
    font-weight: 600;
    color: #2c3e50;
}

.contact-email {
    color: #6c757d;
    font-size: 0.85rem;
}

.message-snippet {
    max-width: 280px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.unread-badge {
    background: var(--warning-color);
    color: white;
    border-radius: 10px;
    padding: 0.2rem 0.6rem;
    font-size: 0.7rem;
    font-weight: 600;
}

.read-status {
    color: #6c757d;
}

.action-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: nowrap;
}

.btn-action {
    width: 35px;
    height: 35px;
    border: none;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    text-decoration: none;
    cursor: pointer;
}

.btn-view {
    background: rgba(0, 0, 255, 0.1);
    color: var(--primary-blue);
}

.btn-view:hover {
    background: var(--primary-blue);
    color: white;
    transform: scale(1.1);
}

.btn-mark-unread {
    background: rgba(255, 193, 7, 0.1);
    color: var(--warning-color);
}

.btn-mark-unread:hover {
    background: var(--warning-color);
    color: white;
    transform: scale(1.1);
}

.btn-delete {
    background: rgba(231, 76, 60, 0.1);
    color: var(--danger-color);
}

.btn-delete:hover {
    background: var(--danger-color);
    color: white;
    transform: scale(1.1);
}

/* Modal Styles */
.contact-modal .modal-header {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    color: white;
    border-bottom: none;
}

.contact-modal .modal-title {
    font-weight: 600;
}

.contact-details {
    padding: 1.5rem 0;
}

.detail-item {
    display: flex;
    margin-bottom: 1rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 10px;
    border-left: 4px solid var(--primary-blue);
}

.detail-label {
    font-weight: 600;
    color: #2c3e50;
    min-width: 100px;
}

.detail-value {
    color: #6c757d;
    flex: 1;
    word-break: break-word;
}

.message-content {
    background: #f8f9fa;
    padding: 1.5rem;
    border-radius: 10px;
    border-left: 4px solid var(--success-color);
    margin-top: 1rem;
    line-height: 1.6;
    white-space: pre-wrap;
    word-break: break-word;
}

/* Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(50px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Status transition animation */
.status-cell {
    transition: all 0.3s ease;
}

.status-cell.text-success {
    color: var(--success-color) !important;
    font-weight: 600;
}

.status-cell.text-warning {
    color: var(--warning-color) !important;
    font-weight: 600;
}

/* Button loading state */
.btn-action.loading i {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Pulse animation for counters */
.counter-update {
    animation: pulse 0.5s ease-in-out;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.1); }
    100% { transform: scale(1); }
}

/* Success checkmark */
.btn-view .fa-check {
    color: var(--success-color) !important;
}

/* Responsive */
@media (max-width: 768px) {
    .stats-cards {
        grid-template-columns: 1fr;
    }
    
    .table-header {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .table-actions {
        flex-direction: column;
        width: 100%;
    }
    
    .search-box {
        min-width: auto;
        width: 100%;
    }
    
    .contact-info {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
    
    .contact-avatar {
        margin-right: 0;
    }
    
    .action-buttons {
        justify-content: flex-start;
        flex-wrap: wrap;
    }
    
    .detail-item {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .detail-label {
        min-width: auto;
    }
}

@media (max-width: 480px) {
    .btn-action {
        width: 32px;
        height: 32px;
        font-size: 0.8rem;
    }
    
    .contact-avatar {
        width: 35px;
        height: 35px;
        font-size: 0.9rem;
    }
    
    .table td, .table th {
        padding: 0.75rem 1rem;
    }
}
</style>

<div class="contact-management">
    <?php if (!empty($message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <!-- Statistics Cards -->
    <div class="stats-cards">
        <div class="stat-card total">
            <div class="stat-number" id="total-contacts"><?php echo $total_contacts; ?></div>
            <div class="stat-label">Total Requests</div>
            <i class="fas fa-envelope fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
        </div>
        <div class="stat-card today">
            <div class="stat-number"><?php echo $today_contacts; ?></div>
            <div class="stat-label">Today's Requests</div>
            <i class="fas fa-calendar-day fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
        </div>
        <div class="stat-card unread">
            <div class="stat-number" id="unread-contacts"><?php echo $unread_contacts; ?></div>
            <div class="stat-label">Unread Messages</div>
            <i class="fas fa-eye-slash fa-2x text-muted position-absolute" style="bottom: 1rem; right: 1rem; opacity: 0.3;"></i>
        </div>
    </div>

    <!-- Contact Requests Table -->
    <div class="contact-table-container">
        <div class="table-header">
            <h3><i class="fas fa-envelope me-2"></i>Contact Requests</h3>
            <div class="table-actions">
                <div class="search-box">
                    <i class="fas fa-search"></i>
                    <input type="text" id="contactSearch" placeholder="Search contacts...">
                </div>
            </div>
        </div>
        
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="contactsTable">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Email</th>
                        <th>Phone</th>
                        <th>Message</th>
                        <th>Received</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($contacts)): ?>
                        <?php foreach ($contacts as $contact): ?>
                            <?php
                                $firstName = trim((string)($contact['fname'] ?? ''));
                                $lastName = trim((string)($contact['lname'] ?? ''));
                                $emailData = trim((string)($contact['email'] ?? ''));
                                $phoneData = trim((string)($contact['phone'] ?? ''));
                                $messageData = (string)($contact['message'] ?? '');
                                $isRead = !empty($contact['is_read']);

                                $fullNameRaw = trim($firstName . ' ' . $lastName);
                                $fullNameDisplay = $fullNameRaw !== '' ? htmlspecialchars($fullNameRaw, ENT_QUOTES, 'UTF-8') : '<span class="text-muted">—</span>';
                                $fullNamePlain = $fullNameRaw !== '' ? $fullNameRaw : 'Unknown Contact';

                                $emailDisplay = $emailData !== ''
                                    ? '<a class="contact-email" href="mailto:' . htmlspecialchars($emailData, ENT_QUOTES, 'UTF-8') . '">' . htmlspecialchars($emailData, ENT_QUOTES, 'UTF-8') . '</a>'
                                    : '<span class="text-muted contact-email">—</span>';

                                $phoneDisplay = $phoneData !== ''
                                    ? '<a href="tel:' . htmlspecialchars($phoneData, ENT_QUOTES, 'UTF-8') . '">' . htmlspecialchars($phoneData, ENT_QUOTES, 'UTF-8') . '</a>'
                                    : '<span class="text-muted">N/A</span>';

                                $messageTitle = $messageData !== '' ? htmlspecialchars($messageData, ENT_QUOTES, 'UTF-8') : 'No message provided';
                                $messageSnippet = $messageData;

                                if ($messageSnippet !== '') {
                                    if (function_exists('mb_strlen') && function_exists('mb_substr')) {
                                        if (mb_strlen($messageSnippet) > 80) {
                                            $messageSnippet = mb_substr($messageSnippet, 0, 77) . '...';
                                        }
                                    } else {
                                        if (strlen($messageSnippet) > 80) {
                                            $messageSnippet = substr($messageSnippet, 0, 77) . '...';
                                        }
                                    }
                                }

                                $messageDisplay = $messageData !== ''
                                    ? htmlspecialchars($messageSnippet, ENT_QUOTES, 'UTF-8')
                                    : '<span class="text-muted">—</span>';

                                $firstInitial = '';
                                $lastInitial = '';

                                if ($firstName !== '') {
                                    $firstInitial = function_exists('mb_substr') ? mb_substr($firstName, 0, 1) : substr($firstName, 0, 1);
                                }

                                if ($lastName !== '') {
                                    $lastInitial = function_exists('mb_substr') ? mb_substr($lastName, 0, 1) : substr($lastName, 0, 1);
                                }

                                $initialsRaw = $firstInitial . $lastInitial;
                                if ($initialsRaw !== '') {
                                    $avatarInitials = function_exists('mb_strtoupper') ? mb_strtoupper($initialsRaw) : strtoupper($initialsRaw);
                                } else {
                                    $avatarInitials = 'WC';
                                }
                            ?>
                            <tr data-contact-id="<?php echo $contact['id']; ?>" class="contact-row">
                                <td>
                                    <div class="contact-info">
                                        <div class="contact-avatar">
                                            <?php echo htmlspecialchars($avatarInitials, ENT_QUOTES, 'UTF-8'); ?>
                                        </div>
                                        <div class="contact-name">
                                            <?php echo $fullNameDisplay; ?>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <?php echo $emailDisplay; ?>
                                </td>
                                <td>
                                    <?php echo $phoneDisplay; ?>
                                </td>
                                <td class="message-snippet" title="<?php echo $messageTitle; ?>">
                                    <?php echo $messageDisplay; ?>
                                </td>
                                <td>
                                    <div class="text-muted">
                                        <?php echo date('M j, Y', strtotime($contact['created'])); ?>
                                    </div>
                                    <small class="text-muted">
                                        <?php echo date('g:i A', strtotime($contact['created'])); ?>
                                    </small>
                                </td>
                                <td class="status-cell">
                                    <?php if (!$isRead): ?>
                                        <span class="unread-badge">Unread</span>
                                    <?php else: ?>
                                        <span class="read-status">Read</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn-action btn-view" 
                                                data-bs-toggle="modal" 
                                                data-bs-target="#viewContactModal"
                                                data-contact-id="<?php echo $contact['id']; ?>"
                                                data-contact-name="<?php echo htmlspecialchars($fullNamePlain, ENT_QUOTES, 'UTF-8'); ?>"
                                                data-contact-email="<?php echo htmlspecialchars($emailData, ENT_QUOTES, 'UTF-8'); ?>"
                                                data-contact-phone="<?php echo htmlspecialchars($phoneData, ENT_QUOTES, 'UTF-8'); ?>"
                                                data-contact-created="<?php echo date('F j, Y g:i A', strtotime($contact['created'])); ?>"
                                                title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php if ($isRead): ?>
                                        <button class="btn-action btn-mark-unread" 
                                                data-contact-id="<?php echo $contact['id']; ?>"
                                                title="Mark as Unread">
                                            <i class="fas fa-eye-slash"></i>
                                        </button>
                                        <?php endif; ?>
                                        <a href="<?php echo url('/admin/contacts.php?action=delete&id=' . $contact['id'] . '&csrf_token=' . generate_csrf_token()); ?>" 
                                           class="btn-action btn-delete"
                                           onclick="return confirm('Are you sure you want to delete this contact request? This action cannot be undone.')"
                                           title="Delete Request">
                                            <i class="fas fa-trash-alt"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr data-empty="true">
                            <td colspan="7" class="text-center py-5">
                                <i class="fas fa-envelope-open fa-4x text-muted mb-3"></i>
                                <h5 class="text-muted">No contact requests</h5>
                                <p class="text-muted">All contact requests will appear here.</p>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- View Contact Modal -->
<div class="modal fade contact-modal" id="viewContactModal" tabindex="-1" aria-labelledby="viewContactModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewContactModalLabel">
                    <i class="fas fa-user-circle me-2"></i>Contact Request Details
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="contact-details">
                    <div class="detail-item">
                        <div class="detail-label">Name:</div>
                        <div class="detail-value" id="modal-contact-name"></div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Email:</div>
                        <div class="detail-value" id="modal-contact-email"></div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Phone:</div>
                        <div class="detail-value" id="modal-contact-phone"></div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Received:</div>
                        <div class="detail-value" id="modal-contact-created"></div>
                    </div>
                </div>
                
                <div class="message-content">
                    <h6 class="mb-3 text-muted"><i class="fas fa-comment me-2"></i>Message:</h6>
                    <p id="modal-contact-message" class="mb-0"></p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Close
                </button>
                <a href="#" class="btn btn-primary" id="modal-reply-btn" target="_blank">
                    <i class="fas fa-reply me-2"></i>Reply via Email
                </a>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Search functionality
    const searchInput = document.getElementById('contactSearch');
    const contactsTable = document.getElementById('contactsTable');
    
    if (searchInput && contactsTable) {
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase().trim();
            const rows = contactsTable.querySelectorAll('tbody tr:not([data-empty="true"])');
            
            rows.forEach(row => {
                const nameText = (row.cells[0]?.textContent || '').toLowerCase();
                const emailText = (row.cells[1]?.textContent || '').toLowerCase();
                const phoneText = (row.cells[2]?.textContent || '').toLowerCase();
                const messageText = (row.cells[3]?.textContent || '').toLowerCase();

                const shouldShow = searchTerm === '' || 
                    [nameText, emailText, phoneText, messageText].some(text => text.includes(searchTerm));
                
                row.style.display = shouldShow ? '' : 'none';
            });
        });
    }
    
    // Update counter with animation
    function updateCounter(element, newValue) {
        if (element) {
            element.textContent = newValue;
            element.classList.add('counter-update');
            setTimeout(() => {
                element.classList.remove('counter-update');
            }, 500);
        }
    }

    // Mark as read function
    function markContactAsRead(contactId, buttonEl) {
        const row = buttonEl.closest('tr');
        const statusCell = row ? row.querySelector('.status-cell') : null;
        const unreadCounter = document.getElementById('unread-contacts');
        
        // Check if already read
        if (statusCell && !statusCell.querySelector('.unread-badge')) {
            return;
        }
        
        // Update UI immediately
        if (statusCell) {
            statusCell.innerHTML = '<span class="read-status">Read</span>';
            statusCell.classList.add('text-success');
            setTimeout(() => statusCell.classList.remove('text-success'), 2000);
        }
        
        // Update unread counter
        if (unreadCounter) {
            const currentCount = parseInt(unreadCounter.textContent) || 0;
            if (currentCount > 0) {
                updateCounter(unreadCounter, currentCount - 1);
            }
        }
        
        // Show unread button if it doesn't exist
        const actionButtons = row.querySelector('.action-buttons');
        if (actionButtons && !actionButtons.querySelector('.btn-mark-unread')) {
            const unreadBtn = document.createElement('button');
            unreadBtn.className = 'btn-action btn-mark-unread';
            unreadBtn.setAttribute('data-contact-id', contactId);
            unreadBtn.setAttribute('title', 'Mark as Unread');
            unreadBtn.innerHTML = '<i class="fas fa-eye-slash"></i>';
            unreadBtn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                markContactAsUnread(contactId, this);
            });
            
            const deleteBtn = actionButtons.querySelector('.btn-delete');
            if (deleteBtn) {
                actionButtons.insertBefore(unreadBtn, deleteBtn);
            }
        }

        // Send AJAX request
        const csrfToken = '<?php echo generate_csrf_token(); ?>';
        
        fetch('', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'mark_read',
                id: contactId,
                csrf_token: csrfToken
            })
        })
        .then(response => response.json())
        .then(data => {
            if (!data.success) {
                // Revert changes on failure
                console.error('Failed to mark as read:', data.message);
                // Note: In a production app, you might want to revert UI changes here
            } else {
                // Visual feedback
                const icon = buttonEl.querySelector('i');
                if (icon) {
                    const originalClass = icon.className;
                    icon.className = 'fas fa-check';
                    setTimeout(() => {
                        icon.className = originalClass;
                    }, 1500);
                }
            }
        })
        .catch(error => {
            console.error('Error:', error);
            // Revert changes on error
        });
    }

    // Mark as unread function
    function markContactAsUnread(contactId, buttonEl) {
        const row = buttonEl.closest('tr');
        const statusCell = row ? row.querySelector('.status-cell') : null;
        const unreadCounter = document.getElementById('unread-contacts');
        
        // Update UI immediately
        if (statusCell) {
            statusCell.innerHTML = '<span class="unread-badge">Unread</span>';
            statusCell.classList.add('text-warning');
            setTimeout(() => statusCell.classList.remove('text-warning'), 2000);
        }
        
        // Update unread counter
        if (unreadCounter) {
            const currentCount = parseInt(unreadCounter.textContent) || 0;
            updateCounter(unreadCounter, currentCount + 1);
        }
        
        // Remove the unread button
        buttonEl.remove();

        // Send AJAX request
        const csrfToken = '<?php echo generate_csrf_token(); ?>';
        
        fetch('', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'mark_unread',
                id: contactId,
                csrf_token: csrfToken
            })
        })
        .then(response => response.json())
        .then(data => {
            if (!data.success) {
                console.error('Failed to mark as unread:', data.message);
                // Note: In a production app, you might want to revert UI changes here
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }

    // Add click handlers to view buttons
    document.addEventListener('click', function(e) {
        // View button click
        if (e.target.closest('.btn-view')) {
            const button = e.target.closest('.btn-view');
            const contactId = button.getAttribute('data-contact-id');
            
            // Mark as read when viewing (with slight delay to ensure modal opens)
            setTimeout(() => {
                markContactAsRead(contactId, button);
            }, 100);
        }
        
        // Mark as unread button click
        if (e.target.closest('.btn-mark-unread')) {
            const button = e.target.closest('.btn-mark-unread');
            const contactId = button.getAttribute('data-contact-id');
            markContactAsUnread(contactId, button);
        }
    });

    // View Contact Modal
    const viewContactModal = document.getElementById('viewContactModal');
    if (viewContactModal) {
        viewContactModal.addEventListener('show.bs.modal', async function(event) {
            const button = event.relatedTarget;
            const contactId = button.getAttribute('data-contact-id');
            const name = button.getAttribute('data-contact-name');
            const email = button.getAttribute('data-contact-email');
            const phone = button.getAttribute('data-contact-phone');
            const created = button.getAttribute('data-contact-created');

            // Update basic fields
            document.getElementById('modal-contact-name').textContent = name || '—';
            document.getElementById('modal-contact-email').textContent = email || '—';
            document.getElementById('modal-contact-phone').textContent = phone || 'N/A';
            document.getElementById('modal-contact-created').textContent = created || '—';

            // Update reply button
            const replyBtn = document.getElementById('modal-reply-btn');
            if (replyBtn && email) {
                const subject = encodeURIComponent('Re: Your Contact Request');
                const body = encodeURIComponent(`Dear ${name},\n\nThank you for your message. We have received your contact request and will get back to you shortly.\n\nBest regards,\n[Your Company Name]`);
                replyBtn.href = `mailto:${email}?subject=${subject}&body=${body}`;
            } else {
                replyBtn.style.display = 'none';
            }

            // Fetch full message
            try {
                const response = await fetch(`contacts.php?action=fetch&id=${encodeURIComponent(contactId)}`);
                const data = await response.json();
                
                if (data.success && data.data) {
                    document.getElementById('modal-contact-message').textContent = data.data.message || 'No message provided.';
                } else {
                    console.error('Fetch failed:', data.message);
                    document.getElementById('modal-contact-message').textContent = data.message || 'Could not load message content.';
                }
            } catch (error) {
                console.error('Error fetching message:', error);
                document.getElementById('modal-contact-message').textContent = 'Error loading message content.';
            }
        });

        // Reset reply button when modal hides
        viewContactModal.addEventListener('hidden.bs.modal', function() {
            const replyBtn = document.getElementById('modal-reply-btn');
            replyBtn.style.display = '';
            replyBtn.href = '#';
        });
    }
    
    // Add animation to table rows
    const tableRows = document.querySelectorAll('#contactsTable tbody tr');
    tableRows.forEach((row, index) => {
        row.style.animationDelay = `${index * 0.05}s`;
    });
    
    // Auto-focus search on page load
    if (searchInput) {
        setTimeout(() => {
            searchInput.focus();
        }, 500);
    }
});
</script>

<?php
$footer_file = __DIR__ . '/includes/footer.php';
if (file_exists($footer_file)) {
    require_once $footer_file;
} else {
    // Minimal fallback footer
    echo "</main></div><!-- .admin-container --></div><!-- .admin-layout -->";
    echo "<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js\"></script>";
    echo "</body></html>";
}
?>