<?php
/**
 * Admin Panel - Enhanced Blog Management System
 * Complete blog management with categories, tags, SEO, rich editor
 */

// Enable detailed error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Load centralized database connection
require_once __DIR__ . '/db.php';
try {
    $test_stmt = $db->query("SHOW TABLES LIKE 'blogs'");
    if (!$test_stmt->fetch()) {
        die('<div class="alert alert-danger m-4" role="alert">
            <h4 class="alert-heading">❌ Blogs Table Not Found</h4>
            <p>The blogs table does not exist in the database. Please run the database migration script:</p>
            <p><strong>Navigate to:</strong> <a href="' . url('/admin/update_blog_schema.php') . '" class="alert-link">update_blog_schema.php</a></p>
        </div>');
    }
    
    error_log('Database and table check passed');
} catch (Exception $e) {
    error_log('Initialization error: ' . $e->getMessage());
    die('<div class="alert alert-danger m-4" role="alert">
        <h4 class="alert-heading">❌ Database Connection Failed</h4>
        <p>Error: ' . htmlspecialchars($e->getMessage()) . '</p>
        <p><strong>DB_HOST:</strong> ' . DB_HOST . '</p>
        <p><strong>DB_NAME:</strong> ' . DB_NAME . '</p>
        <p><strong>DB_USER:</strong> ' . DB_USER . '</p>
    </div>');
}

// Debug helper function
function debug_blog_data($data) {
    error_log('=== BLOG DATA DEBUG ===');
    error_log('Title: ' . ($data['title'] ?? 'NULL'));
    error_log('Slug: ' . ($data['slug'] ?? 'NULL'));
    error_log('Content length: ' . strlen($data['content'] ?? ''));
    error_log('Image: ' . ($data['image'] ?? 'NULL'));
    error_log('Category: ' . ($data['category'] ?? 'NULL'));
    error_log('Status: ' . ($data['status'] ?? 'NULL'));
    error_log('Featured: ' . ($data['featured'] ?? 'NULL'));
    error_log('=== END DEBUG ===');
}

$message = '';
$error = '';

// Helper function to generate slug
function generate_slug($title, $db, $post_id = null) {
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title), '-'));
    $slug = substr($slug, 0, 200);
    
    // Make slug unique
    $counter = 1;
    $original_slug = $slug;
    while (true) {
        $check = $db->prepare("SELECT id FROM blogs WHERE slug = ?" . ($post_id ? " AND id != ?" : ""));
        $params = $post_id ? [$slug, $post_id] : [$slug];
        $check->execute($params);
        if (!$check->fetch()) break;
        $slug = $original_slug . '-' . $counter++;
    }
    
    return $slug;
}

// Handle blog actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CRITICAL DEBUG: Log that we received a POST request
    error_log('========================================');
    error_log('BLOG POST REQUEST RECEIVED AT: ' . date('Y-m-d H:i:s'));
    error_log('REQUEST_METHOD: ' . $_SERVER['REQUEST_METHOD']);
    error_log('POST EMPTY?: ' . (empty($_POST) ? 'YES' : 'NO'));
    error_log('POST count: ' . count($_POST));
    error_log('========================================');
    error_log('POST Data: ' . print_r($_POST, true));
    error_log('FILES Data: ' . print_r($_FILES, true));
    error_log('========================================');
    
    // Temporarily disable CSRF check for debugging
    $csrf_valid = true;
    if (function_exists('verify_csrf_token')) {
        $csrf_valid = isset($_POST['csrf_token']) && verify_csrf_token($_POST['csrf_token']);
        error_log('CSRF validation: ' . ($csrf_valid ? 'PASSED' : 'FAILED'));
    } else {
        error_log('CSRF function not available - skipping validation');
    }
    
    if (!$csrf_valid) {
        $error = 'Invalid CSRF token. Please try again.';
        error_log('CSRF validation failed');
    } else {
        $action = $_POST['action'] ?? '';
        $post_id = $_POST['post_id'] ?? null;
        
        error_log("Action: $action");
        error_log("Post ID: " . ($post_id ?? 'NULL'));
        
        // Get form data with proper sanitization
        $title = isset($_POST['title']) ? trim($_POST['title']) : '';
        $content = isset($_POST['content']) ? $_POST['content'] : ''; // Don't sanitize HTML content
        
        // Generate or use provided slug - ensure uniqueness
        if (!empty($_POST['slug'])) {
            $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $_POST['slug']), '-'));
            // Check if manually entered slug is unique
            $counter = 1;
            $original_slug = $slug;
            while (true) {
                $check = $db->prepare("SELECT id FROM blogs WHERE slug = ?" . ($post_id ? " AND id != ?" : ""));
                $params = $post_id ? [$slug, $post_id] : [$slug];
                $check->execute($params);
                if (!$check->fetch()) break;
                $slug = $original_slug . '-' . $counter++;
            }
        } else {
            $slug = generate_slug($title, $db, $post_id);
        }
        
        // Sanitize other fields
        $excerpt = isset($_POST['excerpt']) ? trim($_POST['excerpt']) : '';
        $category = isset($_POST['category']) ? trim($_POST['category']) : 'General';
        $tags = isset($_POST['tags']) ? trim($_POST['tags']) : '';
        $author = isset($_POST['author']) ? trim($_POST['author']) : 'Admin';
        $meta_description = isset($_POST['meta_description']) ? trim($_POST['meta_description']) : '';
        $meta_keywords = isset($_POST['meta_keywords']) ? trim($_POST['meta_keywords']) : '';
        $status = isset($_POST['status']) ? $_POST['status'] : 'draft';
        $featured = isset($_POST['featured']) ? 1 : 0;
        $image_filename = '';

        // Debug output
        error_log('Blog POST received - Action: ' . $action);
        error_log('Title: ' . $title);
        error_log('Slug: ' . $slug);
        error_log('Content length: ' . strlen($content));
        error_log('Category: ' . $category);
        error_log('Status: ' . $status);
        error_log('Featured: ' . $featured);

        if ($db) {
            // Handle image upload with enhanced debugging
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = BASE_PATH . DS . 'assets' . DS . 'images' . DS . 'blog_images' . DS;
                
                error_log("Upload directory: $upload_dir");
                error_log("Directory exists: " . (is_dir($upload_dir) ? 'Yes' : 'No'));
                error_log("Directory writable: " . (is_writable($upload_dir) ? 'Yes' : 'No'));
                
                if (!is_dir($upload_dir)) {
                    if (!mkdir($upload_dir, 0777, true)) {
                        $error = 'Failed to create blog image directory. Check permissions.';
                        error_log('Directory creation failed: ' . $upload_dir);
                        error_log('Parent directory writable: ' . (is_writable(dirname($upload_dir)) ? 'Yes' : 'No'));
                    } else {
                        chmod($upload_dir, 0777);
                        error_log('Directory created successfully: ' . $upload_dir);
                    }
                }

                if (empty($error)) {
                    $temp_name = $_FILES['image']['tmp_name'];
                    $original_name = basename($_FILES['image']['name']);
                    $safe_filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $original_name);
                    $target_path = $upload_dir . $safe_filename;
                    
                    error_log("Temp file: $temp_name");
                    error_log("Original name: $original_name");
                    error_log("Safe filename: $safe_filename");
                    error_log("Target path: $target_path");
                    
                    // Check file type
                    $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                    $file_type = mime_content_type($temp_name);
                    error_log("File type: $file_type");
                    
                    if (!in_array($file_type, $allowed_types)) {
                        $error = 'Invalid file type. Only JPG, PNG, GIF, and WebP are allowed.';
                        error_log("Invalid file type: $file_type");
                    } elseif (!move_uploaded_file($temp_name, $target_path)) {
                        $error = 'Failed to upload image. Check permissions.';
                        error_log('Failed to move uploaded file. Temp: ' . $temp_name . ' Target: ' . $target_path);
                        error_log('Upload directory writable: ' . (is_writable($upload_dir) ? 'Yes' : 'No'));
                    } else {
                        $image_filename = 'assets/images/blog_images/' . $safe_filename;
                        error_log('Image uploaded successfully: ' . $image_filename);
                    }
                }
            } elseif ($action === 'add') {
                // For new posts, image is required
                if (!isset($_FILES['image']) || $_FILES['image']['error'] === UPLOAD_ERR_NO_FILE) {
                    $error = 'Featured image is required for new blog posts.';
                    error_log('No image uploaded for new post');
                } else {
                    error_log('File upload error code: ' . $_FILES['image']['error']);
                }
            }

            if (empty($error)) {
                if ($action === 'add') {
                    error_log('=== ADD BLOG POST VALIDATION ===');
                    
                    if (empty($title)) {
                        $error = 'Title is required.';
                        error_log('Validation failed: Title is empty');
                    } elseif (empty($content)) {
                        $error = 'Content is required.';
                        error_log('Validation failed: Content is empty');
                    } elseif (empty($image_filename)) {
                        $error = 'Featured image is required.';
                        error_log('Validation failed: No image uploaded');
                    } else {
                        error_log('Validation passed. Proceeding with INSERT...');
                        
                        // Debug data before insert
                        debug_blog_data([
                            'title' => $title,
                            'slug' => $slug,
                            'content' => $content,
                            'image' => $image_filename,
                            'category' => $category,
                            'status' => $status,
                            'featured' => $featured
                        ]);
                        
                        try {
                            $uuid = bin2hex(random_bytes(16));
                            error_log("Generated UUID: $uuid");
                            
                            $sql = "INSERT INTO blogs (uuid, title, slug, content, excerpt, category, tags, author, meta_description, meta_keywords, image, status, featured) 
                                    VALUES (:uuid, :title, :slug, :content, :excerpt, :category, :tags, :author, :meta_description, :meta_keywords, :image, :status, :featured)";
                            
                            $stmt = $db->prepare($sql);
                            
                            $params = [
                                ':uuid' => $uuid,
                                ':title' => $title,
                                ':slug' => $slug,
                                ':content' => $content,
                                ':excerpt' => $excerpt,
                                ':category' => $category,
                                ':tags' => $tags,
                                ':author' => $author,
                                ':meta_description' => $meta_description,
                                ':meta_keywords' => $meta_keywords,
                                ':image' => $image_filename,
                                ':status' => $status,
                                ':featured' => $featured
                            ];
                            
                            error_log('SQL: ' . $sql);
                            error_log('Executing INSERT with params: ' . print_r($params, true));
                            
                            $result = $stmt->execute($params);
                            
                            if ($result) {
                                $insert_id = $db->lastInsertId();
                                $row_count = $stmt->rowCount();
                                error_log('Execute result: ' . ($result ? 'TRUE' : 'FALSE'));
                                error_log('Insert ID: ' . $insert_id);
                                error_log('Rows affected: ' . $row_count);
                                
                                if ($row_count > 0) {
                                    $_SESSION['blog_message'] = 'Blog post added successfully!';
                                    error_log('SUCCESS: Blog post inserted successfully with ID: ' . $insert_id);
                                    error_log('Redirecting to: ' . url('/admin/blogs.php?success=added'));
                                    header('Location: ' . url('/admin/blogs.php?success=added'));
                                    exit;
                                } else {
                                    $error = 'No rows were inserted. Check if data is identical to existing records.';
                                    error_log('WARNING: INSERT executed but no rows affected');
                                }
                            } else {
                                $error_info = $stmt->errorInfo();
                                $error = 'SQL Error: ' . ($error_info[2] ?? 'Unknown error');
                                error_log('ERROR: INSERT failed');
                                error_log('Error Info: ' . print_r($error_info, true));
                            }
                        } catch (PDOException $e) {
                            $error = 'Database error: ' . $e->getMessage();
                            error_log('EXCEPTION: Add blog post error');
                            error_log('Exception message: ' . $e->getMessage());
                            error_log('SQL State: ' . $e->getCode());
                            error_log('Error Info: ' . print_r($e->errorInfo ?? [], true));
                        }
                    }
                } elseif ($action === 'edit') {
                    if (!$post_id) {
                        $error = 'Post ID is missing for editing.';
                    } else {
                        $update_sql = "UPDATE blogs SET title = :title, slug = :slug, content = :content, excerpt = :excerpt, category = :category, tags = :tags, author = :author, meta_description = :meta_description, meta_keywords = :meta_keywords, status = :status, featured = :featured";
                        if (!empty($image_filename)) {
                            $update_sql .= ", image = :image";
                        }
                        $update_sql .= " WHERE id = :id";

                        try {
                            $stmt = $db->prepare($update_sql);
                            $params = [
                                ':title' => $title,
                                ':slug' => $slug,
                                ':content' => $content,
                                ':excerpt' => $excerpt,
                                ':category' => $category,
                                ':tags' => $tags,
                                ':author' => $author,
                                ':meta_description' => $meta_description,
                                ':meta_keywords' => $meta_keywords,
                                ':status' => $status,
                                ':featured' => $featured,
                                ':id' => $post_id
                            ];
                            if (!empty($image_filename)) {
                                $params[':image'] = $image_filename;
                            }
                            $stmt->execute($params);
                            $message = 'Blog post updated successfully!';
                            header('Location: ' . url('/admin/blogs.php?success=updated'));
                            exit;
                        } catch (PDOException $e) {
                            error_log('Edit blog post error: ' . $e->getMessage());
                            $error = 'Error updating blog post: ' . $e->getMessage();
                        }
                    }
                }
            }
        } else {
            $error = 'Database connection failed.';
        }
    }
}

// Handle GET requests
$action = $_GET['action'] ?? '';
$post_id_get = $_GET['id'] ?? null;

if ($action === 'delete') {
    if (!$post_id_get) {
        $error = 'Post ID is missing for deletion.';
    } elseif ($db) {
        try {
            $stmt_img = $db->prepare("SELECT image FROM blogs WHERE id = :id");
            $stmt_img->bindParam(':id', $post_id_get);
            $stmt_img->execute();
            $blog_image_path = $stmt_img->fetchColumn();

            $stmt = $db->prepare("DELETE FROM blogs WHERE id = :id");
            $stmt->bindParam(':id', $post_id_get);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $message = 'Blog post deleted successfully!';
                if ($blog_image_path) {
                    $full_image_path = BASE_PATH . DS . $blog_image_path;
                    if (file_exists($full_image_path)) {
                        unlink($full_image_path);
                    }
                }
            } else {
                $error = 'Blog post not found or could not be deleted.';
            }
            header('Location: ' . url('/admin/blogs.php'));
            exit;
        } catch (PDOException $e) {
            error_log('Delete blog post error: ' . $e->getMessage());
            $error = 'Error deleting blog post. Please try again later.';
        }
    } else {
        $error = 'Database connection failed.';
    }
}

// Fetch blogs with all fields
$blogs = [];
$total_posts = 0;
$published_posts = 0;
$draft_posts = 0;
$categories = [];

if ($db) {
    try {
        $stmt = $db->query("SELECT id, title, slug, excerpt, category, tags, author, image, status, featured, views, created, updated FROM blogs ORDER BY created DESC");
        $blogs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get statistics
        $total_posts = count($blogs);
        $published_posts = count(array_filter($blogs, fn($b) => $b['status'] === 'published'));
        $draft_posts = $total_posts - $published_posts;
        
        // Get unique categories
        $cats = array_unique(array_column($blogs, 'category'));
        $categories = array_filter($cats);
        
    } catch (PDOException $e) {
        error_log('Blog list fetch error: ' . $e->getMessage());
        $error = 'Error fetching blog posts: ' . $e->getMessage();
    }
}

// Success messages from redirects or session
if (isset($_GET['success'])) {
    if ($_GET['success'] === 'added') $message = 'Blog post added successfully!';
    if ($_GET['success'] === 'updated') $message = 'Blog post updated successfully!';
    if ($_GET['success'] === 'deleted') $message = 'Blog post deleted successfully!';
}

// Check for session message
if (isset($_SESSION['blog_message'])) {
    $message = $_SESSION['blog_message'];
    unset($_SESSION['blog_message']);
}

// Display debug info if error
if (!empty($error)) {
    error_log('Blog error displayed: ' . $error);
}

// Include header after all redirects are processed
$admin_header = __DIR__ . '/includes/header.php';
if (file_exists($admin_header)) {
    require_once $admin_header;
} else {
    // Minimal header without authentication for testing
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Blog Management - <?php echo SITE_NAME; ?></title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
        <style>
            body { background: #f5f7fa; padding: 20px; }
            .container { max-width: 1400px; }
        </style>
    </head>
    <body>
    <div class="container">
    <?php
}
?>

<div class="container-fluid py-4">
    <?php if (!empty($message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <?php if ($action === 'add' || $action === 'edit'): ?>
        <?php
        $post_data = [
            'id' => '', 
            'title' => '', 
            'slug' => '',
            'content' => '', 
            'excerpt' => '',
            'category' => '',
            'tags' => '',
            'author' => '',
            'meta_description' => '',
            'meta_keywords' => '',
            'image' => '', 
            'status' => 'draft',
            'featured' => 0
        ];
        if ($action === 'edit' && $post_id_get && $db) {
            try {
                $stmt = $db->prepare("SELECT id, title, slug, content, excerpt, category, tags, author, meta_description, meta_keywords, image, status, featured FROM blogs WHERE id = :id");
                $stmt->bindParam(':id', $post_id_get);
                $stmt->execute();
                $fetched_data = $stmt->fetch(PDO::FETCH_ASSOC);
                if (!$fetched_data) {
                    $error = 'Blog post not found.';
                    $action = '';
                } else {
                    $post_data = $fetched_data;
                }
            } catch (PDOException $e) {
                error_log('Fetch blog post for edit error: ' . $e->getMessage());
                $error = 'Error fetching blog post data.';
            }
        }
        ?>
        
        <div class="card">
            <div class="card-header">
                <h3><i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'edit'; ?> me-2"></i><?php echo ucfirst($action); ?> Blog Post</h3>
            </div>
            <div class="card-body">
                <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST" enctype="multipart/form-data" id="blogForm">
                    <?php echo csrf_token_field(); ?>
                    <input type="hidden" name="action" value="<?php echo $action; ?>">
                    <input type="hidden" name="post_id" value="<?php echo $post_data['id']; ?>">
                    
                    <div class="row">
                        <!-- Title -->
                        <div class="col-12 mb-3">
                            <label for="title" class="form-label">Title <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="title" name="title" 
                                   value="<?php echo htmlspecialchars($post_data['title']); ?>" required>
                        </div>
                        
                        <!-- Slug -->
                        <div class="col-12 mb-3">
                            <label for="slug" class="form-label">Slug (URL)</label>
                            <input type="text" class="form-control" id="slug" name="slug" 
                                   value="<?php echo htmlspecialchars($post_data['slug']); ?>" 
                                   placeholder="Auto-generated from title">
                            <small class="text-muted">Leave empty to auto-generate from title</small>
                        </div>
                        
                        <!-- Excerpt -->
                        <div class="col-12 mb-3">
                            <label for="excerpt" class="form-label">Excerpt</label>
                            <textarea class="form-control" id="excerpt" name="excerpt" rows="3" 
                                      placeholder="Brief summary for blog listing pages"><?php echo htmlspecialchars($post_data['excerpt']); ?></textarea>
                        </div>
                        
                        <!-- Content with TinyMCE -->
                        <div class="col-12 mb-3">
                            <label for="content" class="form-label">Content <span class="text-danger">*</span></label>
                            <textarea class="form-control tinymce-editor" id="content" name="content" rows="15"><?php echo htmlspecialchars($post_data['content'] ?? ''); ?></textarea>
                        </div>
                        
                        <!-- Category and Author -->
                        <div class="col-md-6 mb-3">
                            <label for="category" class="form-label">Category</label>
                            <select class="form-select" id="category" name="category">
                                <option value="">Select Category</option>
                                <option value="General" <?php echo ($post_data['category'] === 'General') ? 'selected' : ''; ?>>General</option>
                                <option value="Advices" <?php echo ($post_data['category'] === 'Advices') ? 'selected' : ''; ?>>Advices</option>
                                <option value="Announcements" <?php echo ($post_data['category'] === 'Announcements') ? 'selected' : ''; ?>>Announcements</option>
                                <option value="News" <?php echo ($post_data['category'] === 'News') ? 'selected' : ''; ?>>News</option>
                                <option value="Consultation" <?php echo ($post_data['category'] === 'Consultation') ? 'selected' : ''; ?>>Consultation</option>
                                <option value="Development" <?php echo ($post_data['category'] === 'Development') ? 'selected' : ''; ?>>Development</option>
                            </select>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="author" class="form-label">Author</label>
                            <input type="text" class="form-control" id="author" name="author" 
                                   value="<?php echo htmlspecialchars($post_data['author'] ?? ''); ?>" 
                                   placeholder="Author name">
                        </div>
                        
                        <!-- Tags -->
                        <div class="col-12 mb-3">
                            <label for="tags" class="form-label">Tags</label>
                            <input type="text" class="form-control" id="tags" name="tags" 
                                   value="<?php echo htmlspecialchars($post_data['tags'] ?? ''); ?>" 
                                   placeholder="Separate tags with commas (e.g., hosting, cloud, vps)">
                            <small class="text-muted">Comma-separated tags for better SEO and organization</small>
                        </div>
                        
                        <!-- SEO Meta Description -->
                        <div class="col-12 mb-3">
                            <label for="meta_description" class="form-label">Meta Description (SEO)</label>
                            <textarea class="form-control" id="meta_description" name="meta_description" rows="2" 
                                      maxlength="160" placeholder="Description for search engines (max 160 characters)"><?php echo htmlspecialchars($post_data['meta_description'] ?? ''); ?></textarea>
                            <small class="text-muted"><span id="metaDescLength">0</span>/160 characters</small>
                        </div>
                        
                        <!-- SEO Meta Keywords -->
                        <div class="col-12 mb-3">
                            <label for="meta_keywords" class="form-label">Meta Keywords (SEO)</label>
                            <input type="text" class="form-control" id="meta_keywords" name="meta_keywords" 
                                   value="<?php echo htmlspecialchars($post_data['meta_keywords'] ?? ''); ?>" 
                                   placeholder="Keywords for search engines (comma-separated)">
                        </div>
                        
                        <!-- Status and Featured -->
                        <div class="col-md-6 mb-3">
                            <label for="status" class="form-label">Status <span class="text-danger">*</span></label>
                            <select class="form-select" id="status" name="status" required>
                                <option value="draft" <?php echo ($post_data['status'] === 'draft') ? 'selected' : ''; ?>>Draft</option>
                                <option value="published" <?php echo ($post_data['status'] === 'published') ? 'selected' : ''; ?>>Published</option>
                            </select>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="image" class="form-label">Featured Image <?php echo ($action === 'add') ? '<span class="text-danger">*</span>' : ''; ?></label>
                            <input type="file" class="form-control" id="image" name="image" accept="image/*" 
                                   <?php echo ($action === 'add') ? 'required' : ''; ?>>
                        </div>
                        
                        <!-- Featured Post Checkbox -->
                        <div class="col-12 mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="featured" name="featured" value="1" 
                                       <?php echo (!empty($post_data['featured']) && $post_data['featured'] == 1) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="featured">
                                    <i class="fas fa-star text-warning"></i> Mark as Featured Post
                                </label>
                                <small class="d-block text-muted mt-1">Featured posts are highlighted on the blog page</small>
                            </div>
                        </div>
                        
                        <?php if ($action === 'edit' && !empty($post_data['image'])): ?>
                            <div class="col-12 mb-3">
                                <div class="image-preview">
                                    <p class="text-muted mb-2">Current Featured Image:</p>
                                    <img src="<?php echo url($post_data['image']); ?>" alt="Current Featured Image" class="img-fluid" style="max-width: 400px; border-radius: 8px;">
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="d-flex gap-2 mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'save'; ?> me-2"></i>
                            <?php echo ucfirst($action); ?> Post
                        </button>
                        <a href="<?php echo url('/admin/blogs.php'); ?>" class="btn btn-secondary">
                            <i class="fas fa-arrow-left me-2"></i>Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- TinyMCE Rich Text Editor Integration -->
        <script src="https://cdn.tiny.cloud/1/xkpqcqm28wbbbhooen28ahwo7ix2dy4e9k62i1enx5p5d078/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>
        <script>
            // Initialize TinyMCE
            tinymce.init({
                selector: '.tinymce-editor',
                height: 500,
                menubar: true,
                plugins: [
                    'advlist', 'autolink', 'lists', 'link', 'image', 'charmap', 'preview',
                    'anchor', 'searchreplace', 'visualblocks', 'code', 'fullscreen',
                    'insertdatetime', 'media', 'table', 'help', 'wordcount'
                ],
                toolbar: 'undo redo | blocks | bold italic underline strikethrough | ' +
                    'alignleft aligncenter alignright alignjustify | ' +
                    'bullist numlist outdent indent | forecolor backcolor | ' +
                    'link image media | removeformat code | help',
                content_style: 'body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif; font-size: 14px; }',
                image_advtab: true,
                file_picker_types: 'image',
                automatic_uploads: true,
                images_upload_url: 'upload_image.php',
                relative_urls: false,
                remove_script_host: false
            });
            
            // Auto-generate slug from title
            document.getElementById('title').addEventListener('input', function() {
                const slugField = document.getElementById('slug');
                if (!slugField.value || slugField.dataset.autoGenerated) {
                    const slug = generateSlug(this.value);
                    slugField.value = slug;
                    slugField.dataset.autoGenerated = 'true';
                }
            });
            
            // Mark slug as manually edited
            document.getElementById('slug').addEventListener('input', function() {
                if (this.value) {
                    delete this.dataset.autoGenerated;
                }
            });
            
            // Slug generator function
            function generateSlug(text) {
                return text.toString().toLowerCase()
                    .trim()
                    .replace(/\s+/g, '-')
                    .replace(/[^\w\-]+/g, '')
                    .replace(/\-\-+/g, '-')
                    .replace(/^-+/, '')
                    .replace(/-+$/, '');
            }
            
            // Meta description character counter
            document.getElementById('meta_description').addEventListener('input', function() {
                document.getElementById('metaDescLength').textContent = this.value.length;
            });
            
            // Initialize character counter on page load
            document.addEventListener('DOMContentLoaded', function() {
                const metaDesc = document.getElementById('meta_description');
                document.getElementById('metaDescLength').textContent = metaDesc.value.length;
            });
            
            // Form validation before submit
            const blogForm = document.getElementById('blogForm');
            if (blogForm) {
                blogForm.addEventListener('submit', function(e) {
                    // Get TinyMCE content
                    const content = tinymce.get('content').getContent();
                    
                    // Validate title
                    const title = document.getElementById('title').value.trim();
                    if (!title) {
                        e.preventDefault();
                        alert('Please enter a blog title');
                        document.getElementById('title').focus();
                        return false;
                    }
                    
                    // Validate content
                    if (!content || content.trim() === '' || content === '<p></p>' || content === '<p><br></p>') {
                        e.preventDefault();
                        alert('Please enter blog content');
                        tinymce.get('content').focus();
                        return false;
                    }
                    
                    // All validation passed
                    return true;
                });
            }
        </script>
        
    <?php else: ?>
        <!-- Statistics Cards -->
        <div class="row stats-cards">
            <div class="col-md-4">
                <div class="card stat-card total">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h3 class="stat-number"><?php echo $total_posts; ?></h3>
                                <p class="stat-label">Total Posts</p>
                            </div>
                            <i class="fas fa-file-alt fa-2x text-muted opacity-25"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card stat-card published">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h3 class="stat-number"><?php echo $published_posts; ?></h3>
                                <p class="stat-label">Published</p>
                            </div>
                            <i class="fas fa-check-circle fa-2x text-muted opacity-25"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card stat-card draft">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h3 class="stat-number"><?php echo $draft_posts; ?></h3>
                                <p class="stat-label">Drafts</p>
                            </div>
                            <i class="fas fa-edit fa-2x text-muted opacity-25"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Blog Posts Grid -->
        <div class="card">
            <div class="card-header">
                <h3><i class="fas fa-blog me-2"></i>Blog Management</h3>
                <div class="table-actions">
                    <div class="input-group search-box">
                        <span class="input-group-text"><i class="fas fa-search"></i></span>
                        <input type="text" id="blogSearch" class="form-control" placeholder="Search blog posts...">
                    </div>
                    <a href="<?php echo url('/admin/blogs.php?action=add'); ?>" class="btn btn-primary">
                        <i class="fas fa-plus"></i> New Post
                    </a>
                </div>
            </div>
            
            <div class="card-body">
                <div class="row blog-grid" id="blogGrid">
                    <?php if (!empty($blogs)): ?>
                        <?php foreach ($blogs as $blog): ?>
                            <div class="col-md-4 mb-4">
                                <div class="blog-card" data-title="<?php echo strtolower(htmlspecialchars($blog['title'])); ?>">
                                    <?php if (!empty($blog['image'])): ?>
                                        <img src="<?php echo url($blog['image']); ?>" alt="<?php echo htmlspecialchars($blog['title']); ?>" class="blog-image">
                                    <?php else: ?>
                                        <div class="blog-image-placeholder">
                                            <i class="fas fa-image fa-3x"></i>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="blog-content">
                                        <h3 class="blog-title"><?php echo htmlspecialchars($blog['title']); ?></h3>
                                        
                                        <div class="blog-meta">
                                            <span class="blog-date">
                                                <?php echo date('M j, Y', strtotime($blog['created'])); ?>
                                            </span>
                                            <span class="badge <?php echo $blog['status'] === 'published' ? 'bg-success' : 'bg-warning'; ?>">
                                                <?php echo ucfirst($blog['status']); ?>
                                            </span>
                                        </div>
                                        
                                        <div class="blog-actions">
                                            <a href="<?php echo url('/admin/blogs.php?action=edit&id=' . $blog['id']); ?>" 
                                               class="btn btn-sm btn-primary">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>
                                            <a href="<?php echo url('/admin/blogs.php?action=delete&id=' . $blog['id']); ?>" 
                                               class="btn btn-sm btn-danger"
                                               onclick="return confirm('Are you sure you want to delete this blog post?')">
                                                <i class="fas fa-trash-alt"></i> Delete
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="col-12 text-center py-5">
                            <i class="fas fa-blog fa-4x text-muted mb-3"></i>
                            <h5 class="text-muted">No blog posts yet</h5>
                            <p class="text-muted">Get started by creating your first blog post.</p>
                            <a href="<?php echo url('/admin/blogs.php?action=add'); ?>" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i>Create First Post
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Search functionality
    const searchInput = document.getElementById('blogSearch');
    const blogGrid = document.getElementById('blogGrid');
    
    if (searchInput && blogGrid) {
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const blogCards = blogGrid.getElementsByClassName('blog-card');
            
            for (let card of blogCards) {
                const title = card.getAttribute('data-title');
                const shouldShow = title.includes(searchTerm);
                card.closest('.col-md-4').style.display = shouldShow ? '' : 'none';
            }
        });
    }
    
    // Image preview for form
    const imageInput = document.getElementById('image');
    if (imageInput) {
        imageInput.addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    // Remove existing preview
                    const existingPreview = document.querySelector('.image-preview');
                    if (existingPreview) {
                        existingPreview.remove();
                    }
                    
                    // Create new preview
                    const previewDiv = document.createElement('div');
                    previewDiv.className = 'image-preview';
                    previewDiv.innerHTML = `
                        <p class="text-muted mb-2">Image Preview:</p>
                        <img src="${e.target.result}" alt="Preview" class="img-fluid">
                    `;
                    
                    imageInput.parentNode.appendChild(previewDiv);
                };
                reader.readAsDataURL(file);
            }
        });
    }
    
    // Add animation to blog cards
    const blogCards = document.querySelectorAll('.blog-card');
    blogCards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.1}s`;
    });
    
    // Auto-focus search on page load
    if (searchInput) {
        setTimeout(() => {
            searchInput.focus();
        }, 500);
    }
});
</script>

<?php
$footer_file = __DIR__ . '/includes/footer.php';
if (file_exists($footer_file)) {
    require_once $footer_file;
} else {
    // Minimal fallback footer to avoid fatal errors if include is missing
    echo "</main></div><!-- .admin-container --></div><!-- .admin-layout -->";
    echo "<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js\"></script>";
}
?>
