<?php
/**
 * Migration Runner Script
 * Applies and rolls back database migrations.
 */

require_once __DIR__ . '/../config.php'; // Load configuration and database connection

if (php_sapi_name() !== 'cli') {
    die("This script can only be run from the command line.");
}

$db = get_db();
if (!$db) {
    echo "Error: Could not connect to the database.\n";
    exit(1);
}

// Create migrations table if it doesn't exist
$db->exec("CREATE TABLE IF NOT EXISTS `migrations` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
    `migration` VARCHAR(255) NOT NULL UNIQUE,
    `batch` INT NOT NULL,
    `migrated_at` DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;");

function get_migrated_migrations(PDO $db): array {
    $stmt = $db->query("SELECT migration FROM migrations ORDER BY batch ASC, migration ASC");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

function get_all_migration_files(string $migrations_path): array {
    $files = glob($migrations_path . '/*.php');
    $migrations = [];
    foreach ($files as $file) {
        $migrations[] = basename($file, '.php');
    }
    sort($migrations);
    return $migrations;
}

function run_migration(PDO $db, string $migration_file, string $migrations_path, int $batch, string $direction = 'up') {
    require_once $migrations_path . DS . $migration_file . '.php';
    if (function_exists($direction)) {
        echo "Running {$direction} migration: {$migration_file}...\n";
        try {
            $db->beginTransaction();
            $direction($db);
            if ($direction === 'up') {
                $stmt = $db->prepare("INSERT INTO migrations (migration, batch) VALUES (:migration, :batch)");
                $stmt->execute([':migration' => $migration_file, ':batch' => $batch]);
            } else { // down
                $stmt = $db->prepare("DELETE FROM migrations WHERE migration = :migration");
                $stmt->execute([':migration' => $migration_file]);
            }
            $db->commit();
            echo "Successfully {$direction}graded: {$migration_file}\n";
        } catch (PDOException $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            echo "Error {$direction}grading migration {$migration_file}: " . $e->getMessage() . "\n";
            exit(1);
        }
    } else {
        echo "Error: Function '{$direction}' not found in migration file {$migration_file}.php\n";
        exit(1);
    }
}

$command = $argv[1] ?? 'status';
$migrations_path = BASE_PATH . DS . 'scripts' . DS . 'migrations';

switch ($command) {
    case 'migrate':
        $migrated = get_migrated_migrations($db);
        $all_migrations = get_all_migration_files($migrations_path);
        $pending_migrations = array_diff($all_migrations, $migrated);

        if (empty($pending_migrations)) {
            echo "No pending migrations.\n";
            break;
        }

        $next_batch = $db->query("SELECT MAX(batch) FROM migrations")->fetchColumn() + 1;
        foreach ($pending_migrations as $migration) {
            run_migration($db, $migration, $migrations_path, $next_batch, 'up');
        }
        echo "Migrations complete.\n";
        break;

    case 'rollback':
        $batch_to_rollback = $argv[2] ?? null;
        $latest_batch = $db->query("SELECT MAX(batch) FROM migrations")->fetchColumn();

        if ($latest_batch === false || $latest_batch === null) {
            echo "No migrations to roll back.\n";
            break;
        }

        $target_batch = $batch_to_rollback ? (int)$batch_to_rollback : $latest_batch;

        $stmt = $db->prepare("SELECT migration FROM migrations WHERE batch = :batch ORDER BY migration DESC");
        $stmt->execute([':batch' => $target_batch]);
        $migrations_in_batch = $stmt->fetchAll(PDO::FETCH_COLUMN);

        if (empty($migrations_in_batch)) {
            echo "No migrations found for batch {$target_batch}.\n";
            break;
        }

        foreach ($migrations_in_batch as $migration) {
            run_migration($db, $migration, $migrations_path, $target_batch, 'down');
        }
        echo "Rollback of batch {$target_batch} complete.\n";
        break;

    case 'status':
        $migrated = get_migrated_migrations($db);
        $all_migrations = get_all_migration_files($migrations_path);

        echo "Migration Status:\n";
        echo "------------------\n";

        foreach ($all_migrations as $migration) {
            $status = in_array($migration, $migrated) ? '[x] Migrated' : '[ ] Pending';
            echo "{$status}: {$migration}\n";
        }
        break;

    default:
        echo "Usage: php scripts/migrate.php [migrate|rollback [batch]|status]\n";
        exit(1);
}
