<?php
// Ensure config.php is loaded to get environment variables and other core settings
if (!defined('BASE_PATH')) {
    require_once __DIR__ . '/../config.php';
}

// Database configuration
// IMPORTANT: In production, these values should be loaded from environment variables
// or a secure, non-version-controlled configuration file (e.g., .env)
define('DB_HOST', $_ENV['DB_HOST'] ?? '127.0.0.1');
define('DB_NAME', $_ENV['DB_NAME'] ?? 'wsplcloud');
define('DB_USER', $_ENV['DB_USER'] ?? 'root');
define('DB_PASS', $_ENV['DB_PASSWORD'] ?? ''); // Use DB_PASSWORD from env

/**
 * Return a singleton PDO connection. Creates database if necessary when possible.
 * @return \PDO|null
 */
function get_db() {
    static $pdo = null;
    if ($pdo instanceof PDO) return $pdo;

    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
    try {
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        // try creating database if it doesn't exist
        // In production, do not attempt to create the database automatically.
        // The database should be created and migrated manually or via deployment scripts.
        if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
            try {
                $tmp = new PDO('mysql:host=' . DB_HOST . ';charset=utf8mb4', DB_USER, DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
                $tmp->exec("CREATE DATABASE IF NOT EXISTS `" . str_replace('`','',DB_NAME) . "` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;");
                // reconnect
                $pdo = new PDO($dsn, DB_USER, DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION, PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC]);
                return $pdo;
            } catch (PDOException $e2) {
                error_log('DB connection error during auto-creation attempt: ' . $e2->getMessage());
                return null;
            }
        } else {
            error_log('DB connection error in production: ' . $e->getMessage());
            return null;
        }
    }
}

// Get the PDO database connection
$link = get_db();

// Check if connection was successful
if ($link === null) {
    // get_db() already logs errors, so we just need to terminate
    die("ERROR: Could not connect to the database.");
}

/**
 * Helper function to get database connection
 * 
 * @return PDO Database connection object
 */
function getDatabaseConnection() {
    global $link;
    return $link;
}

/**
 * Helper function to check if database connection is active
 * 
 * @return bool True if connected, false otherwise
 */
function isDatabaseConnected() {
    global $link;
    try {
        if ($link instanceof PDO) {
            $link->query('SELECT 1');
            return true;
        }
    } catch (PDOException $e) {
        error_log('Database connection check failed: ' . $e->getMessage());
    }
    return false;
}
