<?php
/**
 * Admin Panel - Dashboard
 * Main entry point for the admin area.
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['admin_user'])) {
    header('Location: ' . (function_exists('url') ? url('admin/login.php') : SITE_ROOT . '/admin/login.php'));
    exit;
}

// Include header
require_once __DIR__ . '/includes/header.php';

// Fetch data for dashboard widgets
$user_count = 0;
$blog_count = 0;
$contact_count = 0;
$offer_count = 0;
$domain_count = 0;
$newsletter_count = 0;
$published_blogs = 0;
$draft_blogs = 0;
$active_offers = 0;
$today_contacts = 0;

if ($db) {
    try {
        // Count users
        $user_stmt = $db->query("SELECT COUNT(*) FROM users");
        $user_count = $user_stmt->fetchColumn();

        // Count blog posts and status breakdown
        $blog_stmt = $db->query("SELECT COUNT(*) FROM blogs");
        $blog_count = $blog_stmt->fetchColumn();
        
        $published_stmt = $db->query("SELECT COUNT(*) FROM blogs WHERE status = 'published'");
        $published_blogs = $published_stmt->fetchColumn();
        
        $draft_stmt = $db->query("SELECT COUNT(*) FROM blogs WHERE status = 'draft'");
        $draft_blogs = $draft_stmt->fetchColumn();

        // Count contact requests
        $contact_stmt = $db->query("SELECT COUNT(*) FROM contacts");
        $contact_count = $contact_stmt->fetchColumn();
        
        // Today's contacts
        try {
            $today_stmt = $db->query("SELECT COUNT(*) FROM contacts WHERE DATE(created) = CURDATE()");
            $today_contacts = $today_stmt->fetchColumn();
        } catch (PDOException $e) {
            $today_contacts = 0;
        }

        // Count offers
        try {
            $offer_stmt = $db->query("SELECT COUNT(*) FROM offers");
            $offer_count = $offer_stmt->fetchColumn();
            
            $active_offer_stmt = $db->query("SELECT COUNT(*) FROM offers WHERE status = 'active'");
            $active_offers = $active_offer_stmt->fetchColumn();
        } catch (PDOException $e) {
            $offer_count = 0;
            $active_offers = 0;
        }
        
        // Count domains
        try {
            $domain_stmt = $db->query("SELECT COUNT(*) FROM domains WHERE is_active = 1");
            $domain_count = $domain_stmt->fetchColumn();
        } catch (PDOException $e) {
            $domain_count = 0;
        }
        
        // Count newsletter subscribers
        try {
            $newsletter_stmt = $db->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE status = 'subscribed'");
            $newsletter_count = $newsletter_stmt->fetchColumn();
        } catch (PDOException $e) {
            $newsletter_count = 0;
        }

    } catch (PDOException $e) {
        error_log('Dashboard data fetch error: ' . $e->getMessage());
    }
}
?>

<style>
:root {
    --primary-gradient: linear-gradient(135deg, #0000ff 0%, #0000cc 100%);
    --success-gradient: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
    --warning-gradient: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    --info-gradient: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    --purple-gradient: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%);
    --orange-gradient: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
}

.dashboard-container {
    animation: fadeInUp 0.6s ease-out;
    padding-bottom: 3rem;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.dashboard-header {
    background: linear-gradient(135deg, #0000ff 0%, #0000ff 50%, #0000ff 100%);
    border-radius: 30px;
    padding: 3rem;
    margin-bottom: 3rem;
    border: 3px solid #FFD700;
    color: white;
    position: relative;
    overflow: hidden;
}

.dashboard-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255, 215, 0, 0.2) 0%, transparent 70%);
    animation: pulse 15s ease-in-out infinite;
}

.dashboard-header::after {
    content: '';
    position: absolute;
    bottom: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #FFD700 0%, #FFA500 50%, #FFD700 100%);
    animation: shimmer 3s linear infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 0.5; }
    50% { transform: scale(1.1); opacity: 0.3; }
}

.dashboard-header h1 {
    color: white !important;
    font-weight: 800;
    font-size: 2.8rem;
    margin-bottom: 0.5rem;
    text-shadow: 0 2px 20px rgba(0, 0, 0, 0.3);
    position: relative;
    z-index: 1;
}

.dashboard-header p {
    color: rgba(255, 255, 255, 0.95) !important;
    font-size: 1.15rem;
    position: relative;
    z-index: 1;
}

.welcome-card {
    background: linear-gradient(135deg, rgba(255, 215, 0, 0.3), rgba(255, 165, 0, 0.3));
    backdrop-filter: blur(10px);
    border: 2px solid rgba(255, 215, 0, 0.5);
    color: white;
    border-radius: 20px;
    padding: 1.8rem;
    position: relative;
    z-index: 1;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
    gap: 2rem;
    margin-bottom: 3rem;
}

.stat-card {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    border: 1px solid rgba(0, 0, 0, 0.08);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    cursor: pointer;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: var(--primary-gradient);
    transition: height 0.3s ease;
}

.stat-card:hover::before {
    height: 8px;
    background: linear-gradient(90deg, #0000ff 0%, #FFD700 100%);
}

.stat-card:hover {
    transform: translateY(-10px) scale(1.02);
}

.stat-card.users::before { background: var(--primary-gradient); }
.stat-card.blogs::before { background: var(--primary-gradient); }
.stat-card.contacts::before { background: var(--primary-gradient); }
.stat-card.offers::before { background: var(--primary-gradient); }
.stat-card.domains::before { background: var(--primary-gradient); }
.stat-card.newsletter::before { background: var(--primary-gradient); }

.stat-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.stat-info h3 {
    font-size: 3rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    background: var(--primary-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    line-height: 1;
}

.stat-card.blogs h3 { background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
.stat-card.contacts h3 { background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
.stat-card.offers h3 { background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
.stat-card.domains h3 { background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }
.stat-card.newsletter h3 { background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent; }

.stat-info p {
    color: #6c757d;
    font-weight: 600;
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 1px;
    font-size: 0.85rem;
}

.stat-badge {
    display: inline-block;
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    margin-top: 0.5rem;
    background: #FFD700;
    color: #000000;
}

.stat-icon {
    width: 70px;
    height: 70px;
    border-radius: 18px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    background: var(--primary-gradient);
    color: white;
    transition: all 0.4s cubic-bezier(0.68, -0.55, 0.265, 1.55);
}

.stat-card.blogs .stat-icon { background: var(--primary-gradient); }
.stat-card.contacts .stat-icon { background: var(--primary-gradient); }
.stat-card.offers .stat-icon { background: var(--primary-gradient); }
.stat-card.domains .stat-icon { background: var(--primary-gradient); }
.stat-card.newsletter .stat-icon { background: var(--primary-gradient); }

.stat-card:hover .stat-icon {
    transform: scale(1.15) rotate(8deg);
}

.stat-action {
    margin-top: 1.5rem;
    padding-top: 1rem;
    border-top: 1px solid rgba(0, 0, 0, 0.1);
}

.btn-stat {
    background: transparent;
    border: 2px solid var(--primary-blue);
    color: var(--primary-blue);
    padding: 0.5rem 1.5rem;
    border-radius: 10px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-stat:hover {
    background: var(--primary-blue);
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px var(--blue-glow);
}

.stat-card.blogs .btn-stat { border-color: var(--info-color); color: var(--info-color); }
.stat-card.blogs .btn-stat:hover { background: var(--info-color); }
.stat-card.contacts .btn-stat { border-color: var(--warning-color); color: var(--warning-color); }
.stat-card.contacts .btn-stat:hover { background: var(--warning-color); }
.stat-card.offers .btn-stat { border-color: var(--success-color); color: var(--success-color); }
.stat-card.offers .btn-stat:hover { background: var(--success-color); }

.activity-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.activity-card {
    background: white;
    border-radius: 20px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid rgba(0, 0, 0, 0.05);
    overflow: hidden;
    animation: slideInUp 0.6s ease-out;
}

.activity-header {
    background: linear-gradient(135deg, var(--primary-blue), var(--blue-light));
    color: white;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.activity-header h5 {
    margin: 0;
    font-weight: 600;
}

.activity-body {
    padding: 1.5rem;
}

.activity-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem 0;
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
    transition: background-color 0.3s ease;
}

.activity-item:last-child {
    border-bottom: none;
}

.activity-item:hover {
    background: rgba(0, 0, 255, 0.02);
    border-radius: 10px;
    padding: 1rem;
    margin: 0 -1rem;
}

.activity-avatar {
    width: 45px;
    height: 45px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    color: white;
    flex-shrink: 0;
}

.avatar-primary { background: linear-gradient(135deg, var(--primary-blue), var(--blue-light)); }
.avatar-success { background: linear-gradient(135deg, var(--success-color), #10b981); }
.avatar-warning { background: linear-gradient(135deg, var(--warning-color), #f59e0b); }

.activity-content {
    flex: 1;
}

.activity-title {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.25rem;
}

.activity-meta {
    color: #6c757d;
    font-size: 0.85rem;
}

.activity-time {
    color: #9ca3af;
    font-size: 0.8rem;
    white-space: nowrap;
}

.quick-actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-top: 2rem;
}

.quick-action {
    background: white;
    border-radius: 20px;
    padding: 2.5rem 1.5rem;
    text-align: center;
    text-decoration: none;
    color: #2c3e50;
    border: 2px solid #e0e0e0;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1rem;
}

.quick-action:hover {
    transform: translateY(-5px);
    border-color: #FFD700;
    color: #0000ff;
}

.quick-action i {
    font-size: 3rem;
    color: #0000ff;
    transition: all 0.3s ease;
}

.quick-action:hover i {
    transform: scale(1.1);
}

.quick-action span {
    font-weight: 700;
    font-size: 1rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(50px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive */
@media (max-width: 768px) {
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .activity-grid {
        grid-template-columns: 1fr;
    }
    
    .quick-actions-grid {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .dashboard-header {
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }
    
    .stat-card {
        padding: 1.5rem;
    }
    
    .stat-content {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .stat-icon {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }
}

@media (max-width: 480px) {
    .quick-actions-grid {
        grid-template-columns: 1fr;
    }
    
    .activity-item {
        flex-direction: column;
        text-align: center;
        gap: 0.75rem;
    }
    
    .stat-info h3 {
        font-size: 2rem;
    }
}
</style>

<div class="dashboard-container">
    <!-- Dashboard Header -->
    <div class="dashboard-header">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1 class="h2 mb-2">Welcome back, <?php echo htmlspecialchars($_SESSION['admin_user']['username']); ?>! 👋</h1>
                <p class="text-muted mb-0">Here's what's happening with your site today.</p>
            </div>
            <div class="col-md-4 text-md-end">
                <div class="welcome-card">
                    <div class="row align-items-center">
                        <div class="col">
                            <small class="opacity-75">Last Login</small>
                            <div class="fw-semibold"><?php echo date('M j, Y g:i A'); ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-calendar-check fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics Grid -->
    <div class="stats-grid">
        <div class="stat-card users">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $user_count; ?></h3>
                    <p>Total Users</p>
                    <span class="stat-badge"><i class="fas fa-arrow-up"></i> Active</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-users"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/users.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> Manage
                </a>
            </div>
        </div>

        <div class="stat-card blogs">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $blog_count; ?></h3>
                    <p>Blog Posts</p>
                    <span class="stat-badge"><?php echo $published_blogs; ?> Published</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-blog"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/blogs.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> Manage
                </a>
            </div>
        </div>

        <div class="stat-card contacts">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $contact_count; ?></h3>
                    <p>Contact Requests</p>
                    <span class="stat-badge"><?php echo $today_contacts; ?> Today</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-envelope"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/contacts.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> View
                </a>
            </div>
        </div>

        <div class="stat-card offers">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $offer_count; ?></h3>
                    <p>Special Offers</p>
                    <span class="stat-badge"><?php echo $active_offers; ?> Active</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-tags"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/offers.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> Manage
                </a>
            </div>
        </div>
        
        <div class="stat-card domains">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $domain_count; ?></h3>
                    <p>Domain TLDs</p>
                    <span class="stat-badge"><i class="fas fa-check"></i> Active</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-globe"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/domains.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> Manage
                </a>
            </div>
        </div>
        
        <div class="stat-card newsletter">
            <div class="stat-content">
                <div class="stat-info">
                    <h3><?php echo $newsletter_count; ?></h3>
                    <p>Subscribers</p>
                    <span class="stat-badge"><i class="fas fa-users"></i> Growing</span>
                </div>
                <div class="stat-icon">
                    <i class="fas fa-paper-plane"></i>
                </div>
            </div>
            <div class="stat-action">
                <a href="<?php echo url('/admin/newsletter.php'); ?>" class="btn-stat">
                    <i class="fas fa-arrow-right"></i> View
                </a>
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="activity-card">
        <div class="activity-header" style="color: #0000ff;">
            <i class="fas fa-bolt"></i>
            <h5>Quick Actions</h5>
        </div>
        <div class="activity-body">
            <div class="quick-actions-grid">
                <a href="<?php echo url('/admin/users.php?action=add'); ?>" class="quick-action">
                    <i class="fas fa-user-plus"></i>
                    <span>Add User</span>
                </a>
                <a href="<?php echo url('/admin/blogs.php?action=add'); ?>" class="quick-action">
                    <i class="fas fa-plus-circle"></i>
                    <span>New Post</span>
                </a>
                <a href="<?php echo url('/admin/offers.php?action=add'); ?>" class="quick-action">
                    <i class="fas fa-tag"></i>
                    <span>Create Offer</span>
                </a>
                <a href="<?php echo url('/admin/settings.php'); ?>" class="quick-action">
                    <i class="fas fa-cog"></i>
                    <span>Settings</span>
                </a>
                <a href="<?php echo url('/admin/profile.php'); ?>" class="quick-action">
                    <i class="fas fa-user-cog"></i>
                    <span>Profile</span>
                </a>
                <a href="<?php echo url('/admin/contacts.php'); ?>" class="quick-action">
                    <i class="fas fa-envelope-open"></i>
                    <span>View Messages</span>
                </a>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Add animation delays to stat cards
    const statCards = document.querySelectorAll('.stat-card');
    statCards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.1}s`;
        card.classList.add('animate__animated', 'animate__fadeInUp');
    });

    // Add animation to activity cards
    const activityCards = document.querySelectorAll('.activity-card');
    activityCards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.15}s`;
    });

    // Add hover effects to quick actions
    const quickActions = document.querySelectorAll('.quick-action');
    quickActions.forEach(action => {
        action.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px) scale(1.02)';
        });
        action.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });

    // Add click animation to stat cards
    statCards.forEach(card => {
        card.addEventListener('click', function() {
            this.style.transform = 'translateY(-2px) scale(1.02)';
            setTimeout(() => {
                this.style.transform = 'translateY(-8px) scale(1)';
            }, 150);
        });
    });

    // Real-time clock update
    function updateClock() {
        const now = new Date();
        const timeString = now.toLocaleTimeString('en-US', { 
            hour: '2-digit', 
            minute: '2-digit',
            hour12: true 
        });
        const dateString = now.toLocaleDateString('en-US', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        
        // You could update a clock element if you add one
        // document.getElementById('currentTime').textContent = timeString;
        // document.getElementById('currentDate').textContent = dateString;
    }

    setInterval(updateClock, 60000);
    updateClock();
});
</script>

<?php
require_once __DIR__ . '/includes/footer.php';
?>