<?php
/**
 * Admin Panel - Hosting Plans Management
 * Complete CRUD interface for managing all hosting plans (Shared, VPS, Cloud, Reseller, WordPress)
 */

// Load centralized database connection
require_once __DIR__ . '/db.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['admin_user'])) {
    header('Location: ' . url('admin/login.php'));
    exit;
}

// Helper functions - only define if not already defined in config.php
if (!function_exists('sanitize_input')) {
    function sanitize_input($data) {
        if (is_array($data)) {
            return array_map('sanitize_input', $data);
        }
        return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
    }
}

if (!function_exists('generate_csrf_token')) {
    function generate_csrf_token() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
}

if (!function_exists('verify_csrf_token')) {
    function verify_csrf_token($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

// Generate unique slug
function generate_plan_slug($name, $exclude_id = null) {
    global $db;
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $name)));
    $original_slug = $slug;
    $counter = 1;
    
    while (true) {
        $stmt = $db->prepare("SELECT id FROM hosting_plans WHERE slug = :slug" . ($exclude_id ? " AND id != :exclude_id" : ""));
        $stmt->bindParam(':slug', $slug);
        if ($exclude_id) {
            $stmt->bindParam(':exclude_id', $exclude_id, PDO::PARAM_INT);
        }
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            break;
        }
        $slug = $original_slug . '-' . $counter;
        $counter++;
    }
    
    return $slug;
}

// Initialize variables
$message = '';
$error = '';
$page_action = isset($_GET['action']) ? sanitize_input($_GET['action']) : 'list';
$edit_plan = null;

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid CSRF token. Please try again.';
    } else {
        $action = sanitize_input($_POST['action']);
        
        if ($action === 'add' || $action === 'edit') {
            // Validate required fields
            $plan_type = sanitize_input($_POST['plan_type'] ?? '');
            $plan_name = sanitize_input($_POST['plan_name'] ?? '');
            $description = sanitize_input($_POST['description'] ?? '');
            $monthly_price = floatval($_POST['monthly_price'] ?? 0);
            $yearly_price = floatval($_POST['yearly_price'] ?? 0);
            $storage = sanitize_input($_POST['storage'] ?? '');
            $bandwidth = sanitize_input($_POST['bandwidth'] ?? '');
            $domains = sanitize_input($_POST['domains'] ?? '');
            $databases = sanitize_input($_POST['databases'] ?? '');
            $emails = sanitize_input($_POST['emails'] ?? '');
            $backup = sanitize_input($_POST['backup'] ?? '');
            $display_order = intval($_POST['display_order'] ?? 0);
            $status = sanitize_input($_POST['status'] ?? 'active');
            
            // Parse JSON fields
            $features = $_POST['features'] ?? '[]';
            $developer_features = $_POST['developer_features'] ?? '[]';
            $technical_details = $_POST['technical_details'] ?? '[]';
            
            // Validate JSON
            json_decode($features);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $error = 'Invalid JSON format in Features field.';
            }
            json_decode($developer_features);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $error = 'Invalid JSON format in Developer Features field.';
            }
            json_decode($technical_details);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $error = 'Invalid JSON format in Technical Details field.';
            }
            
            if (empty($plan_name)) {
                $error = 'Plan name is required.';
            }
            
            if (empty($error)) {
                // Handle icon image upload
                $icon_image = '';
                if (isset($_FILES['icon_image']) && $_FILES['icon_image']['error'] === UPLOAD_ERR_OK) {
                    $upload_dir = BASE_PATH . DS . 'assets' . DS . 'images' . DS . 'icons' . DS . 'hosting-plans' . DS;
                    
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $file_tmp = $_FILES['icon_image']['tmp_name'];
                    $file_name = $_FILES['icon_image']['name'];
                    $file_size = $_FILES['icon_image']['size'];
                    $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                    
                    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
                    
                    if (!in_array($file_ext, $allowed_extensions)) {
                        $error = 'Invalid file type. Allowed: jpg, jpeg, png, gif, webp, svg';
                    } elseif ($file_size > 5242880) { // 5MB
                        $error = 'File size too large. Maximum 5MB allowed.';
                    } else {
                        $new_filename = time() . '_' . bin2hex(random_bytes(8)) . '.' . $file_ext;
                        $upload_path = $upload_dir . $new_filename;
                        
                        if (move_uploaded_file($file_tmp, $upload_path)) {
                            $icon_image = 'images/icons/hosting-plans/' . $new_filename;
                        } else {
                            $error = 'Failed to upload icon image.';
                        }
                    }
                }
                
                if (empty($error)) {
                    try {
                        if ($action === 'add') {
                            // Generate slug
                            $slug = generate_plan_slug($plan_name);
                            
                            // Insert new plan
                            $stmt = $db->prepare("INSERT INTO hosting_plans 
                                (plan_type, plan_name, slug, description, icon_image, monthly_price, yearly_price, 
                                storage, bandwidth, domains, `databases`, emails, backup, features, developer_features, 
                                technical_details, display_order, status) 
                                VALUES 
                                (:plan_type, :plan_name, :slug, :description, :icon_image, :monthly_price, :yearly_price,
                                :storage, :bandwidth, :domains, :databases, :emails, :backup, :features, :developer_features,
                                :technical_details, :display_order, :status)");
                            
                            $stmt->bindParam(':plan_type', $plan_type);
                            $stmt->bindParam(':plan_name', $plan_name);
                            $stmt->bindParam(':slug', $slug);
                            $stmt->bindParam(':description', $description);
                            $stmt->bindParam(':icon_image', $icon_image);
                            $stmt->bindParam(':monthly_price', $monthly_price);
                            $stmt->bindParam(':yearly_price', $yearly_price);
                            $stmt->bindParam(':storage', $storage);
                            $stmt->bindParam(':bandwidth', $bandwidth);
                            $stmt->bindParam(':domains', $domains);
                            $stmt->bindParam(':databases', $databases);
                            $stmt->bindParam(':emails', $emails);
                            $stmt->bindParam(':backup', $backup);
                            $stmt->bindParam(':features', $features);
                            $stmt->bindParam(':developer_features', $developer_features);
                            $stmt->bindParam(':technical_details', $technical_details);
                            $stmt->bindParam(':display_order', $display_order);
                            $stmt->bindParam(':status', $status);
                            
                            $stmt->execute();
                            
                            $_SESSION['success_message'] = 'Hosting plan added successfully!';
                            header('Location: ' . url('/admin/hosting_plans.php'));
                            exit;
                            
                        } elseif ($action === 'edit') {
                            $plan_id = intval($_POST['plan_id']);
                            
                            // Get existing plan
                            $stmt = $db->prepare("SELECT * FROM hosting_plans WHERE id = :id");
                            $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
                            $stmt->execute();
                            $existing_plan = $stmt->fetch(PDO::FETCH_ASSOC);
                            
                            if (!$existing_plan) {
                                $error = 'Plan not found.';
                            } else {
                                // Generate slug if name changed
                                $slug = ($plan_name !== $existing_plan['plan_name']) 
                                    ? generate_plan_slug($plan_name, $plan_id) 
                                    : $existing_plan['slug'];
                                
                                // Use existing icon if no new one uploaded
                                if (empty($icon_image)) {
                                    $icon_image = $existing_plan['icon_image'];
                                } else {
                                    // Delete old icon if exists
                                    if (!empty($existing_plan['icon_image'])) {
                                        $old_icon_path = BASE_PATH . DS . 'assets' . DS . $existing_plan['icon_image'];
                                        if (file_exists($old_icon_path)) {
                                            unlink($old_icon_path);
                                        }
                                    }
                                }
                                
                                // Update plan
                                $stmt = $db->prepare("UPDATE hosting_plans SET 
                                    plan_type = :plan_type,
                                    plan_name = :plan_name,
                                    slug = :slug,
                                    description = :description,
                                    icon_image = :icon_image,
                                    monthly_price = :monthly_price,
                                    yearly_price = :yearly_price,
                                    storage = :storage,
                                    bandwidth = :bandwidth,
                                    domains = :domains,
                                    `databases` = :databases,
                                    emails = :emails,
                                    backup = :backup,
                                    features = :features,
                                    developer_features = :developer_features,
                                    technical_details = :technical_details,
                                    display_order = :display_order,
                                    status = :status
                                    WHERE id = :id");
                                
                                $stmt->bindParam(':plan_type', $plan_type);
                                $stmt->bindParam(':plan_name', $plan_name);
                                $stmt->bindParam(':slug', $slug);
                                $stmt->bindParam(':description', $description);
                                $stmt->bindParam(':icon_image', $icon_image);
                                $stmt->bindParam(':monthly_price', $monthly_price);
                                $stmt->bindParam(':yearly_price', $yearly_price);
                                $stmt->bindParam(':storage', $storage);
                                $stmt->bindParam(':bandwidth', $bandwidth);
                                $stmt->bindParam(':domains', $domains);
                                $stmt->bindParam(':databases', $databases);
                                $stmt->bindParam(':emails', $emails);
                                $stmt->bindParam(':backup', $backup);
                                $stmt->bindParam(':features', $features);
                                $stmt->bindParam(':developer_features', $developer_features);
                                $stmt->bindParam(':technical_details', $technical_details);
                                $stmt->bindParam(':display_order', $display_order);
                                $stmt->bindParam(':status', $status);
                                $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
                                
                                $stmt->execute();
                                
                                $_SESSION['success_message'] = 'Hosting plan updated successfully!';
                                header('Location: ' . url('/admin/hosting_plans.php'));
                                exit;
                            }
                        }
                    } catch (PDOException $e) {
                        error_log('Hosting plan save error: ' . $e->getMessage());
                        $error = 'Database error: ' . $e->getMessage();
                    }
                }
            }
        }
    }
}

// Handle delete action
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $plan_id = intval($_GET['id']);
    
    try {
        // Get plan to delete icon
        $stmt = $db->prepare("SELECT icon_image FROM hosting_plans WHERE id = :id");
        $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
        $stmt->execute();
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($plan) {
            // Delete icon file
            if (!empty($plan['icon_image'])) {
                $icon_path = BASE_PATH . DS . 'assets' . DS . $plan['icon_image'];
                if (file_exists($icon_path)) {
                    unlink($icon_path);
                }
            }
            
            // Delete from database
            $stmt = $db->prepare("DELETE FROM hosting_plans WHERE id = :id");
            $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
            $stmt->execute();
            
            $_SESSION['success_message'] = 'Hosting plan deleted successfully!';
        }
    } catch (PDOException $e) {
        error_log('Delete plan error: ' . $e->getMessage());
        $_SESSION['error_message'] = 'Error deleting plan.';
    }
    
    header('Location: ' . url('/admin/hosting_plans.php'));
    exit;
}

// Handle status toggle (AJAX)
if (isset($_GET['action']) && $_GET['action'] === 'toggle_status' && isset($_GET['id'])) {
    $plan_id = intval($_GET['id']);
    
    try {
        $stmt = $db->prepare("SELECT status FROM hosting_plans WHERE id = :id");
        $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
        $stmt->execute();
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($plan) {
            $new_status = ($plan['status'] === 'active') ? 'inactive' : 'active';
            $stmt = $db->prepare("UPDATE hosting_plans SET status = :status WHERE id = :id");
            $stmt->bindParam(':status', $new_status);
            $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
            $stmt->execute();
            
            echo json_encode(['success' => true, 'status' => $new_status]);
            exit;
        }
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// Check for session messages
if (isset($_SESSION['success_message'])) {
    $message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

// Load plan data for editing
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $plan_id = intval($_GET['id']);
    try {
        $stmt = $db->prepare("SELECT * FROM hosting_plans WHERE id = :id");
        $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
        $stmt->execute();
        $edit_plan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$edit_plan) {
            $_SESSION['error_message'] = 'Plan not found.';
            header('Location: ' . url('/admin/hosting_plans.php'));
            exit;
        }
    } catch (PDOException $e) {
        error_log('Fetch plan error: ' . $e->getMessage());
        $_SESSION['error_message'] = 'Error loading plan.';
        header('Location: ' . url('/admin/hosting_plans.php'));
        exit;
    }
}

// Fetch all plans with search and filters
$search = isset($_GET['search']) ? sanitize_input($_GET['search']) : '';
$type_filter = isset($_GET['type']) ? sanitize_input($_GET['type']) : '';
$status_filter = isset($_GET['status']) ? sanitize_input($_GET['status']) : '';

$plans = [];
$total_plans = 0;
$active_plans = 0;
$inactive_plans = 0;
$plans_by_type = [];

if ($db) {
    try {
        // Build query
        $query = "SELECT * FROM hosting_plans WHERE 1=1";
        $params = [];
        
        if (!empty($search)) {
            $query .= " AND (plan_name LIKE :search OR description LIKE :search OR plan_type LIKE :search)";
            $params[':search'] = '%' . $search . '%';
        }
        
        if (!empty($type_filter)) {
            $query .= " AND plan_type = :type";
            $params[':type'] = $type_filter;
        }
        
        if (!empty($status_filter) && in_array($status_filter, ['active', 'inactive'])) {
            $query .= " AND status = :status";
            $params[':status'] = $status_filter;
        }
        
        $query .= " ORDER BY display_order ASC, plan_type ASC, monthly_price ASC";
        
        $stmt = $db->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        $plans = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get statistics
        $stats_stmt = $db->query("SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive
            FROM hosting_plans");
        $stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Plans by type
        $plans_by_type = [];
        $type_stmt = $db->query("SELECT plan_type, COUNT(*) as count FROM hosting_plans GROUP BY plan_type");
        while ($row = $type_stmt->fetch(PDO::FETCH_ASSOC)) {
            $plans_by_type[$row['plan_type']] = $row['count'];
        }
        
    } catch (PDOException $e) {
        error_log('Fetch plans error: ' . $e->getMessage());
        $error = 'Error loading plans.';
    }
}

// Include header
require_once __DIR__ . '/includes/header.php';
?>

<style>
:root {
    --primary-blue: #0000ff;
    --blue-dark: #0000cc;
    --blue-light: #3366ff;
    --blue-glow: rgba(0, 0, 255, 0.2);
    --success-color: #00d97e;
    --warning-color: #ffc107;
    --danger-color: #e74c3c;
    --info-color: #17a2b8;
}

.hosting-plans-admin-container {
    animation: fadeInUp 0.6s ease-out;
    max-width: 100%;
    padding: 2rem;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.page-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid #e9ecef;
}

.page-header h1 {
    font-size: 2rem;
    color: #2c3e50;
    margin: 0;
}

.btn-primary-custom {
    background: var(--primary-blue);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.btn-primary-custom:hover {
    background: var(--blue-dark);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px var(--blue-glow);
    color: white;
}

.alert {
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    animation: slideInDown 0.4s ease-out;
}

@keyframes slideInDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.alert-success {
    background: #d4edda;
    color: #155724;
    border-left: 4px solid var(--success-color);
}

.alert-danger {
    background: #f8d7da;
    color: #721c24;
    border-left: 4px solid var(--danger-color);
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    position: relative;
    overflow: hidden;
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 20px rgba(0,0,0,0.12);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
}

.stat-card.total::before { background: var(--primary-blue); }
.stat-card.active::before { background: var(--success-color); }
.stat-card.inactive::before { background: var(--warning-color); }
.stat-card.shared::before { background: #667eea; }
.stat-card.vps::before { background: #f093fb; }
.stat-card.cloud::before { background: #4facfe; }
.stat-card.reseller::before { background: #43e97b; }
.stat-card.wordpress::before { background: #fa709a; }

.stat-card h3 {
    font-size: 2rem;
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
}

.stat-card p {
    margin: 0;
    color: #7f8c8d;
    font-size: 0.9rem;
}

.search-filter-bar {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    margin-bottom: 2rem;
}

.search-filter-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #2c3e50;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-control {
    padding: 0.75rem;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-blue);
    box-shadow: 0 0 0 3px var(--blue-glow);
}

.plans-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.plan-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.plan-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 20px rgba(0,0,0,0.15);
}

.plan-card.inactive {
    opacity: 0.6;
}

.plan-type-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: var(--primary-blue);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.plan-icon {
    width: 60px;
    height: 60px;
    margin-bottom: 1rem;
    object-fit: contain;
}

.plan-name {
    font-size: 1.5rem;
    font-weight: 700;
    color: #2c3e50;
    margin: 0 0 0.5rem 0;
}

.plan-description {
    color: #7f8c8d;
    font-size: 0.9rem;
    margin-bottom: 1rem;
}

.plan-price {
    font-size: 1.25rem;
    font-weight: 700;
    color: var(--primary-blue);
    margin-bottom: 1rem;
}

.plan-details {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 0.5rem;
    margin-bottom: 1rem;
    padding-top: 1rem;
    border-top: 1px solid #e9ecef;
}

.plan-detail-item {
    font-size: 0.85rem;
    color: #555;
}

.plan-detail-item strong {
    color: #2c3e50;
}

.plan-actions {
    display: flex;
    gap: 0.5rem;
    padding-top: 1rem;
    border-top: 1px solid #e9ecef;
}

.btn-action {
    flex: 1;
    padding: 0.5rem;
    border-radius: 6px;
    border: none;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.25rem;
    font-size: 0.85rem;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-edit {
    background: #3498db;
    color: white;
}

.btn-edit:hover {
    background: #2980b9;
    color: white;
}

.btn-toggle {
    background: #f39c12;
    color: white;
}

.btn-toggle:hover {
    background: #e67e22;
}

.btn-delete {
    background: #e74c3c;
    color: white;
}

.btn-delete:hover {
    background: #c0392b;
    color: white;
}

.form-container {
    background: white;
    padding: 2rem;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
}

.form-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-group-full {
    grid-column: 1 / -1;
}

textarea.form-control {
    min-height: 120px;
    font-family: monospace;
    font-size: 0.9rem;
}

.json-help-text {
    font-size: 0.85rem;
    color: #7f8c8d;
    margin-top: 0.25rem;
}

.btn-submit {
    background: var(--primary-blue);
    color: white;
    padding: 0.75rem 2rem;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.btn-submit:hover {
    background: var(--blue-dark);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px var(--blue-glow);
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: #7f8c8d;
}

.empty-state i {
    font-size: 4rem;
    margin-bottom: 1rem;
    color: #bdc3c7;
}

.empty-state h3 {
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-badge.active {
    background: #d4edda;
    color: #155724;
}

.status-badge.inactive {
    background: #f8d7da;
    color: #721c24;
}

.current-icon-preview {
    max-width: 100px;
    margin-top: 0.5rem;
    border-radius: 8px;
}
</style>

<div class="hosting-plans-admin-container">
    <!-- Page Header -->
    <div class="page-header">
        <h1>
            <i class="fas fa-server"></i> 
            <?php echo $page_action === 'add' ? 'Add New Hosting Plan' : ($page_action === 'edit' ? 'Edit Hosting Plan' : 'Hosting Plans Management'); ?>
        </h1>
        <?php if ($page_action === 'list'): ?>
            <a href="<?php echo url('/admin/hosting_plans.php?action=add'); ?>" class="btn-primary-custom">
                <i class="fas fa-plus"></i> Add New Plan
            </a>
        <?php else: ?>
            <a href="<?php echo url('/admin/hosting_plans.php'); ?>" class="btn-primary-custom">
                <i class="fas fa-arrow-left"></i> Back to List
            </a>
        <?php endif; ?>
    </div>

    <!-- Alerts -->
    <?php if (!empty($message)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <?php if (!empty($error)): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <?php if ($page_action === 'list'): ?>
        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card total">
                <h3><?php echo $stats['total'] ?? 0; ?></h3>
                <p>Total Plans</p>
            </div>
            <div class="stat-card active">
                <h3><?php echo $stats['active'] ?? 0; ?></h3>
                <p>Active Plans</p>
            </div>
            <div class="stat-card inactive">
                <h3><?php echo $stats['inactive'] ?? 0; ?></h3>
                <p>Inactive Plans</p>
            </div>
            <div class="stat-card shared">
                <h3><?php echo $plans_by_type['shared'] ?? 0; ?></h3>
                <p>Shared Hosting</p>
            </div>
            <div class="stat-card vps">
                <h3><?php echo $plans_by_type['vps'] ?? 0; ?></h3>
                <p>VPS Plans</p>
            </div>
            <div class="stat-card cloud">
                <h3><?php echo $plans_by_type['cloud'] ?? 0; ?></h3>
                <p>Cloud Plans</p>
            </div>
            <div class="stat-card reseller">
                <h3><?php echo $plans_by_type['reseller'] ?? 0; ?></h3>
                <p>Reseller Plans</p>
            </div>
            <div class="stat-card wordpress">
                <h3><?php echo $plans_by_type['wordpress'] ?? 0; ?></h3>
                <p>WordPress Plans</p>
            </div>
        </div>

        <!-- Search & Filter -->
        <div class="search-filter-bar">
            <form method="GET" action="<?php echo url('/admin/hosting_plans.php'); ?>">
                <div class="search-filter-grid">
                    <div class="form-group">
                        <label for="search"><i class="fas fa-search"></i> Search Plans</label>
                        <input 
                            type="text" 
                            id="search" 
                            name="search" 
                            class="form-control" 
                            placeholder="Search by name, type, or description..."
                            value="<?php echo htmlspecialchars($search); ?>"
                        >
                    </div>
                    <div class="form-group">
                        <label for="type"><i class="fas fa-filter"></i> Plan Type</label>
                        <select id="type" name="type" class="form-control">
                            <option value="">All Types</option>
                            <option value="shared" <?php echo $type_filter === 'shared' ? 'selected' : ''; ?>>Shared Hosting</option>
                            <option value="vps" <?php echo $type_filter === 'vps' ? 'selected' : ''; ?>>VPS Hosting</option>
                            <option value="cloud" <?php echo $type_filter === 'cloud' ? 'selected' : ''; ?>>Cloud Hosting</option>
                            <option value="reseller" <?php echo $type_filter === 'reseller' ? 'selected' : ''; ?>>Reseller Hosting</option>
                            <option value="wordpress" <?php echo $type_filter === 'wordpress' ? 'selected' : ''; ?>>WordPress Hosting</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="status"><i class="fas fa-toggle-on"></i> Status</label>
                        <select id="status" name="status" class="form-control">
                            <option value="">All Status</option>
                            <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                    <div class="form-group" style="justify-content: flex-end;">
                        <label>&nbsp;</label>
                        <button type="submit" class="btn-primary-custom">
                            <i class="fas fa-search"></i> Apply Filters
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Plans Grid -->
        <?php if (!empty($plans)): ?>
            <div class="plans-grid">
                <?php foreach ($plans as $plan): ?>
                    <div class="plan-card <?php echo $plan['status']; ?>">
                        <span class="plan-type-badge"><?php echo htmlspecialchars($plan['plan_type']); ?></span>
                        
                        <?php if (!empty($plan['icon_image'])): ?>
                            <img src="<?php echo asset($plan['icon_image']); ?>" alt="<?php echo htmlspecialchars($plan['plan_name']); ?>" class="plan-icon">
                        <?php else: ?>
                            <div class="plan-icon" style="background: #e9ecef; border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                                <i class="fas fa-server" style="font-size: 2rem; color: #bdc3c7;"></i>
                            </div>
                        <?php endif; ?>
                        
                        <h3 class="plan-name"><?php echo htmlspecialchars($plan['plan_name']); ?></h3>
                        
                        <?php if (!empty($plan['description'])): ?>
                            <p class="plan-description"><?php echo htmlspecialchars($plan['description']); ?></p>
                        <?php endif; ?>
                        
                        <div class="plan-price">
                            ₹<?php echo number_format($plan['monthly_price'], 2); ?>/month
                            <small style="font-size: 0.75rem; color: #7f8c8d; display: block;">
                                ₹<?php echo number_format($plan['yearly_price'], 2); ?>/year
                            </small>
                        </div>
                        
                        <div class="plan-details">
                            <?php if (!empty($plan['storage'])): ?>
                                <div class="plan-detail-item"><strong>Storage:</strong> <?php echo htmlspecialchars($plan['storage']); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($plan['bandwidth'])): ?>
                                <div class="plan-detail-item"><strong>Bandwidth:</strong> <?php echo htmlspecialchars($plan['bandwidth']); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($plan['domains'])): ?>
                                <div class="plan-detail-item"><strong>Domains:</strong> <?php echo htmlspecialchars($plan['domains']); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($plan['databases'])): ?>
                                <div class="plan-detail-item"><strong>Databases:</strong> <?php echo htmlspecialchars($plan['databases']); ?></div>
                            <?php endif; ?>
                        </div>
                        
                        <div style="margin-bottom: 1rem;">
                            <span class="status-badge <?php echo $plan['status']; ?>">
                                <?php echo htmlspecialchars($plan['status']); ?>
                            </span>
                        </div>
                        
                        <div class="plan-actions">
                            <a href="<?php echo url('/admin/hosting_plans.php?action=edit&id=' . $plan['id']); ?>" class="btn-action btn-edit">
                                <i class="fas fa-edit"></i> Edit
                            </a>
                            <button 
                                type="button" 
                                class="btn-action btn-toggle" 
                                onclick="toggleStatus(<?php echo $plan['id']; ?>, this)"
                                data-status="<?php echo $plan['status']; ?>"
                            >
                                <i class="fas fa-exchange-alt"></i> Toggle
                            </button>
                            <a 
                                href="<?php echo url('/admin/hosting_plans.php?action=delete&id=' . $plan['id']); ?>" 
                                class="btn-action btn-delete"
                                onclick="return confirm('Are you sure you want to delete this plan? This action cannot be undone.');"
                            >
                                <i class="fas fa-trash-alt"></i> Delete
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-server"></i>
                <h3>No hosting plans found</h3>
                <p><?php echo !empty($search) || !empty($type_filter) || !empty($status_filter) ? 'Try adjusting your search filters.' : 'Create your first hosting plan to get started!'; ?></p>
                <a href="<?php echo url('/admin/hosting_plans.php?action=add'); ?>" class="btn-primary-custom">
                    <i class="fas fa-plus"></i> Create First Plan
                </a>
            </div>
        <?php endif; ?>

    <?php else: ?>
        <!-- Add/Edit Form -->
        <div class="form-container">
            <form method="POST" action="<?php echo url('/admin/hosting_plans.php'); ?>" enctype="multipart/form-data" id="planForm">
                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                <input type="hidden" name="action" value="<?php echo $page_action; ?>">
                <?php if ($edit_plan): ?>
                    <input type="hidden" name="plan_id" value="<?php echo $edit_plan['id']; ?>">
                <?php endif; ?>

                <div class="form-row">
                    <div class="form-group">
                        <label for="plan_type">Plan Type <span style="color: red;">*</span></label>
                        <select id="plan_type" name="plan_type" class="form-control" required>
                            <option value="">Select Type</option>
                            <option value="shared" <?php echo ($edit_plan && $edit_plan['plan_type'] === 'shared') ? 'selected' : ''; ?>>Shared Hosting</option>
                            <option value="vps" <?php echo ($edit_plan && $edit_plan['plan_type'] === 'vps') ? 'selected' : ''; ?>>VPS Hosting</option>
                            <option value="cloud" <?php echo ($edit_plan && $edit_plan['plan_type'] === 'cloud') ? 'selected' : ''; ?>>Cloud Hosting</option>
                            <option value="reseller" <?php echo ($edit_plan && $edit_plan['plan_type'] === 'reseller') ? 'selected' : ''; ?>>Reseller Hosting</option>
                            <option value="wordpress" <?php echo ($edit_plan && $edit_plan['plan_type'] === 'wordpress') ? 'selected' : ''; ?>>WordPress Hosting</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="plan_name">Plan Name <span style="color: red;">*</span></label>
                        <input 
                            type="text" 
                            id="plan_name" 
                            name="plan_name" 
                            class="form-control" 
                            placeholder="e.g., Pearl, Starfish"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['plan_name']) : ''; ?>"
                            required
                        >
                    </div>

                    <div class="form-group">
                        <label for="monthly_price">Monthly Price (₹) <span style="color: red;">*</span></label>
                        <input 
                            type="number" 
                            id="monthly_price" 
                            name="monthly_price" 
                            class="form-control" 
                            step="0.01"
                            placeholder="30.00"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['monthly_price']) : ''; ?>"
                            required
                        >
                    </div>

                    <div class="form-group">
                        <label for="yearly_price">Yearly Price (₹) <span style="color: red;">*</span></label>
                        <input 
                            type="number" 
                            id="yearly_price" 
                            name="yearly_price" 
                            class="form-control" 
                            step="0.01"
                            placeholder="360.00"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['yearly_price']) : ''; ?>"
                            required
                        >
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group form-group-full">
                        <label for="description">Description</label>
                        <textarea 
                            id="description" 
                            name="description" 
                            class="form-control" 
                            rows="3"
                            placeholder="Brief description of the plan..."
                        ><?php echo $edit_plan ? htmlspecialchars($edit_plan['description']) : ''; ?></textarea>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="storage">Storage</label>
                        <input 
                            type="text" 
                            id="storage" 
                            name="storage" 
                            class="form-control" 
                            placeholder="e.g., 1GB SSD, 10GB SSD"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['storage']) : ''; ?>"
                        >
                    </div>

                    <div class="form-group">
                        <label for="bandwidth">Bandwidth</label>
                        <input 
                            type="text" 
                            id="bandwidth" 
                            name="bandwidth" 
                            class="form-control" 
                            placeholder="e.g., 100GB, Unlimited"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['bandwidth']) : ''; ?>"
                        >
                    </div>

                    <div class="form-group">
                        <label for="domains">Domains</label>
                        <input 
                            type="text" 
                            id="domains" 
                            name="domains" 
                            class="form-control" 
                            placeholder="e.g., Single, 5, Unlimited"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['domains']) : ''; ?>"
                        >
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="databases">Databases</label>
                        <input 
                            type="text" 
                            id="databases" 
                            name="databases" 
                            class="form-control" 
                            placeholder="e.g., 10, Unlimited"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['databases']) : ''; ?>"
                        >
                    </div>

                    <div class="form-group">
                        <label for="emails">Email Accounts</label>
                        <input 
                            type="text" 
                            id="emails" 
                            name="emails" 
                            class="form-control" 
                            placeholder="e.g., 5, Unlimited"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['emails']) : ''; ?>"
                        >
                    </div>

                    <div class="form-group">
                        <label for="backup">Backup</label>
                        <input 
                            type="text" 
                            id="backup" 
                            name="backup" 
                            class="form-control" 
                            placeholder="e.g., Weekly, Daily, No"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['backup']) : ''; ?>"
                        >
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="display_order">Display Order</label>
                        <input 
                            type="number" 
                            id="display_order" 
                            name="display_order" 
                            class="form-control" 
                            placeholder="0"
                            value="<?php echo $edit_plan ? htmlspecialchars($edit_plan['display_order']) : '0'; ?>"
                        >
                        <small class="json-help-text">Lower numbers appear first</small>
                    </div>

                    <div class="form-group">
                        <label for="status">Status</label>
                        <select id="status" name="status" class="form-control">
                            <option value="active" <?php echo ($edit_plan && $edit_plan['status'] === 'active') ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo ($edit_plan && $edit_plan['status'] === 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="icon_image">Plan Icon</label>
                        <input 
                            type="file" 
                            id="icon_image" 
                            name="icon_image" 
                            class="form-control" 
                            accept="image/*"
                        >
                        <small class="json-help-text">Max 5MB (jpg, png, gif, webp, svg)</small>
                        <?php if ($edit_plan && !empty($edit_plan['icon_image'])): ?>
                            <img src="<?php echo asset($edit_plan['icon_image']); ?>" alt="Current icon" class="current-icon-preview">
                        <?php endif; ?>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group form-group-full">
                        <label for="features">Features (JSON Array)</label>
                        <textarea 
                            id="features" 
                            name="features" 
                            class="form-control" 
                            placeholder='["1GB SSD Web Space", "100 GB Bandwidth", "Single Domain"]'
                        ><?php echo $edit_plan ? htmlspecialchars($edit_plan['features']) : '[]'; ?></textarea>
                        <small class="json-help-text">JSON array format: ["feature 1", "feature 2"]</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group form-group-full">
                        <label for="developer_features">Developer Features (JSON Array)</label>
                        <textarea 
                            id="developer_features" 
                            name="developer_features" 
                            class="form-control" 
                            placeholder='[{"name": "PHP", "available": true}, {"name": "Node.js", "available": false}]'
                        ><?php echo $edit_plan ? htmlspecialchars($edit_plan['developer_features']) : '[]'; ?></textarea>
                        <small class="json-help-text">JSON array of objects: [{"name": "Feature", "available": true/false}]</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group form-group-full">
                        <label for="technical_details">Technical Details (JSON Array)</label>
                        <textarea 
                            id="technical_details" 
                            name="technical_details" 
                            class="form-control" 
                            placeholder='[{"name": "Subdomains", "value": "10", "available": true}]'
                        ><?php echo $edit_plan ? htmlspecialchars($edit_plan['technical_details']) : '[]'; ?></textarea>
                        <small class="json-help-text">JSON array of objects: [{"name": "Detail", "value": "10", "available": true/false}]</small>
                    </div>
                </div>

                <div class="form-row">
                    <button type="submit" class="btn-submit">
                        <i class="fas fa-save"></i> <?php echo $page_action === 'add' ? 'Create Plan' : 'Update Plan'; ?>
                    </button>
                </div>
            </form>
        </div>
    <?php endif; ?>
</div>

<script>
function toggleStatus(planId, button) {
    if (!confirm('Are you sure you want to toggle the status of this plan?')) {
        return;
    }
    
    const originalText = button.innerHTML;
    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
    button.disabled = true;
    
    fetch('<?php echo url('/admin/hosting_plans.php'); ?>?action=toggle_status&id=' + planId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error: ' + (data.error || 'Unknown error'));
                button.innerHTML = originalText;
                button.disabled = false;
            }
        })
        .catch(error => {
            alert('Network error: ' + error);
            button.innerHTML = originalText;
            button.disabled = false;
        });
}

// JSON validation on form submit
document.getElementById('planForm')?.addEventListener('submit', function(e) {
    const fields = ['features', 'developer_features', 'technical_details'];
    let hasError = false;
    
    fields.forEach(field => {
        const textarea = document.getElementById(field);
        if (textarea && textarea.value.trim()) {
            try {
                JSON.parse(textarea.value);
            } catch (error) {
                alert('Invalid JSON in ' + field.replace('_', ' ') + ': ' + error.message);
                textarea.focus();
                hasError = true;
                e.preventDefault();
                return;
            }
        }
    });
});
</script>

<?php
require_once __DIR__ . '/includes/footer.php';
?>
