<?php
/**
 * Email Service Settings
 * Configure SMTP, SendGrid, MailChimp integration
 */

require_once __DIR__ . '/db.php';

$page_title = 'Email Service Settings';

// Create settings table
try {
    $db->exec("CREATE TABLE IF NOT EXISTS email_settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        service VARCHAR(50) NOT NULL,
        api_key TEXT,
        api_secret TEXT,
        sender_email VARCHAR(255),
        sender_name VARCHAR(255),
        smtp_host VARCHAR(255),
        smtp_port INT,
        smtp_username VARCHAR(255),
        smtp_password VARCHAR(255),
        smtp_encryption VARCHAR(10),
        is_active BOOLEAN DEFAULT 0,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
} catch (Exception $e) {
    error_log("Error creating email_settings table: " . $e->getMessage());
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $service = $_POST['service'] ?? 'smtp';
    
    try {
        // Deactivate all services first
        $db->exec("UPDATE email_settings SET is_active = 0");
        
        // Check if settings exist for this service
        $check = $db->prepare("SELECT id FROM email_settings WHERE service = :service");
        $check->execute(['service' => $service]);
        
        if ($check->fetch()) {
            // Update existing
            $stmt = $db->prepare("UPDATE email_settings SET 
                api_key = :api_key,
                api_secret = :api_secret,
                sender_email = :sender_email,
                sender_name = :sender_name,
                smtp_host = :smtp_host,
                smtp_port = :smtp_port,
                smtp_username = :smtp_username,
                smtp_password = :smtp_password,
                smtp_encryption = :smtp_encryption,
                is_active = 1
                WHERE service = :service");
        } else {
            // Insert new
            $stmt = $db->prepare("INSERT INTO email_settings 
                (service, api_key, api_secret, sender_email, sender_name, smtp_host, smtp_port, smtp_username, smtp_password, smtp_encryption, is_active)
                VALUES (:service, :api_key, :api_secret, :sender_email, :sender_name, :smtp_host, :smtp_port, :smtp_username, :smtp_password, :smtp_encryption, 1)");
        }
        
        $stmt->execute([
            'service' => $service,
            'api_key' => $_POST['api_key'] ?? null,
            'api_secret' => $_POST['api_secret'] ?? null,
            'sender_email' => $_POST['sender_email'] ?? null,
            'sender_name' => $_POST['sender_name'] ?? null,
            'smtp_host' => $_POST['smtp_host'] ?? null,
            'smtp_port' => $_POST['smtp_port'] ?? null,
            'smtp_username' => $_POST['smtp_username'] ?? null,
            'smtp_password' => $_POST['smtp_password'] ?? null,
            'smtp_encryption' => $_POST['smtp_encryption'] ?? null
        ]);
        
        $_SESSION['success_message'] = 'Email service settings saved successfully!';
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Error saving settings: ' . $e->getMessage();
    }
    
    header('Location: email_settings.php');
    exit;
}

// Get current settings
$current_settings = [];
$settings_result = $db->query("SELECT * FROM email_settings WHERE is_active = 1 LIMIT 1");
if ($settings_result) {
    $current_settings = $settings_result->fetch() ?: [];
}

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><i class="fas fa-cog"></i> Email Service Configuration</h4>
                </div>
                <div class="card-body">
                    <?php if (isset($_SESSION['success_message'])): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <?= htmlspecialchars($_SESSION['success_message']) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php unset($_SESSION['success_message']); ?>
                    <?php endif; ?>

                    <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <?= htmlspecialchars($_SESSION['error_message']) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php unset($_SESSION['error_message']); ?>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="mb-4">
                            <label class="form-label"><strong>Select Email Service</strong></label>
                            <select name="service" id="emailService" class="form-select" required>
                                <option value="smtp" <?= ($current_settings['service'] ?? '') === 'smtp' ? 'selected' : '' ?>>SMTP (Gmail, Outlook, etc.)</option>
                                <option value="sendgrid" <?= ($current_settings['service'] ?? '') === 'sendgrid' ? 'selected' : '' ?>>SendGrid</option>
                                <option value="mailchimp" <?= ($current_settings['service'] ?? '') === 'mailchimp' ? 'selected' : '' ?>>MailChimp</option>
                                <option value="aws_ses" <?= ($current_settings['service'] ?? '') === 'aws_ses' ? 'selected' : '' ?>>AWS SES</option>
                            </select>
                        </div>

                        <!-- SMTP Settings -->
                        <div id="smtpSettings" class="service-settings">
                            <h5 class="mb-3">SMTP Configuration</h5>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">SMTP Host</label>
                                    <input type="text" name="smtp_host" class="form-control" 
                                           value="<?= htmlspecialchars($current_settings['smtp_host'] ?? '') ?>"
                                           placeholder="smtp.gmail.com">
                                    <small class="text-muted">Gmail: smtp.gmail.com | Outlook: smtp-mail.outlook.com</small>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Port</label>
                                    <input type="number" name="smtp_port" class="form-control" 
                                           value="<?= htmlspecialchars($current_settings['smtp_port'] ?? '587') ?>"
                                           placeholder="587">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Encryption</label>
                                    <select name="smtp_encryption" class="form-select">
                                        <option value="tls" <?= ($current_settings['smtp_encryption'] ?? 'tls') === 'tls' ? 'selected' : '' ?>>TLS</option>
                                        <option value="ssl" <?= ($current_settings['smtp_encryption'] ?? '') === 'ssl' ? 'selected' : '' ?>>SSL</option>
                                    </select>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Username/Email</label>
                                    <input type="text" name="smtp_username" class="form-control" 
                                           value="<?= htmlspecialchars($current_settings['smtp_username'] ?? '') ?>"
                                           placeholder="your-email@gmail.com">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Password</label>
                                    <input type="password" name="smtp_password" class="form-control" 
                                           placeholder="••••••••">
                                    <small class="text-muted">For Gmail, use App Password (not your regular password)</small>
                                </div>
                            </div>
                        </div>

                        <!-- SendGrid Settings -->
                        <div id="sendgridSettings" class="service-settings" style="display: none;">
                            <h5 class="mb-3">SendGrid Configuration</h5>
                            <div class="mb-3">
                                <label class="form-label">API Key</label>
                                <input type="text" name="api_key" class="form-control" 
                                       value="<?= htmlspecialchars($current_settings['api_key'] ?? '') ?>"
                                       placeholder="SG.xxxxxxxxxxxxxxx">
                                <small class="text-muted">Get your API key from <a href="https://app.sendgrid.com/settings/api_keys" target="_blank">SendGrid Dashboard</a></small>
                            </div>
                        </div>

                        <!-- MailChimp Settings -->
                        <div id="mailchimpSettings" class="service-settings" style="display: none;">
                            <h5 class="mb-3">MailChimp Configuration</h5>
                            <div class="mb-3">
                                <label class="form-label">API Key</label>
                                <input type="text" name="api_key" class="form-control" 
                                       value="<?= htmlspecialchars($current_settings['api_key'] ?? '') ?>"
                                       placeholder="xxxxxxxxxxxxxxxxxxxxx-us1">
                                <small class="text-muted">Get your API key from <a href="https://admin.mailchimp.com/account/api/" target="_blank">MailChimp Account</a></small>
                            </div>
                        </div>

                        <!-- AWS SES Settings -->
                        <div id="awsSesSettings" class="service-settings" style="display: none;">
                            <h5 class="mb-3">AWS SES Configuration</h5>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Access Key ID</label>
                                    <input type="text" name="api_key" class="form-control" 
                                           value="<?= htmlspecialchars($current_settings['api_key'] ?? '') ?>"
                                           placeholder="AKIAIOSFODNN7EXAMPLE">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Secret Access Key</label>
                                    <input type="password" name="api_secret" class="form-control" 
                                           placeholder="wJalrXUtnFEMI/K7MDENG/bPxRfiCYEXAMPLEKEY">
                                </div>
                            </div>
                        </div>

                        <!-- Common Settings -->
                        <hr class="my-4">
                        <h5 class="mb-3">Sender Information</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Sender Name</label>
                                <input type="text" name="sender_name" class="form-control" 
                                       value="<?= htmlspecialchars($current_settings['sender_name'] ?? 'WsplCloud') ?>"
                                       placeholder="WsplCloud">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Sender Email</label>
                                <input type="email" name="sender_email" class="form-control" 
                                       value="<?= htmlspecialchars($current_settings['sender_email'] ?? 'noreply@wsplcloud.com') ?>"
                                       placeholder="noreply@wsplcloud.com">
                            </div>
                        </div>

                        <div class="d-flex justify-content-between">
                            <a href="send_campaign.php" class="btn btn-secondary">
                                <i class="fas fa-arrow-left"></i> Back to Campaigns
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> Save Configuration
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="card mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="fas fa-info-circle"></i> Setup Guides</h5>
                </div>
                <div class="card-body">
                    <h6>Gmail SMTP Setup:</h6>
                    <ol class="small">
                        <li>Enable 2-Step Verification in Google Account</li>
                        <li>Generate App Password</li>
                        <li>Use App Password (not regular password)</li>
                        <li>Host: smtp.gmail.com, Port: 587</li>
                    </ol>

                    <hr>

                    <h6>SendGrid Setup:</h6>
                    <ol class="small">
                        <li>Sign up at SendGrid.com (Free 100 emails/day)</li>
                        <li>Verify sender email</li>
                        <li>Create API Key with "Mail Send" permission</li>
                        <li>Copy API key and paste above</li>
                    </ol>

                    <hr>

                    <h6>Benefits by Service:</h6>
                    <ul class="small">
                        <li><strong>Gmail:</strong> Easy setup, 500 emails/day</li>
                        <li><strong>SendGrid:</strong> 100 free emails/day, better deliverability</li>
                        <li><strong>MailChimp:</strong> Marketing automation, analytics</li>
                        <li><strong>AWS SES:</strong> Cheapest for bulk, $0.10/1000 emails</li>
                    </ul>
                </div>
            </div>

            <div class="card">
                <div class="card-header bg-warning text-white">
                    <h5 class="mb-0"><i class="fas fa-exclamation-triangle"></i> Important Notes</h5>
                </div>
                <div class="card-body">
                    <ul class="small mb-0">
                        <li>Test your configuration with a small batch first</li>
                        <li>Keep API keys secure (never share)</li>
                        <li>Monitor bounce rates and spam reports</li>
                        <li>Include unsubscribe links (legal requirement)</li>
                        <li>Check your service's sending limits</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('emailService').addEventListener('change', function() {
    // Hide all settings
    document.querySelectorAll('.service-settings').forEach(el => el.style.display = 'none');
    
    // Show selected service settings
    const service = this.value;
    const settingsMap = {
        'smtp': 'smtpSettings',
        'sendgrid': 'sendgridSettings',
        'mailchimp': 'mailchimpSettings',
        'aws_ses': 'awsSesSettings'
    };
    
    if (settingsMap[service]) {
        document.getElementById(settingsMap[service]).style.display = 'block';
    }
});

// Trigger change on page load to show current service
document.getElementById('emailService').dispatchEvent(new Event('change'));
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
